<?php
// Define API access constant
define('API_ACCESS', true);

require_once '../config.php';

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Require authentication
require_auth();

$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            if (isset($_GET['id'])) {
                getDocument($_GET['id']);
            } elseif (isset($_GET['download']) && isset($_GET['id'])) {
                downloadDocument($_GET['id']);
            } else {
                getDocuments();
            }
            break;
        case 'POST':
            createDocument();
            break;
        case 'PUT':
            if (isset($_GET['id'])) {
                updateDocument($_GET['id']);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'Document ID required for update']);
            }
            break;
        case 'DELETE':
            if (isset($_GET['id'])) {
                deleteDocument($_GET['id']);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'Document ID required for delete']);
            }
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}

function getDocuments() {
    global $pdo;
    
    try {
        $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
        $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 20;
        $offset = ($page - 1) * $limit;
        
        $search = isset($_GET['search']) ? $_GET['search'] : '';
        $category = isset($_GET['category']) ? $_GET['category'] : '';
        $type = isset($_GET['type']) ? $_GET['type'] : '';
        
        $whereConditions = [];
        $params = [];
        
        if (!empty($search)) {
            $whereConditions[] = "(title LIKE ? OR description LIKE ?)";
            $params[] = "%$search%";
            $params[] = "%$search%";
        }
        
        if (!empty($category)) {
            $whereConditions[] = "category = ?";
            $params[] = $category;
        }
        
        if (!empty($type)) {
            $whereConditions[] = "document_type = ?";
            $params[] = $type;
        }
        
        $whereClause = !empty($whereConditions) ? "WHERE " . implode(" AND ", $whereConditions) : "";
        
        // Get total count
        $countSql = "SELECT COUNT(*) as total FROM documents $whereClause";
        $countStmt = $pdo->prepare($countSql);
        $countStmt->execute($params);
        $total = $countStmt->fetch(PDO::FETCH_ASSOC)['total'];
        
        // Get documents
        $sql = "SELECT * FROM documents $whereClause ORDER BY created_at DESC LIMIT ? OFFSET ?";
        $params[] = $limit;
        $params[] = $offset;
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $documents = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        echo json_encode([
            'success' => true,
            'data' => $documents,
            'pagination' => [
                'page' => $page,
                'limit' => $limit,
                'total' => $total,
                'pages' => ceil($total / $limit)
            ]
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}

function getDocument($id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("SELECT * FROM documents WHERE id = ?");
        $stmt->execute([$id]);
        $document = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$document) {
            http_response_code(404);
            echo json_encode(['error' => 'Document not found']);
            return;
        }
        
        echo json_encode([
            'success' => true,
            'data' => $document
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}

function createDocument() {
    global $pdo;
    
    try {
        // Handle file upload
        if (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) {
            http_response_code(400);
            echo json_encode(['error' => 'File upload required']);
            return;
        }
        
        $uploadDir = '../../uploads/documents/';
        if (!file_exists($uploadDir)) {
            mkdir($uploadDir, 0755, true);
        }
        
        $file = $_FILES['file'];
        $allowedTypes = ['pdf', 'doc', 'docx', 'txt', 'jpg', 'jpeg', 'png'];
        $fileExtension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        
        if (!in_array($fileExtension, $allowedTypes)) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid file type']);
            return;
        }
        
        $fileName = uniqid() . '.' . $fileExtension;
        $filePath = $uploadDir . $fileName;
        
        if (!move_uploaded_file($file['tmp_name'], $filePath)) {
            http_response_code(500);
            echo json_encode(['error' => 'Failed to upload file']);
            return;
        }
        
        // Get form data
        $title = $_POST['title'] ?? '';
        $description = $_POST['description'] ?? '';
        $category = $_POST['category'] ?? 'general';
        $documentType = $_POST['document_type'] ?? 'document';
        $isPublic = isset($_POST['is_public']) ? (bool)$_POST['is_public'] : false;
        
        if (empty($title)) {
            unlink($filePath); // Clean up uploaded file
            http_response_code(400);
            echo json_encode(['error' => 'Title is required']);
            return;
        }
        
        $sql = "INSERT INTO documents (title, description, category, document_type, file_path, file_name, file_size, is_public, uploaded_by, created_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            $title,
            $description,
            $category,
            $documentType,
            $fileName,
            $file['name'],
            $file['size'],
            $isPublic,
            $_SESSION['admin_id']
        ]);
        
        $documentId = $pdo->lastInsertId();
        
        // Get the created document
        $stmt = $pdo->prepare("SELECT * FROM documents WHERE id = ?");
        $stmt->execute([$documentId]);
        $document = $stmt->fetch(PDO::FETCH_ASSOC);
        
        echo json_encode([
            'success' => true,
            'message' => 'Document uploaded successfully',
            'data' => $document
        ]);
    } catch (Exception $e) {
        // Clean up uploaded file if exists
        if (isset($filePath) && file_exists($filePath)) {
            unlink($filePath);
        }
        throw $e;
    }
}

function updateDocument($id) {
    global $pdo;
    
    try {
        // Check if document exists
        $stmt = $pdo->prepare("SELECT * FROM documents WHERE id = ?");
        $stmt->execute([$id]);
        $document = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$document) {
            http_response_code(404);
            echo json_encode(['error' => 'Document not found']);
            return;
        }
        
        // Handle file replacement if new file uploaded
        $newFilePath = null;
        if (isset($_FILES['file']) && $_FILES['file']['error'] === UPLOAD_ERR_OK) {
            $uploadDir = '../../uploads/documents/';
            $file = $_FILES['file'];
            $allowedTypes = ['pdf', 'doc', 'docx', 'txt', 'jpg', 'jpeg', 'png'];
            $fileExtension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
            
            if (!in_array($fileExtension, $allowedTypes)) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid file type']);
                return;
            }
            
            $fileName = uniqid() . '.' . $fileExtension;
            $filePath = $uploadDir . $fileName;
            
            if (!move_uploaded_file($file['tmp_name'], $filePath)) {
                http_response_code(500);
                echo json_encode(['error' => 'Failed to upload file']);
                return;
            }
            
            $newFilePath = $fileName;
        }
        
        // Update document metadata
        $updateFields = [];
        $params = [];
        
        if (isset($_POST['title'])) {
            $updateFields[] = "title = ?";
            $params[] = $_POST['title'];
        }
        
        if (isset($_POST['description'])) {
            $updateFields[] = "description = ?";
            $params[] = $_POST['description'];
        }
        
        if (isset($_POST['category'])) {
            $updateFields[] = "category = ?";
            $params[] = $_POST['category'];
        }
        
        if (isset($_POST['document_type'])) {
            $updateFields[] = "document_type = ?";
            $params[] = $_POST['document_type'];
        }
        
        if (isset($_POST['is_public'])) {
            $updateFields[] = "is_public = ?";
            $params[] = (bool)$_POST['is_public'];
        }
        
        if ($newFilePath) {
            $updateFields[] = "file_path = ?, file_name = ?, file_size = ?";
            $params[] = $newFilePath;
            $params[] = $_FILES['file']['name'];
            $params[] = $_FILES['file']['size'];
        }
        
        if (empty($updateFields)) {
            http_response_code(400);
            echo json_encode(['error' => 'No valid fields to update']);
            return;
        }
        
        $updateFields[] = "updated_at = NOW()";
        $params[] = $id;
        
        $sql = "UPDATE documents SET " . implode(', ', $updateFields) . " WHERE id = ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        
        // Delete old file if new file was uploaded
        if ($newFilePath && $document['file_path']) {
            $oldFilePath = '../../uploads/documents/' . $document['file_path'];
            if (file_exists($oldFilePath)) {
                unlink($oldFilePath);
            }
        }
        
        // Get updated document
        $stmt = $pdo->prepare("SELECT * FROM documents WHERE id = ?");
        $stmt->execute([$id]);
        $updatedDocument = $stmt->fetch(PDO::FETCH_ASSOC);
        
        echo json_encode([
            'success' => true,
            'message' => 'Document updated successfully',
            'data' => $updatedDocument
        ]);
    } catch (Exception $e) {
        // Clean up uploaded file if exists
        if (isset($filePath) && file_exists($filePath)) {
            unlink($filePath);
        }
        throw $e;
    }
}

function deleteDocument($id) {
    global $pdo;
    
    try {
        // Get document info for file deletion
        $stmt = $pdo->prepare("SELECT file_path FROM documents WHERE id = ?");
        $stmt->execute([$id]);
        $document = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$document) {
            http_response_code(404);
            echo json_encode(['error' => 'Document not found']);
            return;
        }
        
        // Delete from database
        $stmt = $pdo->prepare("DELETE FROM documents WHERE id = ?");
        $stmt->execute([$id]);
        
        // Delete file
        if ($document['file_path']) {
            $filePath = '../../uploads/documents/' . $document['file_path'];
            if (file_exists($filePath)) {
                unlink($filePath);
            }
        }
        
        echo json_encode([
            'success' => true,
            'message' => 'Document deleted successfully'
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}

function downloadDocument($id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("SELECT * FROM documents WHERE id = ?");
        $stmt->execute([$id]);
        $document = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$document) {
            http_response_code(404);
            echo json_encode(['error' => 'Document not found']);
            return;
        }
        
        $filePath = '../../uploads/documents/' . $document['file_path'];
        
        if (!file_exists($filePath)) {
            http_response_code(404);
            echo json_encode(['error' => 'File not found']);
            return;
        }
        
        // Set headers for file download
        header('Content-Type: application/octet-stream');
        header('Content-Disposition: attachment; filename="' . $document['file_name'] . '"');
        header('Content-Length: ' . filesize($filePath));
        header('Cache-Control: must-revalidate');
        header('Pragma: public');
        
        readfile($filePath);
        exit;
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => $e->getMessage()]);
    }
}
?>
