<?php
/**
 * Email Management API for KSO Chandigarh Management System
 * Handles email configuration, sending, templates, and statistics
 */

require_once '../config.php';
require_once '../services/EmailService.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

// Initialize response
$response = ['success' => false, 'message' => '', 'data' => null];

try {
    $method = $_SERVER['REQUEST_METHOD'];
    $path = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
    $pathParts = explode('/', trim($path, '/'));
    $action = end($pathParts);
    
    $emailService = new EmailService();
    
    switch ($method) {
        case 'GET':
            switch ($action) {
                case 'settings':
                    // Get email configuration settings
                    $stmt = $pdo->prepare("
                        SELECT setting_key, setting_value, description 
                        FROM settings 
                        WHERE setting_key LIKE 'email_%' OR setting_key LIKE 'smtp_%'
                        ORDER BY setting_key
                    ");
                    $stmt->execute();
                    $settings = $stmt->fetchAll(PDO::FETCH_ASSOC);
                    
                    $response['success'] = true;
                    $response['data'] = $settings;
                    break;
                    
                case 'stats':
                    // Get email statistics
                    $stats = $emailService->getStats();
                    $response['success'] = true;
                    $response['data'] = $stats;
                    break;
                    
                case 'logs':
                    // Get recent email logs
                    $limit = isset($_GET['limit']) ? intval($_GET['limit']) : 50;
                    $logs = $emailService->getRecentLogs($limit);
                    $response['success'] = true;
                    $response['data'] = $logs;
                    break;
                    
                case 'templates':
                    // Get available email templates
                    $templates_dir = __DIR__ . '/../templates/email/';
                    $templates = [];
                    
                    if (is_dir($templates_dir)) {
                        $files = scandir($templates_dir);
                        foreach ($files as $file) {
                            if (pathinfo($file, PATHINFO_EXTENSION) === 'html') {
                                $name = pathinfo($file, PATHINFO_FILENAME);
                                $templates[] = [
                                    'name' => $name,
                                    'file' => $file,
                                    'path' => $templates_dir . $file,
                                    'exists' => true,
                                    'size' => filesize($templates_dir . $file)
                                ];
                            }
                        }
                    }
                    
                    $response['success'] = true;
                    $response['data'] = $templates;
                    break;
                    
                default:
                    $response['message'] = 'Invalid GET endpoint';
                    break;
            }
            break;
            
        case 'POST':
            $input = json_decode(file_get_contents('php://input'), true);
            
            switch ($action) {
                case 'save-settings':
                    // Save email configuration settings
                    if (!$input) {
                        $response['message'] = 'Invalid input data';
                        break;
                    }
                    
                    $pdo->beginTransaction();
                    
                    try {
                        foreach ($input as $key => $value) {
                            $stmt = $pdo->prepare("
                                UPDATE settings 
                                SET setting_value = ?, updated_at = NOW() 
                                WHERE setting_key = ?
                            ");
                            $stmt->execute([$value, $key]);
                        }
                        
                        $pdo->commit();
                        $response['success'] = true;
                        $response['message'] = 'Email settings saved successfully';
                    } catch (Exception $e) {
                        $pdo->rollBack();
                        $response['message'] = 'Failed to save settings: ' . $e->getMessage();
                    }
                    break;
                    
                case 'test-email':
                    // Send test email
                    if (!isset($input['email'])) {
                        $response['message'] = 'Email address is required';
                        break;
                    }
                    
                    $result = $emailService->testEmail($input['email']);
                    
                    if ($result) {
                        $response['success'] = true;
                        $response['message'] = 'Test email sent successfully';
                    } else {
                        $response['message'] = 'Failed to send test email';
                    }
                    break;
                    
                case 'send-email':
                    // Send custom email
                    if (!isset($input['to']) || !isset($input['subject']) || !isset($input['body'])) {
                        $response['message'] = 'To, subject, and body are required';
                        break;
                    }
                    
                    $result = $emailService->sendEmail(
                        $input['to'],
                        $input['subject'],
                        $input['body'],
                        $input['isHTML'] ?? true
                    );
                    
                    if ($result) {
                        $response['success'] = true;
                        $response['message'] = 'Email sent successfully';
                    } else {
                        $response['message'] = 'Failed to send email';
                    }
                    break;
                    
                case 'send-template':
                    // Send template-based email
                    if (!isset($input['template']) || !isset($input['to']) || !isset($input['subject'])) {
                        $response['message'] = 'Template, to, and subject are required';
                        break;
                    }
                    
                    $data = $input['data'] ?? [];
                    $result = $emailService->sendTemplate(
                        $input['template'],
                        $data,
                        $input['to'],
                        $input['subject']
                    );
                    
                    if ($result) {
                        $response['success'] = true;
                        $response['message'] = 'Template email sent successfully';
                    } else {
                        $response['message'] = 'Failed to send template email';
                    }
                    break;
                    
                case 'bulk-email':
                    // Send bulk email
                    if (!isset($input['recipients']) || !isset($input['subject']) || !isset($input['body'])) {
                        $response['message'] = 'Recipients, subject, and body are required';
                        break;
                    }
                    
                    $result = $emailService->sendBulkEmail(
                        $input['recipients'],
                        $input['subject'],
                        $input['body'],
                        $input['template'] ?? null
                    );
                    
                    $response['success'] = true;
                    $response['message'] = "Bulk email completed: {$result['success']} sent, {$result['failed']} failed";
                    $response['data'] = $result;
                    break;
                    
                default:
                    $response['message'] = 'Invalid POST endpoint';
                    break;
            }
            break;
            
        default:
            $response['message'] = 'Method not allowed';
            break;
    }
    
} catch (Exception $e) {
    $response['message'] = 'Server error: ' . $e->getMessage();
    error_log("Email API Error: " . $e->getMessage());
}

echo json_encode($response);
?>
