<?php
require_once '../config.php';

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Require authentication
require_auth();

$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            if (isset($_GET['id'])) {
                getExpense($_GET['id']);
            } else {
                getExpenses();
            }
            break;
        case 'POST':
            createExpense();
            break;
        case 'PUT':
            if (isset($_GET['id'])) {
                updateExpense($_GET['id']);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'Expense ID required for update']);
            }
            break;
        case 'DELETE':
            if (isset($_GET['id'])) {
                deleteExpense($_GET['id']);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'Expense ID required for delete']);
            }
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}

function getExpenses() {
    global $pdo;
    
    try {
        $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
        $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 20;
        $offset = ($page - 1) * $limit;
        
        $search = isset($_GET['search']) ? $_GET['search'] : '';
        $category = isset($_GET['category']) ? $_GET['category'] : '';
        $status = isset($_GET['status']) ? $_GET['status'] : '';
        $dateFrom = isset($_GET['date_from']) ? $_GET['date_from'] : '';
        $dateTo = isset($_GET['date_to']) ? $_GET['date_to'] : '';
        
        $whereConditions = [];
        $params = [];
        
        if (!empty($search)) {
            $whereConditions[] = "(description LIKE ? OR vendor LIKE ?)";
            $params[] = "%$search%";
            $params[] = "%$search%";
        }
        
        if (!empty($category)) {
            $whereConditions[] = "category = ?";
            $params[] = $category;
        }
        
        if (!empty($status)) {
            $whereConditions[] = "status = ?";
            $params[] = $status;
        }
        
        if (!empty($dateFrom)) {
            $whereConditions[] = "expense_date >= ?";
            $params[] = $dateFrom;
        }
        
        if (!empty($dateTo)) {
            $whereConditions[] = "expense_date <= ?";
            $params[] = $dateTo;
        }
        
        $whereClause = !empty($whereConditions) ? "WHERE " . implode(" AND ", $whereConditions) : "";
        
        // Get total count
        $countSql = "SELECT COUNT(*) as total FROM expenses $whereClause";
        $countStmt = $pdo->prepare($countSql);
        $countStmt->execute($params);
        $total = $countStmt->fetch(PDO::FETCH_ASSOC)['total'];
        
        // Get expenses
        $sql = "SELECT * FROM expenses $whereClause ORDER BY expense_date DESC, created_at DESC LIMIT ? OFFSET ?";
        $params[] = $limit;
        $params[] = $offset;
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $expenses = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        echo json_encode([
            'success' => true,
            'data' => $expenses,
            'pagination' => [
                'page' => $page,
                'limit' => $limit,
                'total' => $total,
                'pages' => ceil($total / $limit)
            ]
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}

function getExpense($id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("SELECT * FROM expenses WHERE id = ?");
        $stmt->execute([$id]);
        $expense = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$expense) {
            http_response_code(404);
            echo json_encode(['error' => 'Expense not found']);
            return;
        }
        
        echo json_encode([
            'success' => true,
            'data' => $expense
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}

function createExpense() {
    global $pdo;
    
    try {
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid JSON input']);
            return;
        }
        
        // Validate required fields
        $required = ['description', 'amount', 'category', 'expense_date'];
        foreach ($required as $field) {
            if (!isset($input[$field]) || empty($input[$field])) {
                http_response_code(400);
                echo json_encode(['error' => "Field '$field' is required"]);
                return;
            }
        }
        
        // Validate category
        $validCategories = ['office', 'travel', 'utilities', 'maintenance', 'events', 'supplies', 'other'];
        if (!in_array($input['category'], $validCategories)) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid category']);
            return;
        }
        
        $sql = "INSERT INTO expenses (description, amount, category, vendor, expense_date, receipt_number, notes, status, created_by, created_at) VALUES (?, ?, ?, ?, ?, ?, ?, 'pending', ?, NOW())";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            $input['description'],
            $input['amount'],
            $input['category'],
            $input['vendor'] ?? null,
            $input['expense_date'],
            $input['receipt_number'] ?? null,
            $input['notes'] ?? null,
            $_SESSION['admin_id']
        ]);
        
        $expenseId = $pdo->lastInsertId();
        
        // Get the created expense
        $stmt = $pdo->prepare("SELECT * FROM expenses WHERE id = ?");
        $stmt->execute([$expenseId]);
        $expense = $stmt->fetch(PDO::FETCH_ASSOC);
        
        echo json_encode([
            'success' => true,
            'message' => 'Expense created successfully',
            'data' => $expense
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}

function updateExpense($id) {
    global $pdo;
    
    try {
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid JSON input']);
            return;
        }
        
        // Check if expense exists
        $stmt = $pdo->prepare("SELECT id FROM expenses WHERE id = ?");
        $stmt->execute([$id]);
        if (!$stmt->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Expense not found']);
            return;
        }
        
        $updateFields = [];
        $params = [];
        
        $allowedFields = ['description', 'amount', 'category', 'vendor', 'expense_date', 'receipt_number', 'notes', 'status'];
        foreach ($allowedFields as $field) {
            if (isset($input[$field])) {
                $updateFields[] = "$field = ?";
                $params[] = $input[$field];
            }
        }
        
        if (empty($updateFields)) {
            http_response_code(400);
            echo json_encode(['error' => 'No valid fields to update']);
            return;
        }
        
        $updateFields[] = "updated_at = NOW()";
        $params[] = $id;
        
        $sql = "UPDATE expenses SET " . implode(', ', $updateFields) . " WHERE id = ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        
        // Get updated expense
        $stmt = $pdo->prepare("SELECT * FROM expenses WHERE id = ?");
        $stmt->execute([$id]);
        $expense = $stmt->fetch(PDO::FETCH_ASSOC);
        
        echo json_encode([
            'success' => true,
            'message' => 'Expense updated successfully',
            'data' => $expense
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}

function deleteExpense($id) {
    global $pdo;
    
    try {
        // Check if expense exists
        $stmt = $pdo->prepare("SELECT id FROM expenses WHERE id = ?");
        $stmt->execute([$id]);
        if (!$stmt->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Expense not found']);
            return;
        }
        
        $stmt = $pdo->prepare("DELETE FROM expenses WHERE id = ?");
        $stmt->execute([$id]);
        
        echo json_encode([
            'success' => true,
            'message' => 'Expense deleted successfully'
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}
?>
