<?php
// Define API access constant
define('API_ACCESS', true);

// Include configuration and authentication files
require_once '../config.php';
require_once '../auth/authenticate.php';

// Require admin authentication
requireAuth();

// Get the request method
$method = $_SERVER['REQUEST_METHOD'];

// Process the request based on the method
try {
    switch ($method) {
        case 'GET':
            getFinanceData();
            break;
        case 'POST':
            createFinanceRecord();
            break;
        case 'PUT':
            updateFinanceRecord();
            break;
        case 'DELETE':
            deleteFinanceRecord();
            break;
        default:
            sendResponse(405, false, 'Method not allowed');
    }
} catch (Exception $e) {
    // Log and send error response
    error_log("Error in finance API: " . $e->getMessage());
    sendResponse(500, false, 'Internal server error: ' . $e->getMessage());
}

/**
 * Get finance data
 */
function getFinanceData() {
    global $pdo;
    
    try {
        if (isset($_GET['id'])) {
            // Get a single finance record by ID
            $stmt = $pdo->prepare("SELECT * FROM finance WHERE id = ?");
            $stmt->execute([$_GET['id']]);
            $record = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($record) {
                sendResponse(200, true, 'Finance record retrieved successfully', $record);
            } else {
                sendResponse(404, false, 'Finance record not found');
            }
        } else {
            // Get all finance records
            $stmt = $pdo->query("SELECT * FROM finance ORDER BY created_at DESC");
            $records = $stmt->fetchAll(PDO::FETCH_ASSOC);
            sendResponse(200, true, 'Finance data retrieved successfully', $records);
        }
    } catch (Exception $e) {
        error_log("Error getting finance data: " . $e->getMessage());
        sendResponse(500, false, 'Failed to retrieve finance data');
    }
}

/**
 * Create a new finance record
 */
function createFinanceRecord() {
    global $pdo;
    
    // Get the input data
    $data = json_decode(file_get_contents('php://input'), true);
    
    // Validate input data
    if (!isset($data['transaction_type'], $data['amount'], $data['status'], $data['description'])) {
        sendResponse(400, false, 'Invalid input data');
        return;
    }
    
    try {
        // Insert the new record into the database
        $stmt = $pdo->prepare("
            INSERT INTO finance (transaction_type, amount, status, description, created_by)
            VALUES (?, ?, ?, ?, ?)
        ");
        
        // Assuming created_by is stored in the session
        $createdBy = $_SESSION['user_id'] ?? null;
        
        $stmt->execute([
            $data['transaction_type'],
            $data['amount'],
            $data['status'],
            $data['description'],
            $createdBy
        ]);
        
        $newRecordId = $pdo->lastInsertId();
        sendResponse(201, true, 'Finance record created successfully', ['id' => $newRecordId]);
        
    } catch (Exception $e) {
        error_log("Error creating finance record: " . $e->getMessage());
        sendResponse(500, false, 'Failed to create finance record');
    }
}

/**
 * Update an existing finance record
 */
function updateFinanceRecord() {
    global $pdo;
    
    // Get the input data
    $data = json_decode(file_get_contents('php://input'), true);
    
    // Validate input data
    if (!isset($data['id'], $data['transaction_type'], $data['amount'], $data['status'], $data['description'])) {
        sendResponse(400, false, 'Invalid input data');
        return;
    }
    
    try {
        // Update the record in the database
        $stmt = $pdo->prepare("
            UPDATE finance
            SET transaction_type = ?, amount = ?, status = ?, description = ?
            WHERE id = ?
        ");
        
        $stmt->execute([
            $data['transaction_type'],
            $data['amount'],
            $data['status'],
            $data['description'],
            $data['id']
        ]);
        
        if ($stmt->rowCount() > 0) {
            sendResponse(200, true, 'Finance record updated successfully');
        } else {
            sendResponse(404, false, 'Finance record not found or no changes made');
        }
        
    } catch (Exception $e) {
        error_log("Error updating finance record: " . $e->getMessage());
        sendResponse(500, false, 'Failed to update finance record');
    }
}

/**
 * Delete a finance record
 */
function deleteFinanceRecord() {
    global $pdo;
    
    // Get the ID from the query string
    if (!isset($_GET['id'])) {
        sendResponse(400, false, 'Finance record ID is required');
        return;
    }
    
    try {
        // Delete the record from the database
        $stmt = $pdo->prepare("DELETE FROM finance WHERE id = ?");
        $stmt->execute([$_GET['id']]);
        
        if ($stmt->rowCount() > 0) {
            sendResponse(200, true, 'Finance record deleted successfully');
        } else {
            sendResponse(404, false, 'Finance record not found');
        }
        
    } catch (Exception $e) {
        error_log("Error deleting finance record: " . $e->getMessage());
        sendResponse(500, false, 'Failed to delete finance record');
    }
}
?>
