<?php
require_once '../config.php';

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Require authentication
require_auth();

$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            if (isset($_GET['id'])) {
                getIdCard($_GET['id']);
            } elseif (isset($_GET['member_id'])) {
                getMemberIdCards($_GET['member_id']);
            } elseif (isset($_GET['generate']) && isset($_GET['member_id'])) {
                generateIdCard($_GET['member_id']);
            } else {
                getIdCards();
            }
            break;
        case 'POST':
            createIdCard();
            break;
        case 'PUT':
            if (isset($_GET['id'])) {
                updateIdCard($_GET['id']);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID card ID required for update']);
            }
            break;
        case 'DELETE':
            if (isset($_GET['id'])) {
                deleteIdCard($_GET['id']);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID card ID required for delete']);
            }
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}

function getIdCards() {
    global $pdo;
    
    try {
        $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
        $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 20;
        $offset = ($page - 1) * $limit;
        
        $search = isset($_GET['search']) ? $_GET['search'] : '';
        $status = isset($_GET['status']) ? $_GET['status'] : '';
        
        $whereConditions = [];
        $params = [];
        
        if (!empty($search)) {
            $whereConditions[] = "(m.first_name LIKE ? OR m.last_name LIKE ? OR m.kso_id LIKE ?)";
            $params[] = "%$search%";
            $params[] = "%$search%";
            $params[] = "%$search%";
        }
        
        if (!empty($status)) {
            $whereConditions[] = "ic.status = ?";
            $params[] = $status;
        }
        
        $whereClause = !empty($whereConditions) ? "WHERE " . implode(" AND ", $whereConditions) : "";
        
        // Get total count
        $countSql = "SELECT COUNT(*) as total FROM id_cards ic JOIN members m ON ic.member_id = m.id $whereClause";
        $countStmt = $pdo->prepare($countSql);
        $countStmt->execute($params);
        $total = $countStmt->fetch(PDO::FETCH_ASSOC)['total'];
        
        // Get ID cards with member details
        $sql = "SELECT 
                    ic.*, 
                    m.first_name, m.last_name, m.kso_id, m.email, m.phone, m.membership_type
                FROM id_cards ic 
                JOIN members m ON ic.member_id = m.id 
                $whereClause 
                ORDER BY ic.created_at DESC 
                LIMIT ? OFFSET ?";
        $params[] = $limit;
        $params[] = $offset;
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $idCards = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        echo json_encode([
            'success' => true,
            'data' => $idCards,
            'pagination' => [
                'page' => $page,
                'limit' => $limit,
                'total' => $total,
                'pages' => ceil($total / $limit)
            ]
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}

function getIdCard($id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT 
                ic.*, 
                m.first_name, m.last_name, m.kso_id, m.email, m.phone, 
                m.membership_type, m.date_of_birth, m.address
            FROM id_cards ic 
            JOIN members m ON ic.member_id = m.id 
            WHERE ic.id = ?
        ");
        $stmt->execute([$id]);
        $idCard = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$idCard) {
            http_response_code(404);
            echo json_encode(['error' => 'ID card not found']);
            return;
        }
        
        echo json_encode([
            'success' => true,
            'data' => $idCard
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}

function getMemberIdCards($memberId) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("SELECT * FROM id_cards WHERE member_id = ? ORDER BY created_at DESC");
        $stmt->execute([$memberId]);
        $idCards = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        echo json_encode([
            'success' => true,
            'data' => $idCards
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}

function createIdCard() {
    global $pdo;
    
    try {
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid JSON input']);
            return;
        }
        
        // Validate required fields
        $required = ['member_id'];
        foreach ($required as $field) {
            if (!isset($input[$field]) || empty($input[$field])) {
                http_response_code(400);
                echo json_encode(['error' => "Field '$field' is required"]);
                return;
            }
        }
        
        // Check if member exists
        $stmt = $pdo->prepare("SELECT * FROM members WHERE id = ?");
        $stmt->execute([$input['member_id']]);
        $member = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$member) {
            http_response_code(400);
            echo json_encode(['error' => 'Member not found']);
            return;
        }
        
        // Generate card number
        $cardNumber = generateCardNumber();
        
        $sql = "INSERT INTO id_cards (member_id, card_number, issue_date, expiry_date, status, created_by, created_at) VALUES (?, ?, ?, ?, 'active', ?, NOW())";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            $input['member_id'],
            $cardNumber,
            $input['issue_date'] ?? date('Y-m-d'),
            $input['expiry_date'] ?? date('Y-m-d', strtotime('+1 year')),
            $_SESSION['admin_id']
        ]);
        
        $cardId = $pdo->lastInsertId();
        
        // Get the created ID card with member details
        $stmt = $pdo->prepare("
            SELECT 
                ic.*, 
                m.first_name, m.last_name, m.kso_id, m.email, m.phone, m.membership_type
            FROM id_cards ic 
            JOIN members m ON ic.member_id = m.id 
            WHERE ic.id = ?
        ");
        $stmt->execute([$cardId]);
        $idCard = $stmt->fetch(PDO::FETCH_ASSOC);
        
        echo json_encode([
            'success' => true,
            'message' => 'ID card created successfully',
            'data' => $idCard
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}

function updateIdCard($id) {
    global $pdo;
    
    try {
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid JSON input']);
            return;
        }
        
        // Check if ID card exists
        $stmt = $pdo->prepare("SELECT id FROM id_cards WHERE id = ?");
        $stmt->execute([$id]);
        if (!$stmt->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'ID card not found']);
            return;
        }
        
        $updateFields = [];
        $params = [];
        
        $allowedFields = ['issue_date', 'expiry_date', 'status', 'notes'];
        foreach ($allowedFields as $field) {
            if (isset($input[$field])) {
                $updateFields[] = "$field = ?";
                $params[] = $input[$field];
            }
        }
        
        if (empty($updateFields)) {
            http_response_code(400);
            echo json_encode(['error' => 'No valid fields to update']);
            return;
        }
        
        $updateFields[] = "updated_at = NOW()";
        $params[] = $id;
        
        $sql = "UPDATE id_cards SET " . implode(', ', $updateFields) . " WHERE id = ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        
        // Get updated ID card with member details
        $stmt = $pdo->prepare("
            SELECT 
                ic.*, 
                m.first_name, m.last_name, m.kso_id, m.email, m.phone, m.membership_type
            FROM id_cards ic 
            JOIN members m ON ic.member_id = m.id 
            WHERE ic.id = ?
        ");
        $stmt->execute([$id]);
        $idCard = $stmt->fetch(PDO::FETCH_ASSOC);
        
        echo json_encode([
            'success' => true,
            'message' => 'ID card updated successfully',
            'data' => $idCard
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}

function deleteIdCard($id) {
    global $pdo;
    
    try {
        // Check if ID card exists
        $stmt = $pdo->prepare("SELECT id FROM id_cards WHERE id = ?");
        $stmt->execute([$id]);
        if (!$stmt->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'ID card not found']);
            return;
        }
        
        $stmt = $pdo->prepare("DELETE FROM id_cards WHERE id = ?");
        $stmt->execute([$id]);
        
        echo json_encode([
            'success' => true,
            'message' => 'ID card deleted successfully'
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}

function generateIdCard($memberId) {
    global $pdo;
    
    try {
        // Get member details
        $stmt = $pdo->prepare("SELECT * FROM members WHERE id = ?");
        $stmt->execute([$memberId]);
        $member = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$member) {
            http_response_code(404);
            echo json_encode(['error' => 'Member not found']);
            return;
        }
        
        // Get or create ID card
        $stmt = $pdo->prepare("SELECT * FROM id_cards WHERE member_id = ? AND status = 'active'");
        $stmt->execute([$memberId]);
        $idCard = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$idCard) {
            // Create new ID card
            $cardNumber = generateCardNumber();
            $stmt = $pdo->prepare("INSERT INTO id_cards (member_id, card_number, issue_date, expiry_date, status, created_by, created_at) VALUES (?, ?, ?, ?, 'active', ?, NOW())");
            $stmt->execute([
                $memberId,
                $cardNumber,
                date('Y-m-d'),
                date('Y-m-d', strtotime('+1 year')),
                $_SESSION['admin_id']
            ]);
            
            $cardId = $pdo->lastInsertId();
            $stmt = $pdo->prepare("SELECT * FROM id_cards WHERE id = ?");
            $stmt->execute([$cardId]);
            $idCard = $stmt->fetch(PDO::FETCH_ASSOC);
        }
        
        // Generate ID card data for frontend
        $cardData = [
            'member' => $member,
            'card' => $idCard,
            'qr_code_data' => json_encode([
                'kso_id' => $member['kso_id'],
                'name' => $member['first_name'] . ' ' . $member['last_name'],
                'card_number' => $idCard['card_number'],
                'valid_until' => $idCard['expiry_date']
            ])
        ];
        
        echo json_encode([
            'success' => true,
            'data' => $cardData
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}

function generateCardNumber() {
    global $pdo;
    
    do {
        $cardNumber = 'KSO' . date('Y') . str_pad(rand(1, 9999), 4, '0', STR_PAD_LEFT);
        
        // Check if card number already exists
        $stmt = $pdo->prepare("SELECT id FROM id_cards WHERE card_number = ?");
        $stmt->execute([$cardNumber]);
        $exists = $stmt->fetch();
    } while ($exists);
    
    return $cardNumber;
}
?>
