<?php
require_once '../config.php';

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Require authentication
require_auth();

$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            if (isset($_GET['id'])) {
                getMembershipFee($_GET['id']);
            } elseif (isset($_GET['member_id'])) {
                getMemberFees($_GET['member_id']);
            } else {
                getMembershipFees();
            }
            break;
        case 'POST':
            createMembershipFee();
            break;
        case 'PUT':
            if (isset($_GET['id'])) {
                updateMembershipFee($_GET['id']);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'Membership fee ID required for update']);
            }
            break;
        case 'DELETE':
            if (isset($_GET['id'])) {
                deleteMembershipFee($_GET['id']);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'Membership fee ID required for delete']);
            }
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}

function getMembershipFees() {
    global $pdo;
    
    try {
        $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
        $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 20;
        $offset = ($page - 1) * $limit;
        
        $search = isset($_GET['search']) ? $_GET['search'] : '';
        $status = isset($_GET['status']) ? $_GET['status'] : '';
        $year = isset($_GET['year']) ? $_GET['year'] : '';
        $membershipType = isset($_GET['membership_type']) ? $_GET['membership_type'] : '';
        
        $whereConditions = [];
        $params = [];
        
        if (!empty($search)) {
            $whereConditions[] = "(m.first_name LIKE ? OR m.last_name LIKE ? OR m.kso_id LIKE ?)";
            $params[] = "%$search%";
            $params[] = "%$search%";
            $params[] = "%$search%";
        }
        
        if (!empty($status)) {
            $whereConditions[] = "mf.status = ?";
            $params[] = $status;
        }
        
        if (!empty($year)) {
            $whereConditions[] = "mf.fee_year = ?";
            $params[] = $year;
        }
        
        if (!empty($membershipType)) {
            $whereConditions[] = "m.membership_type = ?";
            $params[] = $membershipType;
        }
        
        $whereClause = !empty($whereConditions) ? "WHERE " . implode(" AND ", $whereConditions) : "";
        
        // Get total count
        $countSql = "SELECT COUNT(*) as total FROM membership_fees mf JOIN members m ON mf.member_id = m.id $whereClause";
        $countStmt = $pdo->prepare($countSql);
        $countStmt->execute($params);
        $total = $countStmt->fetch(PDO::FETCH_ASSOC)['total'];
        
        // Get membership fees with member details
        $sql = "SELECT 
                    mf.*, 
                    m.first_name, m.last_name, m.kso_id, m.membership_type, m.email, m.phone
                FROM membership_fees mf 
                JOIN members m ON mf.member_id = m.id 
                $whereClause 
                ORDER BY mf.fee_year DESC, mf.created_at DESC 
                LIMIT ? OFFSET ?";
        $params[] = $limit;
        $params[] = $offset;
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $fees = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        echo json_encode([
            'success' => true,
            'data' => $fees,
            'pagination' => [
                'page' => $page,
                'limit' => $limit,
                'total' => $total,
                'pages' => ceil($total / $limit)
            ]
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}

function getMembershipFee($id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT 
                mf.*, 
                m.first_name, m.last_name, m.kso_id, m.membership_type, m.email, m.phone
            FROM membership_fees mf 
            JOIN members m ON mf.member_id = m.id 
            WHERE mf.id = ?
        ");
        $stmt->execute([$id]);
        $fee = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$fee) {
            http_response_code(404);
            echo json_encode(['error' => 'Membership fee not found']);
            return;
        }
        
        echo json_encode([
            'success' => true,
            'data' => $fee
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}

function getMemberFees($memberId) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("SELECT * FROM membership_fees WHERE member_id = ? ORDER BY fee_year DESC");
        $stmt->execute([$memberId]);
        $fees = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        echo json_encode([
            'success' => true,
            'data' => $fees
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}

function createMembershipFee() {
    global $pdo;
    
    try {
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid JSON input']);
            return;
        }
        
        // Validate required fields
        $required = ['member_id', 'fee_year', 'amount'];
        foreach ($required as $field) {
            if (!isset($input[$field]) || empty($input[$field])) {
                http_response_code(400);
                echo json_encode(['error' => "Field '$field' is required"]);
                return;
            }
        }
        
        // Check if member exists
        $stmt = $pdo->prepare("SELECT id FROM members WHERE id = ?");
        $stmt->execute([$input['member_id']]);
        if (!$stmt->fetch()) {
            http_response_code(400);
            echo json_encode(['error' => 'Member not found']);
            return;
        }
        
        // Check if fee already exists for this member and year
        $stmt = $pdo->prepare("SELECT id FROM membership_fees WHERE member_id = ? AND fee_year = ?");
        $stmt->execute([$input['member_id'], $input['fee_year']]);
        if ($stmt->fetch()) {
            http_response_code(400);
            echo json_encode(['error' => 'Membership fee already exists for this member and year']);
            return;
        }
        
        $sql = "INSERT INTO membership_fees (member_id, fee_year, amount, due_date, payment_date, payment_method, receipt_number, notes, status, created_by, created_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'pending', ?, NOW())";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            $input['member_id'],
            $input['fee_year'],
            $input['amount'],
            $input['due_date'] ?? null,
            $input['payment_date'] ?? null,
            $input['payment_method'] ?? null,
            $input['receipt_number'] ?? null,
            $input['notes'] ?? null,
            $_SESSION['admin_id']
        ]);
        
        $feeId = $pdo->lastInsertId();
        
        // Get the created fee with member details
        $stmt = $pdo->prepare("
            SELECT 
                mf.*, 
                m.first_name, m.last_name, m.kso_id, m.membership_type
            FROM membership_fees mf 
            JOIN members m ON mf.member_id = m.id 
            WHERE mf.id = ?
        ");
        $stmt->execute([$feeId]);
        $fee = $stmt->fetch(PDO::FETCH_ASSOC);
        
        echo json_encode([
            'success' => true,
            'message' => 'Membership fee created successfully',
            'data' => $fee
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}

function updateMembershipFee($id) {
    global $pdo;
    
    try {
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid JSON input']);
            return;
        }
        
        // Check if fee exists
        $stmt = $pdo->prepare("SELECT id FROM membership_fees WHERE id = ?");
        $stmt->execute([$id]);
        if (!$stmt->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Membership fee not found']);
            return;
        }
        
        $updateFields = [];
        $params = [];
        
        $allowedFields = ['amount', 'due_date', 'payment_date', 'payment_method', 'receipt_number', 'notes', 'status'];
        foreach ($allowedFields as $field) {
            if (isset($input[$field])) {
                $updateFields[] = "$field = ?";
                $params[] = $input[$field];
            }
        }
        
        if (empty($updateFields)) {
            http_response_code(400);
            echo json_encode(['error' => 'No valid fields to update']);
            return;
        }
        
        $updateFields[] = "updated_at = NOW()";
        $params[] = $id;
        
        $sql = "UPDATE membership_fees SET " . implode(', ', $updateFields) . " WHERE id = ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        
        // Get updated fee with member details
        $stmt = $pdo->prepare("
            SELECT 
                mf.*, 
                m.first_name, m.last_name, m.kso_id, m.membership_type
            FROM membership_fees mf 
            JOIN members m ON mf.member_id = m.id 
            WHERE mf.id = ?
        ");
        $stmt->execute([$id]);
        $fee = $stmt->fetch(PDO::FETCH_ASSOC);
        
        echo json_encode([
            'success' => true,
            'message' => 'Membership fee updated successfully',
            'data' => $fee
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}

function deleteMembershipFee($id) {
    global $pdo;
    
    try {
        // Check if fee exists
        $stmt = $pdo->prepare("SELECT id FROM membership_fees WHERE id = ?");
        $stmt->execute([$id]);
        if (!$stmt->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Membership fee not found']);
            return;
        }
        
        $stmt = $pdo->prepare("DELETE FROM membership_fees WHERE id = ?");
        $stmt->execute([$id]);
        
        echo json_encode([
            'success' => true,
            'message' => 'Membership fee deleted successfully'
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}
?>
