<?php
/**
 * Enhanced Photo Upload API
 * Handles chunked uploads, real-time progress, and optimization
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once '../config.php';
require_once '../security/security.php';
require_once '../services/EnhancedPhotoService.php';

// Initialize security and services
$security = new Security();
$photoService = new EnhancedPhotoService();

// Validate API access
if (!defined('API_ACCESS') || !API_ACCESS) {
    http_response_code(403);
    echo json_encode(['error' => 'API access denied']);
    exit();
}

$method = $_SERVER['REQUEST_METHOD'];
$path = $_GET['path'] ?? '';

try {
    switch ($method) {
        case 'POST':
            handlePostRequest($path, $photoService, $security);
            break;
        case 'GET':
            handleGetRequest($path, $photoService, $security);
            break;
        case 'PUT':
            handlePutRequest($path, $photoService, $security);
            break;
        case 'DELETE':
            handleDeleteRequest($path, $photoService, $security);
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    error_log("Enhanced Photo API Error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error']);
}

/**
 * Handle POST requests
 */
function handlePostRequest($path, $photoService, $security) {
    global $pdo;
    
    switch ($path) {
        case 'upload-chunk':
            handleChunkedUpload($photoService, $security);
            break;
            
        case 'upload-complete':
            handleUploadComplete($photoService, $security);
            break;
            
        case 'upload-traditional':
            handleTraditionalUpload($photoService, $security);
            break;
            
        case 'set-profile-photo':
            handleSetProfilePhoto($photoService, $security);
            break;
            
        default:
            http_response_code(404);
            echo json_encode(['error' => 'Endpoint not found']);
            break;
    }
}

/**
 * Handle GET requests
 */
function handleGetRequest($path, $photoService, $security) {
    global $pdo;
    
    switch ($path) {
        case 'member-photos':
            getMemberPhotos($photoService, $security);
            break;
            
        case 'photo-info':
            getPhotoInfo($photoService, $security);
            break;
            
        case 'upload-status':
            getUploadStatus($photoService, $security);
            break;
            
        case 'activity-status':
            getMemberActivityStatus($security);
            break;
            
        case 'online-members':
            getOnlineMembers($security);
            break;
            
        default:
            http_response_code(404);
            echo json_encode(['error' => 'Endpoint not found']);
            break;
    }
}

/**
 * Handle PUT requests
 */
function handlePutRequest($path, $photoService, $security) {
    switch ($path) {
        case 'update-activity':
            updateMemberActivity($security);
            break;
            
        default:
            http_response_code(404);
            echo json_encode(['error' => 'Endpoint not found']);
            break;
    }
}

/**
 * Handle DELETE requests
 */
function handleDeleteRequest($path, $photoService, $security) {
    switch ($path) {
        case 'photo':
            deletePhoto($photoService, $security);
            break;
            
        case 'cancel-upload':
            cancelUpload($photoService, $security);
            break;
            
        default:
            http_response_code(404);
            echo json_encode(['error' => 'Endpoint not found']);
            break;
    }
}

/**
 * Handle chunked file upload
 */
function handleChunkedUpload($photoService, $security) {
    // Validate session and CSRF
    if (!$security->validateSession() || !$security->validateCSRF()) {
        http_response_code(403);
        echo json_encode(['error' => 'Security validation failed']);
        return;
    }
    
    // Get chunk data
    $chunk_data = file_get_contents('php://input');
    $headers = getallheaders();
    
    $chunk_info = [
        'chunk_index' => intval($headers['X-Chunk-Index'] ?? 0),
        'total_chunks' => intval($headers['X-Total-Chunks'] ?? 1),
        'file_id' => $security->sanitizeInput($headers['X-File-Id'] ?? ''),
        'original_name' => $security->sanitizeInput($headers['X-Original-Name'] ?? ''),
        'total_size' => intval($headers['X-Total-Size'] ?? 0),
        'chunk_hash' => $headers['X-Chunk-Hash'] ?? '',
        'member_id' => intval($_GET['member_id'] ?? 0)
    ];
    
    // Validate chunk hash
    $calculated_hash = hash('sha256', $chunk_data);
    if ($chunk_info['chunk_hash'] && $calculated_hash !== $chunk_info['chunk_hash']) {
        http_response_code(400);
        echo json_encode(['error' => 'Chunk integrity check failed']);
        return;
    }
    
    // Rate limiting
    if (!$security->checkRateLimit('photo_upload', 10, 60)) {
        http_response_code(429);
        echo json_encode(['error' => 'Upload rate limit exceeded']);
        return;
    }
    
    // Process chunk
    $result = $photoService->handleChunkedUpload($chunk_data, $chunk_info);
    
    // Log upload activity
    logUploadActivity($chunk_info['file_id'], 'chunk_upload', [
        'chunk_index' => $chunk_info['chunk_index'],
        'chunk_size' => strlen($chunk_data),
        'total_chunks' => $chunk_info['total_chunks']
    ]);
    
    echo json_encode($result);
}

/**
 * Handle upload completion
 */
function handleUploadComplete($photoService, $security) {
    if (!$security->validateSession() || !$security->validateCSRF()) {
        http_response_code(403);
        echo json_encode(['error' => 'Security validation failed']);
        return;
    }
    
    $data = json_decode(file_get_contents('php://input'), true);
    $file_id = $security->sanitizeInput($data['file_id'] ?? '');
    $member_id = intval($data['member_id'] ?? 0);
    
    // Complete the upload process
    $result = $photoService->completeUpload($file_id, $member_id);
    
    if ($result['success']) {
        // Update member activity
        updateMemberActivityLog($member_id, 'photo_upload', [
            'photo_id' => $result['photo_id'],
            'filename' => $result['filename']
        ]);
    }
    
    echo json_encode($result);
}

/**
 * Handle traditional upload (non-chunked)
 */
function handleTraditionalUpload($photoService, $security) {
    if (!$security->validateSession() || !$security->validateCSRF()) {
        http_response_code(403);
        echo json_encode(['error' => 'Security validation failed']);
        return;
    }
    
    if (!isset($_FILES['photo'])) {
        http_response_code(400);
        echo json_encode(['error' => 'No file uploaded']);
        return;
    }
    
    $member_id = intval($_POST['member_id'] ?? 0);
    
    // Rate limiting
    if (!$security->checkRateLimit('photo_upload', 5, 300)) {
        http_response_code(429);
        echo json_encode(['error' => 'Upload rate limit exceeded']);
        return;
    }
    
    $result = $photoService->handleTraditionalUpload($_FILES['photo'], $member_id);
    
    if ($result['success']) {
        updateMemberActivityLog($member_id, 'photo_upload', [
            'photo_id' => $result['photo_id'],
            'filename' => $result['filename']
        ]);
    }
    
    echo json_encode($result);
}

/**
 * Set member profile photo
 */
function handleSetProfilePhoto($photoService, $security) {
    if (!$security->validateSession() || !$security->validateCSRF()) {
        http_response_code(403);
        echo json_encode(['error' => 'Security validation failed']);
        return;
    }
    
    $data = json_decode(file_get_contents('php://input'), true);
    $member_id = intval($data['member_id'] ?? 0);
    $photo_id = intval($data['photo_id'] ?? 0);
    
    $result = $photoService->updateMemberPhoto($member_id, $photo_id);
    echo json_encode($result);
}

/**
 * Get member photos
 */
function getMemberPhotos($photoService, $security) {
    global $pdo;
    
    $member_id = intval($_GET['member_id'] ?? 0);
    $limit = intval($_GET['limit'] ?? 20);
    $offset = intval($_GET['offset'] ?? 0);
    
    $stmt = $pdo->prepare("
        SELECT 
            mp.*,
            m.first_name,
            m.last_name,
            CONCAT('/uploads/member-photos/thumbnails/', mp.filename) as thumbnail_url,
            CONCAT('/uploads/member-photos/optimized/', mp.filename) as optimized_url,
            CONCAT('/uploads/member-photos/thumbnails/', 
                   SUBSTRING_INDEX(mp.filename, '.', 1), '.webp') as webp_thumbnail,
            CONCAT('/uploads/member-photos/optimized/', 
                   SUBSTRING_INDEX(mp.filename, '.', 1), '.webp') as webp_optimized
        FROM member_photos mp
        LEFT JOIN members m ON mp.member_id = m.id
        WHERE (? = 0 OR mp.member_id = ?)
        AND mp.processing_status = 'completed'
        ORDER BY mp.upload_date DESC
        LIMIT ? OFFSET ?
    ");
    
    $stmt->execute([$member_id, $member_id, $limit, $offset]);
    $photos = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo json_encode([
        'success' => true,
        'photos' => $photos,
        'total' => getTotalPhotoCount($member_id)
    ]);
}

/**
 * Get photo information
 */
function getPhotoInfo($photoService, $security) {
    $photo_id = intval($_GET['photo_id'] ?? 0);
    
    if (!$photo_id) {
        http_response_code(400);
        echo json_encode(['error' => 'Photo ID required']);
        return;
    }
    
    $photo = $photoService->getPhotoInfo($photo_id);
    
    if ($photo) {
        $photo['thumbnail_url'] = '/uploads/member-photos/thumbnails/' . $photo['filename'];
        $photo['optimized_url'] = '/uploads/member-photos/optimized/' . $photo['filename'];
        $photo['webp_thumbnail'] = '/uploads/member-photos/thumbnails/' . 
                                   pathinfo($photo['filename'], PATHINFO_FILENAME) . '.webp';
        $photo['webp_optimized'] = '/uploads/member-photos/optimized/' . 
                                   pathinfo($photo['filename'], PATHINFO_FILENAME) . '.webp';
    }
    
    echo json_encode([
        'success' => $photo !== false,
        'photo' => $photo
    ]);
}

/**
 * Get upload status
 */
function getUploadStatus($photoService, $security) {
    global $pdo;
    
    $session_id = $security->sanitizeInput($_GET['session_id'] ?? '');
    
    if (!$session_id) {
        http_response_code(400);
        echo json_encode(['error' => 'Session ID required']);
        return;
    }
    
    $stmt = $pdo->prepare("
        SELECT * FROM photo_upload_sessions 
        WHERE session_id = ? AND expires_at > NOW()
    ");
    $stmt->execute([$session_id]);
    $session = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$session) {
        echo json_encode([
            'success' => false,
            'error' => 'Upload session not found or expired'
        ]);
        return;
    }
    
    $progress = $session['chunks_total'] > 0 ? 
                ($session['chunks_uploaded'] / $session['chunks_total']) * 100 : 0;
    
    echo json_encode([
        'success' => true,
        'status' => $session['upload_status'],
        'progress' => round($progress, 2),
        'chunks_uploaded' => $session['chunks_uploaded'],
        'chunks_total' => $session['chunks_total']
    ]);
}

/**
 * Get member activity status
 */
function getMemberActivityStatus($security) {
    global $pdo;
    
    $member_id = intval($_GET['member_id'] ?? 0);
    
    if (!$member_id) {
        http_response_code(400);
        echo json_encode(['error' => 'Member ID required']);
        return;
    }
    
    $stmt = $pdo->prepare("
        SELECT 
            m.is_online,
            m.last_activity_at,
            mos.last_login,
            mos.last_logout,
            mos.activity_type,
            CASE 
                WHEN m.is_online = 1 THEN 'online'
                WHEN m.last_activity_at > DATE_SUB(NOW(), INTERVAL 5 MINUTE) THEN 'recently_active'
                WHEN m.last_activity_at > DATE_SUB(NOW(), INTERVAL 1 HOUR) THEN 'away'
                ELSE 'offline'
            END as status_label,
            TIMESTAMPDIFF(MINUTE, m.last_activity_at, NOW()) as minutes_since_activity
        FROM members m
        LEFT JOIN member_online_status mos ON m.id = mos.member_id
        WHERE m.id = ?
    ");
    
    $stmt->execute([$member_id]);
    $status = $stmt->fetch(PDO::FETCH_ASSOC);
    
    echo json_encode([
        'success' => true,
        'status' => $status
    ]);
}

/**
 * Get online members
 */
function getOnlineMembers($security) {
    global $pdo;
    
    $limit = intval($_GET['limit'] ?? 50);
    $status_filter = $_GET['status'] ?? 'all'; // all, online, recently_active
    
    $where_clause = "WHERE m.approval_status = 'approved' AND m.status = 'active'";
    
    if ($status_filter === 'online') {
        $where_clause .= " AND m.is_online = 1";
    } elseif ($status_filter === 'recently_active') {
        $where_clause .= " AND m.last_activity_at > DATE_SUB(NOW(), INTERVAL 1 HOUR)";
    }
    
    $stmt = $pdo->prepare("
        SELECT 
            m.id,
            m.kso_id,
            m.first_name,
            m.last_name,
            m.is_online,
            m.last_activity_at,
            mos.last_login,
            mos.activity_type,
            CASE 
                WHEN m.is_online = 1 THEN 'online'
                WHEN m.last_activity_at > DATE_SUB(NOW(), INTERVAL 5 MINUTE) THEN 'recently_active'
                WHEN m.last_activity_at > DATE_SUB(NOW(), INTERVAL 1 HOUR) THEN 'away'
                ELSE 'offline'
            END as status_label,
            TIMESTAMPDIFF(MINUTE, m.last_activity_at, NOW()) as minutes_since_activity,
            mp.filename as profile_photo,
            CONCAT('/uploads/member-photos/thumbnails/', mp.filename) as profile_thumbnail
        FROM members m
        LEFT JOIN member_online_status mos ON m.id = mos.member_id
        LEFT JOIN member_photos mp ON m.photo_id = mp.id
        {$where_clause}
        ORDER BY m.is_online DESC, m.last_activity_at DESC
        LIMIT ?
    ");
    
    $stmt->execute([$limit]);
    $members = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo json_encode([
        'success' => true,
        'members' => $members,
        'online_count' => getOnlineCount(),
        'total_active' => getTotalActiveMembers()
    ]);
}

/**
 * Update member activity
 */
function updateMemberActivity($security) {
    if (!$security->validateSession()) {
        http_response_code(403);
        echo json_encode(['error' => 'Session validation failed']);
        return;
    }
    
    $data = json_decode(file_get_contents('php://input'), true);
    $member_id = intval($data['member_id'] ?? 0);
    $activity_type = $security->sanitizeInput($data['activity_type'] ?? 'page_view');
    
    if (!$member_id) {
        http_response_code(400);
        echo json_encode(['error' => 'Member ID required']);
        return;
    }
    
    updateMemberActivityLog($member_id, $activity_type, $data['details'] ?? null);
    
    echo json_encode(['success' => true]);
}

/**
 * Delete photo
 */
function deletePhoto($photoService, $security) {
    if (!$security->validateSession() || !$security->validateCSRF()) {
        http_response_code(403);
        echo json_encode(['error' => 'Security validation failed']);
        return;
    }
    
    $photo_id = intval($_GET['photo_id'] ?? 0);
    
    if (!$photo_id) {
        http_response_code(400);
        echo json_encode(['error' => 'Photo ID required']);
        return;
    }
    
    $result = $photoService->deletePhoto($photo_id);
    echo json_encode($result);
}

/**
 * Cancel upload
 */
function cancelUpload($photoService, $security) {
    global $pdo;
    
    if (!$security->validateSession()) {
        http_response_code(403);
        echo json_encode(['error' => 'Security validation failed']);
        return;
    }
    
    $session_id = $security->sanitizeInput($_GET['session_id'] ?? '');
    
    if (!$session_id) {
        http_response_code(400);
        echo json_encode(['error' => 'Session ID required']);
        return;
    }
    
    // Update upload session status
    $stmt = $pdo->prepare("
        UPDATE photo_upload_sessions 
        SET upload_status = 'cancelled', updated_at = NOW()
        WHERE session_id = ?
    ");
    $stmt->execute([$session_id]);
    
    // Clean up temporary files
    $photoService->cleanupUploadSession($session_id);
    
    echo json_encode(['success' => true]);
}

/**
 * Helper functions
 */

function updateMemberActivityLog($member_id, $activity_type, $details = null) {
    global $pdo;
    
    $session_id = session_id();
    $ip_address = $_SERVER['REMOTE_ADDR'] ?? null;
    $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? null;
    
    $stmt = $pdo->prepare("
        INSERT INTO member_activity_log 
        (member_id, activity_type, activity_details, ip_address, user_agent, session_id)
        VALUES (?, ?, ?, ?, ?, ?)
    ");
    
    $stmt->execute([
        $member_id,
        $activity_type,
        $details ? json_encode($details) : null,
        $ip_address,
        $user_agent,
        $session_id
    ]);
}

function logUploadActivity($session_id, $action, $details = null, $error = null) {
    global $pdo;
    
    $stmt = $pdo->prepare("
        INSERT INTO photo_upload_log 
        (session_id, action, details, error_message)
        VALUES (?, ?, ?, ?)
    ");
    
    $stmt->execute([
        $session_id,
        $action,
        $details ? json_encode($details) : null,
        $error
    ]);
}

function getTotalPhotoCount($member_id = 0) {
    global $pdo;
    
    $stmt = $pdo->prepare("
        SELECT COUNT(*) FROM member_photos 
        WHERE (? = 0 OR member_id = ?) AND processing_status = 'completed'
    ");
    $stmt->execute([$member_id, $member_id]);
    return $stmt->fetchColumn();
}

function getOnlineCount() {
    global $pdo;
    
    $stmt = $pdo->prepare("
        SELECT COUNT(*) FROM members 
        WHERE is_online = 1 AND approval_status = 'approved' AND status = 'active'
    ");
    $stmt->execute();
    return $stmt->fetchColumn();
}

function getTotalActiveMembers() {
    global $pdo;
    
    $stmt = $pdo->prepare("
        SELECT COUNT(*) FROM members 
        WHERE approval_status = 'approved' AND status = 'active'
    ");
    $stmt->execute();
    return $stmt->fetchColumn();
}
?>
