<?php
/**
 * Real-time Notifications using Server-Sent Events (SSE)
 */

// Define API access before including config
define('API_ACCESS', true);
require_once '../config.php';
require_once '../services/NotificationService.php';

// Set SSE headers
header('Content-Type: text/event-stream');
header('Cache-Control: no-cache');
header('Connection: keep-alive');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Cache-Control');

// Disable output buffering
if (ob_get_level()) {
    ob_end_clean();
}

// Start session to get user info
session_start();

// Alternative: Get user info from query parameters if session not available
$userId = $_SESSION['user_id'] ?? $_GET['user_id'] ?? null;
$userType = $_SESSION['user_type'] ?? $_GET['user_type'] ?? 'member';

if (!$userId) {
    echo "event: error\n";
    echo "data: " . json_encode(['error' => 'User not authenticated']) . "\n\n";
    flush();
    exit();
}

$notificationService = new NotificationService();
$lastEventId = $_SERVER['HTTP_LAST_EVENT_ID'] ?? 0;

// Send initial connection message
echo "event: connected\n";
echo "data: Connected to notification stream\n\n";
flush();

// Main SSE loop
$startTime = time();
$maxExecutionTime = 300; // 5 minutes

while (time() - $startTime < $maxExecutionTime) {
    try {
        // Check for new notifications
        $unreadNotifications = $notificationService->getUnreadNotifications($userId, $userType);
        
        foreach ($unreadNotifications as $notification) {
            if ($notification['id'] > $lastEventId) {
                // Send notification event
                echo "event: notification\n";
                echo "id: {$notification['id']}\n";
                echo "data: " . json_encode([
                    'id' => $notification['id'],
                    'type' => $notification['type'],
                    'title' => $notification['title'],
                    'message' => $notification['message'],
                    'priority' => $notification['priority'],
                    'created_at' => $notification['created_at'],
                    'delivered_at' => $notification['delivered_at']
                ]) . "\n\n";
                
                $lastEventId = $notification['id'];
                flush();
            }
        }
        
        // Send heartbeat every 30 seconds
        if (time() % 30 == 0) {
            echo "event: heartbeat\n";
            echo "data: " . json_encode(['timestamp' => time(), 'unread_count' => count($unreadNotifications)]) . "\n\n";
            flush();
        }
        
        // Check if client is still connected
        if (connection_aborted()) {
            break;
        }
        
        sleep(2); // Check every 2 seconds
        
    } catch (Exception $e) {
        echo "event: error\n";
        echo "data: " . json_encode(['error' => $e->getMessage()]) . "\n\n";
        flush();
        break;
    }
}

echo "event: close\n";
echo "data: Connection closed\n\n";
flush();
?>
