<?php
// Define API access constant
define('API_ACCESS', true);

// Include configuration and authentication files
require_once '../config.php';
require_once '../auth/authenticate.php';

// Require admin authentication
requireAuth();

// Get the request method
$method = $_SERVER['REQUEST_METHOD'];

// Process the request based on the method
try {
    switch ($method) {
        case 'GET':
            handleGetRequest();
            break;
        case 'POST':
            handlePostRequest();
            break;
        case 'DELETE':
            deleteReceipt();
            break;
        default:
            sendResponse(405, false, 'Method not allowed');
    }
} catch (Exception $e) {
    // Log and send error response
    error_log("Error in receipts API: " . $e->getMessage());
    sendResponse(500, false, 'Internal server error: ' . $e->getMessage());
}

/**
 * Handle GET requests
 */
function handleGetRequest() {
    if (isset($_GET['id'])) {
        getReceiptDetails($_GET['id']);
    } elseif (isset($_GET['action']) && $_GET['action'] === 'list_transactions') {
        getTransactionsForReceipts();
    } else {
        getReceiptsList();
    }
}

/**
 * Handle POST requests
 */
function handlePostRequest() {
    $data = json_decode(file_get_contents('php://input'), true);
    if (isset($data['action']) && $data['action'] === 'bulk_generate') {
        bulkGenerateReceipts($data);
    } else {
        generateReceipt($data);
    }
}

/**
 * Get a list of all receipts with filtering and pagination
 */
function getReceiptsList() {
    global $pdo;
    
    // Pagination
    $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
    $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 20;
    $offset = ($page - 1) * $limit;

    // Base query
    $sql = "
        SELECT 
            r.id,
            r.receipt_number,
            r.amount,
            r.receipt_date,
            r.status,
            f.transaction_type,
            CONCAT(m.first_name, ' ', m.last_name) as member_name,
            m.kso_id
        FROM receipts r
        JOIN finance f ON r.transaction_id = f.id
        JOIN members m ON r.member_id = m.id
    ";
    
    $where = [];
    $params = [];

    // Filtering
    if (!empty($_GET['search'])) {
        $where[] = "(r.receipt_number LIKE ? OR CONCAT(m.first_name, ' ', m.last_name) LIKE ? OR m.kso_id LIKE ?)";
        $searchTerm = '%' . $_GET['search'] . '%';
        array_push($params, $searchTerm, $searchTerm, $searchTerm);
    }
    if (!empty($_GET['transaction_type'])) {
        $where[] = "f.transaction_type = ?";
        $params[] = $_GET['transaction_type'];
    }
    if (!empty($_GET['year'])) {
        $where[] = "YEAR(r.receipt_date) = ?";
        $params[] = $_GET['year'];
    }

    if (count($where) > 0) {
        $sql .= " WHERE " . implode(' AND ', $where);
    }

    // Get total count for pagination
    $countSql = "SELECT COUNT(*) FROM (" . $sql . ") as count_table";
    $countStmt = $pdo->prepare($countSql);
    $countStmt->execute($params);
    $totalReceipts = $countStmt->fetchColumn();
    $totalPages = ceil($totalReceipts / $limit);

    // Add ordering and pagination to main query
    $sql .= " ORDER BY r.receipt_date DESC LIMIT ? OFFSET ?";
    $params[] = $limit;
    $params[] = $offset;
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $receipts = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    sendResponse(200, true, 'Receipts retrieved successfully', [
        'receipts' => $receipts,
        'pagination' => [
            'page' => $page,
            'limit' => $limit,
            'total' => (int)$totalReceipts,
            'pages' => $totalPages
        ]
    ]);
}

/**
 * Get details for a single receipt
 */
function getReceiptDetails($id) {
    global $pdo;
    
    $stmt = $pdo->prepare("
        SELECT 
            r.*, 
            f.transaction_type, 
            f.description,
            CONCAT(m.first_name, ' ', m.last_name) as member_name,
            m.kso_id,
            m.address
        FROM receipts r
        JOIN finance f ON r.transaction_id = f.id
        JOIN members m ON r.member_id = m.id
        WHERE r.id = ?
    ");
    $stmt->execute([$id]);
    $receipt = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($receipt) {
        sendResponse(200, true, 'Receipt details retrieved', $receipt);
    } else {
        sendResponse(404, false, 'Receipt not found');
    }
}

/**
 * Get a list of completed transactions that do not have a receipt yet
 */
function getTransactionsForReceipts() {
    global $pdo;
    
    $sql = "
        SELECT f.id, f.description, f.amount, f.transaction_type, CONCAT(m.first_name, ' ', m.last_name) as member_name
        FROM finance f
        JOIN members m ON f.member_id = m.id
        LEFT JOIN receipts r ON f.id = r.transaction_id
        WHERE f.status = 'completed' AND r.id IS NULL
        ORDER BY f.created_at DESC
    ";
    
    $stmt = $pdo->query($sql);
    $transactions = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    sendResponse(200, true, 'Transactions for receipt generation retrieved', $transactions);
}

/**
 * Generate a new receipt for a single transaction
 */
function generateReceipt($data) {
    global $pdo;
    
    if (!isset($data['transaction_id'])) {
        sendResponse(400, false, 'Transaction ID is required');
        return;
    }
    
    $transactionId = $data['transaction_id'];
    
    // Check if receipt already exists
    $stmt = $pdo->prepare("SELECT id FROM receipts WHERE transaction_id = ?");
    $stmt->execute([$transactionId]);
    if ($stmt->fetch()) {
        sendResponse(409, false, 'A receipt for this transaction already exists.');
        return;
    }

    // Get transaction details
    $stmt = $pdo->prepare("SELECT * FROM finance WHERE id = ? AND status = 'completed'");
    $stmt->execute([$transactionId]);
    $transaction = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$transaction) {
        sendResponse(404, false, 'Completed transaction not found.');
        return;
    }

    // Generate receipt number
    $receiptNumber = 'KSO-RCPT-' . date('Ymd') . '-' . strtoupper(uniqid());

    $pdo->beginTransaction();
    try {
        $stmt = $pdo->prepare("
            INSERT INTO receipts (receipt_number, transaction_id, member_id, amount, receipt_date, status, created_by)
            VALUES (?, ?, ?, ?, NOW(), 'issued', ?)
        ");
        $stmt->execute([
            $receiptNumber,
            $transaction['id'],
            $transaction['member_id'],
            $transaction['amount'],
            $_SESSION['user_id'] ?? null
        ]);
        $receiptId = $pdo->lastInsertId();
        $pdo->commit();
        sendResponse(201, true, 'Receipt generated successfully', ['receipt_id' => $receiptId, 'receipt_number' => $receiptNumber]);
    } catch (Exception $e) {
        $pdo->rollBack();
        error_log("Error generating receipt: " . $e->getMessage());
        sendResponse(500, false, 'Failed to generate receipt.');
    }
}

/**
 * Bulk generate receipts for all eligible transactions
 */
function bulkGenerateReceipts($data) {
    global $pdo;
    
    $stmt = $pdo->query("
        SELECT f.id, f.member_id, f.amount
        FROM finance f
        LEFT JOIN receipts r ON f.id = r.transaction_id
        WHERE f.status = 'completed' AND r.id IS NULL
    ");
    $transactions = $stmt->fetchAll(PDO::FETCH_ASSOC);

    if (empty($transactions)) {
        sendResponse(200, true, 'No new receipts to generate.', ['generated_count' => 0]);
        return;
    }

    $pdo->beginTransaction();
    $generatedCount = 0;
    try {
        $stmt = $pdo->prepare("
            INSERT INTO receipts (receipt_number, transaction_id, member_id, amount, receipt_date, status, created_by)
            VALUES (?, ?, ?, ?, NOW(), 'issued', ?)
        ");
        
        foreach ($transactions as $transaction) {
            $receiptNumber = 'KSO-RCPT-' . date('Ymd') . '-' . strtoupper(uniqid());
            $stmt->execute([
                $receiptNumber,
                $transaction['id'],
                $transaction['member_id'],
                $transaction['amount'],
                $_SESSION['user_id'] ?? null
            ]);
            $generatedCount++;
        }
        
        $pdo->commit();
        sendResponse(200, true, 'Bulk receipt generation completed.', ['generated_count' => $generatedCount]);
    } catch (Exception $e) {
        $pdo->rollBack();
        error_log("Error in bulk receipt generation: " . $e->getMessage());
        sendResponse(500, false, 'Failed during bulk generation.');
    }
}

/**
 * Delete a receipt
 */
function deleteReceipt() {
    global $pdo;
    
    if (!isset($_GET['id'])) {
        sendResponse(400, false, 'Receipt ID is required');
        return;
    }
    
    $id = $_GET['id'];
    
    $stmt = $pdo->prepare("DELETE FROM receipts WHERE id = ?");
    $stmt->execute([$id]);
    
    if ($stmt->rowCount() > 0) {
        sendResponse(200, true, 'Receipt deleted successfully');
    } else {
        sendResponse(404, false, 'Receipt not found or already deleted.');
    }
}
?>
