<?php
/**
 * Security Configuration for KSO Chandigarh Management System
 * Enhanced security measures and protection settings
 */

// Prevent direct access to this file
if (!defined('SECURITY_INIT')) {
    die('Access denied');
}

class SecurityConfig {
    
    // Security constants
    const SESSION_LIFETIME = 3600; // 1 hour
    const MAX_LOGIN_ATTEMPTS = 5;
    const LOCKOUT_DURATION = 1800; // 30 minutes
    const PASSWORD_MIN_LENGTH = 8;
    const CSRF_TOKEN_LIFETIME = 3600;
    
    /**
     * Initialize security measures
     */
    public static function init() {
        // Set secure session parameters
        self::configureSession();
        
        // Set security headers
        self::setSecurityHeaders();
        
        // Input sanitization
        self::sanitizeInputs();
        
        // Rate limiting
        self::checkRateLimit();
    }
    
    /**
     * Configure secure session settings
     */
    private static function configureSession() {
        // Prevent session fixation
        if (session_status() === PHP_SESSION_NONE) {
            ini_set('session.cookie_httponly', 1);
            ini_set('session.cookie_secure', isset($_SERVER['HTTPS']));
            ini_set('session.use_strict_mode', 1);
            ini_set('session.cookie_samesite', 'Strict');
            ini_set('session.gc_maxlifetime', self::SESSION_LIFETIME);
            
            session_start();
            
            // Regenerate session ID periodically
            if (!isset($_SESSION['last_regeneration'])) {
                $_SESSION['last_regeneration'] = time();
            } elseif (time() - $_SESSION['last_regeneration'] > 300) {
                session_regenerate_id(true);
                $_SESSION['last_regeneration'] = time();
            }
        }
    }
    
    /**
     * Set comprehensive security headers
     */
    private static function setSecurityHeaders() {
        // Prevent MIME type sniffing
        header('X-Content-Type-Options: nosniff');
        
        // Prevent clickjacking
        header('X-Frame-Options: DENY');
        
        // XSS Protection
        header('X-XSS-Protection: 1; mode=block');
        
        // Referrer Policy
        header('Referrer-Policy: strict-origin-when-cross-origin');
        
        // Content Security Policy
        header("Content-Security-Policy: default-src 'self'; script-src 'self' 'unsafe-inline'; style-src 'self' 'unsafe-inline'; img-src 'self' data:; font-src 'self'");
        
        // HSTS (if HTTPS is enabled)
        if (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') {
            header('Strict-Transport-Security: max-age=31536000; includeSubDomains');
        }
        
        // Remove server information
        header_remove('X-Powered-By');
        header_remove('Server');
    }
    
    /**
     * Sanitize all inputs
     */
    private static function sanitizeInputs() {
        // Sanitize GET parameters
        if (!empty($_GET)) {
            $_GET = self::sanitizeArray($_GET);
        }
        
        // Sanitize POST parameters
        if (!empty($_POST)) {
            $_POST = self::sanitizeArray($_POST);
        }
        
        // Sanitize COOKIE parameters
        if (!empty($_COOKIE)) {
            $_COOKIE = self::sanitizeArray($_COOKIE);
        }
    }
    
    /**
     * Sanitize array recursively
     */
    private static function sanitizeArray($array) {
        foreach ($array as $key => $value) {
            if (is_array($value)) {
                $array[$key] = self::sanitizeArray($value);
            } else {
                // Basic sanitization
                $array[$key] = trim($value);
                $array[$key] = stripslashes($array[$key]);
                $array[$key] = htmlspecialchars($array[$key], ENT_QUOTES, 'UTF-8');
            }
        }
        return $array;
    }
    
    /**
     * Check for suspicious patterns and block them
     */
    public static function validateInput($input) {
        // SQL injection patterns
        $sql_patterns = [
            '/(\bunion\b.*\bselect\b)/i',
            '/(\bdrop\b.*\btable\b)/i',
            '/(\binsert\b.*\binto\b)/i',
            '/(\bupdate\b.*\bset\b)/i',
            '/(\bdelete\b.*\bfrom\b)/i',
            '/(\bselect\b.*\bfrom\b)/i',
            '/(\bor\b.*1.*=.*1)/i',
            '/(\band\b.*1.*=.*1)/i'
        ];
        
        // XSS patterns
        $xss_patterns = [
            '/<script\b[^<]*(?:(?!<\/script>)<[^<]*)*<\/script>/i',
            '/javascript:/i',
            '/vbscript:/i',
            '/onload\s*=/i',
            '/onerror\s*=/i',
            '/onclick\s*=/i',
            '/onmouseover\s*=/i'
        ];
        
        // File inclusion patterns
        $inclusion_patterns = [
            '/\.\.\//i',
            '/\.\.\\\/i',
            '/etc\/passwd/i',
            '/proc\/self\/environ/i',
            '/localhost/i',
            '/127\.0\.0\.1/i'
        ];
        
        $all_patterns = array_merge($sql_patterns, $xss_patterns, $inclusion_patterns);
        
        foreach ($all_patterns as $pattern) {
            if (preg_match($pattern, $input)) {
                self::logSecurityEvent('Suspicious input detected', [
                    'input' => $input,
                    'pattern' => $pattern,
                    'ip' => $_SERVER['REMOTE_ADDR'] ?? 'unknown'
                ]);
                return false;
            }
        }
        
        return true;
    }
    
    /**
     * Rate limiting check
     */
    private static function checkRateLimit() {
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        $key = 'rate_limit_' . md5($ip);
        
        if (!isset($_SESSION[$key])) {
            $_SESSION[$key] = ['count' => 1, 'time' => time()];
        } else {
            $data = $_SESSION[$key];
            
            // Reset counter if more than 1 minute has passed
            if (time() - $data['time'] > 60) {
                $_SESSION[$key] = ['count' => 1, 'time' => time()];
            } else {
                $_SESSION[$key]['count']++;
                
                // Block if more than 100 requests per minute
                if ($_SESSION[$key]['count'] > 100) {
                    self::logSecurityEvent('Rate limit exceeded', ['ip' => $ip]);
                    http_response_code(429);
                    die('Rate limit exceeded. Please try again later.');
                }
            }
        }
    }
    
    /**
     * Generate CSRF token
     */
    public static function generateCSRFToken() {
        if (!isset($_SESSION['csrf_token']) || !isset($_SESSION['csrf_time']) || 
            time() - $_SESSION['csrf_time'] > self::CSRF_TOKEN_LIFETIME) {
            $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
            $_SESSION['csrf_time'] = time();
        }
        return $_SESSION['csrf_token'];
    }
    
    /**
     * Validate CSRF token
     */
    public static function validateCSRFToken($token) {
        if (!isset($_SESSION['csrf_token']) || !isset($_SESSION['csrf_time'])) {
            return false;
        }
        
        if (time() - $_SESSION['csrf_time'] > self::CSRF_TOKEN_LIFETIME) {
            return false;
        }
        
        return hash_equals($_SESSION['csrf_token'], $token);
    }
    
    /**
     * Validate password strength
     */
    public static function validatePassword($password) {
        if (strlen($password) < self::PASSWORD_MIN_LENGTH) {
            return false;
        }
        
        // Require at least one lowercase letter
        if (!preg_match('/[a-z]/', $password)) {
            return false;
        }
        
        // Require at least one uppercase letter
        if (!preg_match('/[A-Z]/', $password)) {
            return false;
        }
        
        // Require at least one digit
        if (!preg_match('/\d/', $password)) {
            return false;
        }
        
        // Require at least one special character
        if (!preg_match('/[^a-zA-Z\d]/', $password)) {
            return false;
        }
        
        return true;
    }
    
    /**
     * Secure password hashing
     */
    public static function hashPassword($password) {
        return password_hash($password, PASSWORD_ARGON2ID, [
            'memory_cost' => 65536,
            'time_cost' => 4,
            'threads' => 3
        ]);
    }
    
    /**
     * Verify password
     */
    public static function verifyPassword($password, $hash) {
        return password_verify($password, $hash);
    }
    
    /**
     * Log security events
     */
    private static function logSecurityEvent($event, $data = []) {
        $log_entry = [
            'timestamp' => date('Y-m-d H:i:s'),
            'event' => $event,
            'ip' => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'unknown',
            'request_uri' => $_SERVER['REQUEST_URI'] ?? 'unknown',
            'data' => $data
        ];
        
        // Log to file (create secure log directory)
        $log_dir = __DIR__ . '/../../logs/security/';
        if (!is_dir($log_dir)) {
            mkdir($log_dir, 0750, true);
        }
        
        $log_file = $log_dir . 'security_' . date('Y-m-d') . '.log';
        error_log(json_encode($log_entry) . PHP_EOL, 3, $log_file);
    }
    
    /**
     * Check for common attack patterns in request
     */
    public static function checkRequest() {
        $request_uri = $_SERVER['REQUEST_URI'] ?? '';
        $query_string = $_SERVER['QUERY_STRING'] ?? '';
        $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        
        // Check for suspicious patterns in URI
        $suspicious_patterns = [
            '/\.\.\//',
            '/etc\/passwd/',
            '/proc\/self\/environ/',
            '/\<script/',
            '/javascript:/',
            '/vbscript:/',
            '/union.*select/i',
            '/drop.*table/i',
            '/insert.*into/i'
        ];
        
        foreach ($suspicious_patterns as $pattern) {
            if (preg_match($pattern, $request_uri . $query_string)) {
                self::logSecurityEvent('Suspicious request pattern', [
                    'pattern' => $pattern,
                    'uri' => $request_uri,
                    'query' => $query_string
                ]);
                http_response_code(403);
                die('Access denied');
            }
        }
        
        // Check for suspicious user agents
        $blocked_user_agents = [
            '/sqlmap/',
            '/nikto/',
            '/dirbuster/',
            '/nmap/',
            '/masscan/',
            '/grabber/',
            '/w3af/'
        ];
        
        foreach ($blocked_user_agents as $pattern) {
            if (preg_match($pattern, strtolower($user_agent))) {
                self::logSecurityEvent('Blocked user agent', ['user_agent' => $user_agent]);
                http_response_code(403);
                die('Access denied');
            }
        }
    }
    
    /**
     * Secure file upload validation
     */
    public static function validateFileUpload($file) {
        // Check file size (max 10MB)
        if ($file['size'] > 10 * 1024 * 1024) {
            return false;
        }
        
        // Check MIME type
        $allowed_types = [
            'image/jpeg',
            'image/png',
            'image/gif',
            'application/pdf',
            'application/msword',
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document'
        ];
        
        if (!in_array($file['type'], $allowed_types)) {
            return false;
        }
        
        // Check file extension
        $allowed_extensions = ['jpg', 'jpeg', 'png', 'gif', 'pdf', 'doc', 'docx'];
        $file_extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        
        if (!in_array($file_extension, $allowed_extensions)) {
            return false;
        }
        
        // Additional security checks
        if (strpos($file['name'], '..') !== false) {
            return false;
        }
        
        return true;
    }
    
    /**
     * Clean and secure filename
     */
    public static function secureFilename($filename) {
        // Remove dangerous characters
        $filename = preg_replace('/[^a-zA-Z0-9._-]/', '', $filename);
        
        // Limit length
        if (strlen($filename) > 255) {
            $filename = substr($filename, 0, 255);
        }
        
        // Add timestamp to prevent conflicts and enhance security
        $extension = pathinfo($filename, PATHINFO_EXTENSION);
        $name = pathinfo($filename, PATHINFO_FILENAME);
        
        return $name . '_' . time() . '.' . $extension;
    }
}

// Initialize security measures if not already done
if (!defined('SECURITY_INITIALIZED')) {
    define('SECURITY_INIT', true);
    define('SECURITY_INITIALIZED', true);
    SecurityConfig::init();
}
?>
