<?php
// Define API access constant
define('API_ACCESS', true);

require_once '../config.php';

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Require authentication
require_auth();

$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            if (isset($_GET['key'])) {
                getSetting($_GET['key']);
            } else {
                getSettings();
            }
            break;
        case 'POST':
            createSetting();
            break;
        case 'PUT':
            if (isset($_GET['key'])) {
                updateSetting($_GET['key']);
            } else {
                updateSettings();
            }
            break;
        case 'DELETE':
            if (isset($_GET['key'])) {
                deleteSetting($_GET['key']);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'Setting key required for delete']);
            }
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}

function getSettings() {
    global $pdo;
    
    try {
        $category = isset($_GET['category']) ? $_GET['category'] : '';
        
        $whereClause = !empty($category) ? "WHERE category = ?" : "";
        $params = !empty($category) ? [$category] : [];
        
        $stmt = $pdo->prepare("SELECT * FROM settings $whereClause ORDER BY category, setting_key");
        $stmt->execute($params);
        $settings = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Group settings by category
        $groupedSettings = [];
        foreach ($settings as $setting) {
            $category = $setting['category'] ?? 'general';
            if (!isset($groupedSettings[$category])) {
                $groupedSettings[$category] = [];
            }
            $groupedSettings[$category][$setting['setting_key']] = [
                'value' => $setting['setting_value'],
                'description' => $setting['description'],
                'type' => $setting['value_type'],
                'updated_at' => $setting['updated_at']
            ];
        }
        
        echo json_encode([
            'success' => true,
            'data' => $groupedSettings
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}

function getSetting($key) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("SELECT * FROM settings WHERE setting_key = ?");
        $stmt->execute([$key]);
        $setting = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$setting) {
            http_response_code(404);
            echo json_encode(['error' => 'Setting not found']);
            return;
        }
        
        echo json_encode([
            'success' => true,
            'data' => [
                'key' => $setting['setting_key'],
                'value' => $setting['setting_value'],
                'description' => $setting['description'],
                'type' => $setting['value_type'],
                'category' => $setting['category'],
                'updated_at' => $setting['updated_at']
            ]
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}

function createSetting() {
    global $pdo;
    
    try {
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid JSON input']);
            return;
        }
        
        // Validate required fields
        $required = ['setting_key', 'setting_value'];
        foreach ($required as $field) {
            if (!isset($input[$field])) {
                http_response_code(400);
                echo json_encode(['error' => "Field '$field' is required"]);
                return;
            }
        }
        
        // Check if setting already exists
        $stmt = $pdo->prepare("SELECT id FROM settings WHERE setting_key = ?");
        $stmt->execute([$input['setting_key']]);
        if ($stmt->fetch()) {
            http_response_code(400);
            echo json_encode(['error' => 'Setting already exists']);
            return;
        }
        
        $sql = "INSERT INTO settings (setting_key, setting_value, description, value_type, category, created_by, created_at) VALUES (?, ?, ?, ?, ?, ?, NOW())";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            $input['setting_key'],
            $input['setting_value'],
            $input['description'] ?? null,
            $input['value_type'] ?? 'string',
            $input['category'] ?? 'general',
            $_SESSION['admin_id']
        ]);
        
        echo json_encode([
            'success' => true,
            'message' => 'Setting created successfully'
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}

function updateSetting($key) {
    global $pdo;
    
    try {
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid JSON input']);
            return;
        }
        
        // Check if setting exists
        $stmt = $pdo->prepare("SELECT id FROM settings WHERE setting_key = ?");
        $stmt->execute([$key]);
        if (!$stmt->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Setting not found']);
            return;
        }
        
        $updateFields = [];
        $params = [];
        
        $allowedFields = ['setting_value', 'description', 'value_type', 'category'];
        foreach ($allowedFields as $field) {
            if (isset($input[$field])) {
                $updateFields[] = "$field = ?";
                $params[] = $input[$field];
            }
        }
        
        if (empty($updateFields)) {
            http_response_code(400);
            echo json_encode(['error' => 'No valid fields to update']);
            return;
        }
        
        $updateFields[] = "updated_at = NOW()";
        $params[] = $key;
        
        $sql = "UPDATE settings SET " . implode(', ', $updateFields) . " WHERE setting_key = ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        
        echo json_encode([
            'success' => true,
            'message' => 'Setting updated successfully'
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}

function updateSettings() {
    global $pdo;
    
    try {
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input || !isset($input['settings'])) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid JSON input or missing settings']);
            return;
        }
        
        $pdo->beginTransaction();
        
        foreach ($input['settings'] as $setting) {
            if (!isset($setting['key']) || !isset($setting['value'])) {
                continue;
            }
            
            // Check if setting exists
            $stmt = $pdo->prepare("SELECT id FROM settings WHERE setting_key = ?");
            $stmt->execute([$setting['key']]);
            $exists = $stmt->fetch();
            
            if ($exists) {
                // Update existing setting
                $stmt = $pdo->prepare("UPDATE settings SET setting_value = ?, updated_at = NOW() WHERE setting_key = ?");
                $stmt->execute([$setting['value'], $setting['key']]);
            } else {
                // Create new setting
                $stmt = $pdo->prepare("INSERT INTO settings (setting_key, setting_value, description, value_type, category, created_by, created_at) VALUES (?, ?, ?, ?, ?, ?, NOW())");
                $stmt->execute([
                    $setting['key'],
                    $setting['value'],
                    $setting['description'] ?? null,
                    $setting['type'] ?? 'string',
                    $setting['category'] ?? 'general',
                    $_SESSION['admin_id']
                ]);
            }
        }
        
        $pdo->commit();
        
        echo json_encode([
            'success' => true,
            'message' => 'Settings updated successfully'
        ]);
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
}

function deleteSetting($key) {
    global $pdo;
    
    try {
        // Check if setting exists
        $stmt = $pdo->prepare("SELECT id FROM settings WHERE setting_key = ?");
        $stmt->execute([$key]);
        if (!$stmt->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Setting not found']);
            return;
        }
        
        $stmt = $pdo->prepare("DELETE FROM settings WHERE setting_key = ?");
        $stmt->execute([$key]);
        
        echo json_encode([
            'success' => true,
            'message' => 'Setting deleted successfully'
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}
?>
