<?php
/**
 * Real-time Member Status API
 * Handles member activity tracking and online status updates
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once '../config.php';
require_once '../security/security.php';

// Initialize security
$security = new Security();

// Validate API access
if (!defined('API_ACCESS') || !API_ACCESS) {
    http_response_code(403);
    echo json_encode(['error' => 'API access denied']);
    exit();
}

$method = $_SERVER['REQUEST_METHOD'];
$path = $_GET['path'] ?? '';

try {
    switch ($method) {
        case 'GET':
            handleGetRequest($path, $security);
            break;
        case 'POST':
            handlePostRequest($path, $security);
            break;
        case 'PUT':
            handlePutRequest($path, $security);
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    error_log("Real-time Status API Error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error']);
}

/**
 * Handle GET requests
 */
function handleGetRequest($path, $security) {
    switch ($path) {
        case 'online-status':
            getOnlineStatus($security);
            break;
        case 'member-activity':
            getMemberActivity($security);
            break;
        case 'activity-feed':
            getActivityFeed($security);
            break;
        case 'heartbeat':
            handleHeartbeat($security);
            break;
        default:
            http_response_code(404);
            echo json_encode(['error' => 'Endpoint not found']);
            break;
    }
}

/**
 * Handle POST requests
 */
function handlePostRequest($path, $security) {
    switch ($path) {
        case 'track-activity':
            trackMemberActivity($security);
            break;
        case 'login-activity':
            trackLoginActivity($security);
            break;
        case 'logout-activity':
            trackLogoutActivity($security);
            break;
        default:
            http_response_code(404);
            echo json_encode(['error' => 'Endpoint not found']);
            break;
    }
}

/**
 * Handle PUT requests
 */
function handlePutRequest($path, $security) {
    switch ($path) {
        case 'heartbeat':
            updateHeartbeat($security);
            break;
        case 'status':
            updateMemberStatus($security);
            break;
        default:
            http_response_code(404);
            echo json_encode(['error' => 'Endpoint not found']);
            break;
    }
}

/**
 * Get online status for members
 */
function getOnlineStatus($security) {
    global $pdo;
    
    $limit = intval($_GET['limit'] ?? 50);
    $status_filter = $_GET['status'] ?? 'all';
    $include_offline = $_GET['include_offline'] ?? 'false';
    
    $where_conditions = ["m.approval_status = 'approved'", "m.status = 'active'"];
    
    if ($status_filter === 'online') {
        $where_conditions[] = "m.is_online = 1";
    } elseif ($status_filter === 'active') {
        $where_conditions[] = "m.last_activity_at > DATE_SUB(NOW(), INTERVAL 1 HOUR)";
    }
    
    if ($include_offline === 'false') {
        $where_conditions[] = "(m.is_online = 1 OR m.last_activity_at > DATE_SUB(NOW(), INTERVAL 1 HOUR))";
    }
    
    $where_clause = "WHERE " . implode(" AND ", $where_conditions);
    
    $stmt = $pdo->prepare("
        SELECT 
            m.id,
            m.kso_id,
            m.first_name,
            m.last_name,
            m.email,
            m.is_online,
            m.last_activity_at,
            mos.last_login,
            mos.last_logout,
            mos.activity_type,
            mos.session_id,
            mos.ip_address,
            CASE 
                WHEN m.is_online = 1 THEN 'online'
                WHEN m.last_activity_at > DATE_SUB(NOW(), INTERVAL 5 MINUTE) THEN 'recently_active'
                WHEN m.last_activity_at > DATE_SUB(NOW(), INTERVAL 1 HOUR) THEN 'away'
                ELSE 'offline'
            END as status_label,
            TIMESTAMPDIFF(MINUTE, m.last_activity_at, NOW()) as minutes_since_activity,
            TIMESTAMPDIFF(SECOND, m.last_activity_at, NOW()) as seconds_since_activity,
            mp.filename as profile_photo,
            CASE 
                WHEN mp.filename IS NOT NULL THEN 
                    CONCAT('/uploads/member-photos/thumbnails/', mp.filename)
                ELSE NULL 
            END as profile_thumbnail,
            CASE 
                WHEN mp.filename IS NOT NULL THEN 
                    CONCAT('/uploads/member-photos/thumbnails/', 
                           SUBSTRING_INDEX(mp.filename, '.', 1), '.webp')
                ELSE NULL 
            END as profile_webp
        FROM members m
        LEFT JOIN member_online_status mos ON m.id = mos.member_id
        LEFT JOIN member_photos mp ON m.photo_id = mp.id
        {$where_clause}
        ORDER BY 
            m.is_online DESC, 
            m.last_activity_at DESC
        LIMIT ?
    ");
    
    $stmt->execute([$limit]);
    $members = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get counts
    $counts = getStatusCounts();
    
    echo json_encode([
        'success' => true,
        'members' => $members,
        'counts' => $counts,
        'timestamp' => time()
    ]);
}

/**
 * Get member activity history
 */
function getMemberActivity($security) {
    global $pdo;
    
    $member_id = intval($_GET['member_id'] ?? 0);
    $limit = intval($_GET['limit'] ?? 20);
    $hours = intval($_GET['hours'] ?? 24);
    
    if (!$member_id) {
        http_response_code(400);
        echo json_encode(['error' => 'Member ID required']);
        return;
    }
    
    $stmt = $pdo->prepare("
        SELECT 
            activity_type,
            activity_details,
            ip_address,
            created_at,
            TIMESTAMPDIFF(MINUTE, created_at, NOW()) as minutes_ago
        FROM member_activity_log 
        WHERE member_id = ? 
        AND created_at > DATE_SUB(NOW(), INTERVAL ? HOUR)
        ORDER BY created_at DESC 
        LIMIT ?
    ");
    
    $stmt->execute([$member_id, $hours, $limit]);
    $activities = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo json_encode([
        'success' => true,
        'activities' => $activities,
        'member_id' => $member_id
    ]);
}

/**
 * Get activity feed for dashboard
 */
function getActivityFeed($security) {
    global $pdo;
    
    $limit = intval($_GET['limit'] ?? 20);
    $activity_types = $_GET['types'] ?? 'login,logout,photo_upload,profile_update';
    $types_array = explode(',', $activity_types);
    
    $placeholders = str_repeat('?,', count($types_array) - 1) . '?';
    
    $stmt = $pdo->prepare("
        SELECT 
            mal.activity_type,
            mal.activity_details,
            mal.created_at,
            m.first_name,
            m.last_name,
            m.kso_id,
            TIMESTAMPDIFF(MINUTE, mal.created_at, NOW()) as minutes_ago,
            CASE 
                WHEN mp.filename IS NOT NULL THEN 
                    CONCAT('/uploads/member-photos/thumbnails/', mp.filename)
                ELSE NULL 
            END as profile_thumbnail
        FROM member_activity_log mal
        JOIN members m ON mal.member_id = m.id
        LEFT JOIN member_photos mp ON m.photo_id = mp.id
        WHERE mal.activity_type IN ({$placeholders})
        AND m.approval_status = 'approved'
        ORDER BY mal.created_at DESC 
        LIMIT ?
    ");
    
    $params = array_merge($types_array, [$limit]);
    $stmt->execute($params);
    $activities = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo json_encode([
        'success' => true,
        'activities' => $activities,
        'timestamp' => time()
    ]);
}

/**
 * Handle heartbeat ping
 */
function handleHeartbeat($security) {
    $member_id = intval($_GET['member_id'] ?? 0);
    
    if (!$member_id) {
        http_response_code(400);
        echo json_encode(['error' => 'Member ID required']);
        return;
    }
    
    updateMemberHeartbeat($member_id);
    
    echo json_encode([
        'success' => true,
        'timestamp' => time(),
        'member_id' => $member_id
    ]);
}

/**
 * Track member activity
 */
function trackMemberActivity($security) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    $member_id = intval($data['member_id'] ?? 0);
    $activity_type = $security->sanitizeInput($data['activity_type'] ?? 'page_view');
    $details = $data['details'] ?? null;
    
    if (!$member_id) {
        http_response_code(400);
        echo json_encode(['error' => 'Member ID required']);
        return;
    }
    
    // Rate limiting for activity tracking
    if (!$security->checkRateLimit("activity_tracking_{$member_id}", 60, 60)) {
        http_response_code(429);
        echo json_encode(['error' => 'Activity tracking rate limit exceeded']);
        return;
    }
    
    $activity_id = logMemberActivity($member_id, $activity_type, $details);
    
    echo json_encode([
        'success' => true,
        'activity_id' => $activity_id,
        'timestamp' => time()
    ]);
}

/**
 * Track login activity
 */
function trackLoginActivity($security) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    $member_id = intval($data['member_id'] ?? 0);
    $session_id = $security->sanitizeInput($data['session_id'] ?? session_id());
    
    if (!$member_id) {
        http_response_code(400);
        echo json_encode(['error' => 'Member ID required']);
        return;
    }
    
    // Log login activity
    $activity_id = logMemberActivity($member_id, 'login', [
        'session_id' => $session_id,
        'login_time' => date('Y-m-d H:i:s')
    ]);
    
    // Update online status
    updateMemberOnlineStatus($member_id, true, 'login', $session_id);
    
    echo json_encode([
        'success' => true,
        'activity_id' => $activity_id,
        'timestamp' => time()
    ]);
}

/**
 * Track logout activity
 */
function trackLogoutActivity($security) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    $member_id = intval($data['member_id'] ?? 0);
    $session_id = $security->sanitizeInput($data['session_id'] ?? session_id());
    
    if (!$member_id) {
        http_response_code(400);
        echo json_encode(['error' => 'Member ID required']);
        return;
    }
    
    // Log logout activity
    $activity_id = logMemberActivity($member_id, 'logout', [
        'session_id' => $session_id,
        'logout_time' => date('Y-m-d H:i:s')
    ]);
    
    // Update online status
    updateMemberOnlineStatus($member_id, false, 'logout', $session_id);
    
    echo json_encode([
        'success' => true,
        'activity_id' => $activity_id,
        'timestamp' => time()
    ]);
}

/**
 * Update heartbeat
 */
function updateHeartbeat($security) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    $member_id = intval($data['member_id'] ?? 0);
    
    if (!$member_id) {
        http_response_code(400);
        echo json_encode(['error' => 'Member ID required']);
        return;
    }
    
    updateMemberHeartbeat($member_id);
    
    echo json_encode([
        'success' => true,
        'timestamp' => time()
    ]);
}

/**
 * Update member status manually
 */
function updateMemberStatus($security) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    $member_id = intval($data['member_id'] ?? 0);
    $is_online = $data['is_online'] ?? true;
    $activity_type = $security->sanitizeInput($data['activity_type'] ?? 'active');
    
    if (!$member_id) {
        http_response_code(400);
        echo json_encode(['error' => 'Member ID required']);
        return;
    }
    
    updateMemberOnlineStatus($member_id, $is_online, $activity_type);
    
    echo json_encode([
        'success' => true,
        'timestamp' => time()
    ]);
}

/**
 * Helper functions
 */

function getStatusCounts() {
    global $pdo;
    
    $stmt = $pdo->prepare("
        SELECT 
            SUM(CASE WHEN is_online = 1 THEN 1 ELSE 0 END) as online,
            SUM(CASE WHEN last_activity_at > DATE_SUB(NOW(), INTERVAL 5 MINUTE) AND is_online = 0 THEN 1 ELSE 0 END) as recently_active,
            SUM(CASE WHEN last_activity_at > DATE_SUB(NOW(), INTERVAL 1 HOUR) AND last_activity_at <= DATE_SUB(NOW(), INTERVAL 5 MINUTE) THEN 1 ELSE 0 END) as away,
            COUNT(*) as total_active
        FROM members 
        WHERE approval_status = 'approved' AND status = 'active'
    ");
    
    $stmt->execute();
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

function logMemberActivity($member_id, $activity_type, $details = null) {
    global $pdo;
    
    $session_id = session_id();
    $ip_address = $_SERVER['REMOTE_ADDR'] ?? null;
    $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? null;
    
    $stmt = $pdo->prepare("
        INSERT INTO member_activity_log 
        (member_id, activity_type, activity_details, ip_address, user_agent, session_id)
        VALUES (?, ?, ?, ?, ?, ?)
    ");
    
    $stmt->execute([
        $member_id,
        $activity_type,
        $details ? json_encode($details) : null,
        $ip_address,
        $user_agent,
        $session_id
    ]);
    
    return $pdo->lastInsertId();
}

function updateMemberOnlineStatus($member_id, $is_online, $activity_type = 'active', $session_id = null) {
    global $pdo;
    
    $session_id = $session_id ?: session_id();
    $ip_address = $_SERVER['REMOTE_ADDR'] ?? null;
    $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? null;
    
    // Update members table
    $stmt = $pdo->prepare("
        UPDATE members 
        SET 
            is_online = ?,
            last_activity_at = NOW(),
            updated_at = NOW()
        WHERE id = ?
    ");
    $stmt->execute([$is_online ? 1 : 0, $member_id]);
    
    // Update or insert online status record
    $stmt = $pdo->prepare("
        INSERT INTO member_online_status 
            (member_id, is_online, last_activity, session_id, ip_address, user_agent, activity_type, last_login, last_logout)
        VALUES 
            (?, ?, NOW(), ?, ?, ?, ?, 
             CASE WHEN ? = 'login' THEN NOW() ELSE NULL END,
             CASE WHEN ? = 'logout' THEN NOW() ELSE NULL END)
        ON DUPLICATE KEY UPDATE
            is_online = VALUES(is_online),
            last_activity = NOW(),
            session_id = VALUES(session_id),
            ip_address = VALUES(ip_address),
            user_agent = VALUES(user_agent),
            activity_type = VALUES(activity_type),
            last_login = CASE WHEN VALUES(activity_type) = 'login' THEN NOW() ELSE last_login END,
            last_logout = CASE WHEN VALUES(activity_type) = 'logout' THEN NOW() ELSE last_logout END,
            updated_at = NOW()
    ");
    
    $stmt->execute([
        $member_id,
        $is_online ? 1 : 0,
        $session_id,
        $ip_address,
        $user_agent,
        $activity_type,
        $activity_type, // for login check
        $activity_type  // for logout check
    ]);
}

function updateMemberHeartbeat($member_id) {
    global $pdo;
    
    // Simple heartbeat update - just update last activity
    $stmt = $pdo->prepare("
        UPDATE members 
        SET 
            last_activity_at = NOW(),
            is_online = 1
        WHERE id = ?
    ");
    $stmt->execute([$member_id]);
    
    // Update online status
    $stmt = $pdo->prepare("
        UPDATE member_online_status 
        SET 
            last_activity = NOW(),
            is_online = 1,
            activity_type = 'active'
        WHERE member_id = ?
    ");
    $stmt->execute([$member_id]);
}
?>
