<?php
session_start();
require_once '../config/database.php';
require_once '../services/EmailService.php';
require_once '../services/EmailTemplateManager.php';
require_once '../services/EmailQueueProcessor.php';

// Check if user is logged in and is admin
if (!isset($_SESSION['user_id']) || $_SESSION['user_type'] !== 'admin') {
    header("Location: ../login.php");
    exit();
}

$emailService = new EmailService();
$templateManager = new EmailTemplateManager();
$queueProcessor = new EmailQueueProcessor();

// Handle AJAX requests
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    header('Content-Type: application/json');
    
    switch ($_POST['action']) {
        case 'test_config':
            $result = $emailService->testEmailConfiguration($_POST['test_email'] ?? null);
            echo json_encode($result);
            break;
            
        case 'update_config':
            $result = $emailService->updateConfig($_POST['settings']);
            echo json_encode($result);
            break;
            
        case 'send_test_email':
            $result = $emailService->sendEmail($_POST['to'], $_POST['subject'], $_POST['body']);
            echo json_encode($result);
            break;
            
        case 'process_queue':
            $result = $queueProcessor->processQueue($_POST['limit'] ?? 10);
            echo json_encode($result);
            break;
            
        case 'retry_failed':
            $result = $queueProcessor->retryFailedQueue($_POST['limit'] ?? 5);
            echo json_encode($result);
            break;
            
        case 'clear_queue':
            $result = $queueProcessor->clearQueue($_POST['status'] ?? null);
            echo json_encode($result);
            break;
            
        case 'pause_queue':
            $result = $queueProcessor->pauseQueue();
            echo json_encode($result);
            break;
            
        case 'resume_queue':
            $result = $queueProcessor->resumeQueue();
            echo json_encode($result);
            break;
            
        case 'create_template':
            $result = $templateManager->createTemplate(
                $_POST['name'],
                $_POST['subject'],
                $_POST['body'],
                $_POST['description'] ?? '',
                $_POST['category'] ?? 'general'
            );
            echo json_encode($result);
            break;
            
        case 'update_template':
            $result = $templateManager->updateTemplate(
                $_POST['id'],
                $_POST['name'],
                $_POST['subject'],
                $_POST['body'],
                $_POST['description'] ?? '',
                $_POST['category'] ?? 'general'
            );
            echo json_encode($result);
            break;
            
        case 'delete_template':
            $result = $templateManager->deleteTemplate($_POST['id']);
            echo json_encode($result);
            break;
            
        case 'toggle_template':
            $result = $templateManager->toggleTemplate($_POST['id']);
            echo json_encode($result);
            break;
            
        case 'preview_template':
            $templateData = [
                'subject' => $_POST['subject'],
                'body' => $_POST['body']
            ];
            $result = $templateManager->previewTemplate($templateData, $_POST['sample_data'] ?? []);
            echo json_encode(['success' => true, 'preview' => $result]);
            break;
            
        case 'create_defaults':
            $result = $templateManager->createDefaultTemplates();
            echo json_encode($result);
            break;
    }
    exit();
}

// Get data for dashboard
$queueStats = $queueProcessor->getQueueStats();
$emailLogs = $emailService->getEmailLogs(20);
$templates = $templateManager->getAllTemplates();
$queueItems = $queueProcessor->getQueueStatus(20);
$emailConfig = $emailService->getConfig();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Email Management Dashboard - KSO Chandigarh</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/admin-lte/3.2.0/css/adminlte.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/select2/4.0.13/css/select2.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.2/codemirror.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.2/theme/monokai.min.css">
    <style>
        .email-preview {
            border: 1px solid #ddd;
            border-radius: 5px;
            padding: 15px;
            background: #f9f9f9;
            max-height: 400px;
            overflow-y: auto;
        }
        .queue-status {
            padding: 5px 10px;
            border-radius: 3px;
            color: white;
            font-size: 12px;
        }
        .queue-status.pending { background-color: #ffc107; }
        .queue-status.processing { background-color: #17a2b8; }
        .queue-status.sent { background-color: #28a745; }
        .queue-status.failed { background-color: #dc3545; }
        .queue-status.paused { background-color: #6c757d; }
        
        .priority-badge {
            padding: 2px 6px;
            border-radius: 3px;
            font-size: 11px;
            color: white;
        }
        .priority-badge.high { background-color: #dc3545; }
        .priority-badge.normal { background-color: #28a745; }
        .priority-badge.low { background-color: #6c757d; }
        
        .stat-card {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
        }
        
        .template-card {
            border: 1px solid #ddd;
            border-radius: 8px;
            padding: 15px;
            margin-bottom: 15px;
            transition: box-shadow 0.3s;
        }
        
        .template-card:hover {
            box-shadow: 0 4px 8px rgba(0,0,0,0.1);
        }
        
        .template-inactive {
            opacity: 0.6;
        }
        
        .CodeMirror {
            height: 200px;
            border: 1px solid #ddd;
            border-radius: 4px;
        }
    </style>
</head>
<body class="hold-transition sidebar-mini layout-fixed">
    <div class="wrapper">
        <?php include '../templates/navbar.php'; ?>
        <?php include '../templates/sidebar.php'; ?>
        
        <div class="content-wrapper">
            <div class="content-header">
                <div class="container-fluid">
                    <div class="row mb-2">
                        <div class="col-sm-6">
                            <h1 class="m-0">Email Management Dashboard</h1>
                        </div>
                        <div class="col-sm-6">
                            <ol class="breadcrumb float-sm-right">
                                <li class="breadcrumb-item"><a href="../dashboard.php">Dashboard</a></li>
                                <li class="breadcrumb-item active">Email Management</li>
                            </ol>
                        </div>
                    </div>
                </div>
            </div>
            
            <section class="content">
                <div class="container-fluid">
                    
                    <!-- Email Statistics -->
                    <div class="row">
                        <div class="col-lg-3 col-6">
                            <div class="small-box bg-info">
                                <div class="inner">
                                    <h3><?php echo $queueStats['last_24_hours']['total'] ?? 0; ?></h3>
                                    <p>Emails Last 24h</p>
                                </div>
                                <div class="icon">
                                    <i class="fas fa-envelope"></i>
                                </div>
                            </div>
                        </div>
                        <div class="col-lg-3 col-6">
                            <div class="small-box bg-success">
                                <div class="inner">
                                    <h3><?php echo $queueStats['last_24_hours']['sent'] ?? 0; ?></h3>
                                    <p>Sent Successfully</p>
                                </div>
                                <div class="icon">
                                    <i class="fas fa-paper-plane"></i>
                                </div>
                            </div>
                        </div>
                        <div class="col-lg-3 col-6">
                            <div class="small-box bg-warning">
                                <div class="inner">
                                    <h3><?php echo $queueStats['by_status']['pending'] ?? 0; ?></h3>
                                    <p>Pending in Queue</p>
                                </div>
                                <div class="icon">
                                    <i class="fas fa-clock"></i>
                                </div>
                            </div>
                        </div>
                        <div class="col-lg-3 col-6">
                            <div class="small-box bg-danger">
                                <div class="inner">
                                    <h3><?php echo $queueStats['by_status']['failed'] ?? 0; ?></h3>
                                    <p>Failed</p>
                                </div>
                                <div class="icon">
                                    <i class="fas fa-exclamation-triangle"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Main Tabs -->
                    <div class="card">
                        <div class="card-header p-2">
                            <ul class="nav nav-pills">
                                <li class="nav-item"><a class="nav-link active" href="#queue" data-toggle="tab">Email Queue</a></li>
                                <li class="nav-item"><a class="nav-link" href="#templates" data-toggle="tab">Templates</a></li>
                                <li class="nav-item"><a class="nav-link" href="#logs" data-toggle="tab">Email Logs</a></li>
                                <li class="nav-item"><a class="nav-link" href="#settings" data-toggle="tab">Settings</a></li>
                                <li class="nav-item"><a class="nav-link" href="#test" data-toggle="tab">Test Email</a></li>
                            </ul>
                        </div>
                        <div class="card-body">
                            <div class="tab-content">
                                
                                <!-- Email Queue Tab -->
                                <div class="active tab-pane" id="queue">
                                    <div class="row mb-3">
                                        <div class="col-md-6">
                                            <h4>Email Queue Management</h4>
                                        </div>
                                        <div class="col-md-6 text-right">
                                            <button class="btn btn-primary" onclick="processQueue()">
                                                <i class="fas fa-play"></i> Process Queue
                                            </button>
                                            <button class="btn btn-warning" onclick="retryFailed()">
                                                <i class="fas fa-redo"></i> Retry Failed
                                            </button>
                                            <div class="btn-group">
                                                <button class="btn btn-secondary" onclick="pauseQueue()">
                                                    <i class="fas fa-pause"></i> Pause
                                                </button>
                                                <button class="btn btn-secondary" onclick="resumeQueue()">
                                                    <i class="fas fa-play"></i> Resume
                                                </button>
                                                <button class="btn btn-danger" onclick="clearQueue()">
                                                    <i class="fas fa-trash"></i> Clear
                                                </button>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="table-responsive">
                                        <table class="table table-striped" id="queueTable">
                                            <thead>
                                                <tr>
                                                    <th>ID</th>
                                                    <th>To</th>
                                                    <th>Subject</th>
                                                    <th>Status</th>
                                                    <th>Priority</th>
                                                    <th>Created</th>
                                                    <th>Actions</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php foreach ($queueItems as $item): ?>
                                                <tr>
                                                    <td><?php echo $item['id']; ?></td>
                                                    <td><?php echo htmlspecialchars($item['to_email']); ?></td>
                                                    <td><?php echo htmlspecialchars(substr($item['subject'], 0, 50)); ?></td>
                                                    <td><span class="queue-status <?php echo $item['status']; ?>"><?php echo ucfirst($item['status']); ?></span></td>
                                                    <td><span class="priority-badge <?php echo $item['priority_text']; ?>"><?php echo ucfirst($item['priority_text']); ?></span></td>
                                                    <td><?php echo date('Y-m-d H:i', strtotime($item['created_at'])); ?></td>
                                                    <td>
                                                        <button class="btn btn-sm btn-info" onclick="viewQueueItem(<?php echo $item['id']; ?>)">
                                                            <i class="fas fa-eye"></i>
                                                        </button>
                                                        <button class="btn btn-sm btn-danger" onclick="deleteQueueItem(<?php echo $item['id']; ?>)">
                                                            <i class="fas fa-trash"></i>
                                                        </button>
                                                    </td>
                                                </tr>
                                                <?php endforeach; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                </div>
                                
                                <!-- Templates Tab -->
                                <div class="tab-pane" id="templates">
                                    <div class="row mb-3">
                                        <div class="col-md-6">
                                            <h4>Email Templates</h4>
                                        </div>
                                        <div class="col-md-6 text-right">
                                            <button class="btn btn-success" onclick="createDefaultTemplates()">
                                                <i class="fas fa-plus"></i> Create Defaults
                                            </button>
                                            <button class="btn btn-primary" onclick="showTemplateModal()">
                                                <i class="fas fa-plus"></i> New Template
                                            </button>
                                        </div>
                                    </div>
                                    
                                    <div class="row">
                                        <?php foreach ($templates as $template): ?>
                                        <div class="col-md-6">
                                            <div class="template-card <?php echo $template['is_active'] ? '' : 'template-inactive'; ?>">
                                                <div class="row">
                                                    <div class="col-md-8">
                                                        <h5><?php echo htmlspecialchars($template['name']); ?></h5>
                                                        <p class="text-muted"><?php echo htmlspecialchars($template['description']); ?></p>
                                                        <small class="text-info">Category: <?php echo ucfirst($template['category']); ?></small>
                                                    </div>
                                                    <div class="col-md-4 text-right">
                                                        <div class="btn-group-vertical">
                                                            <button class="btn btn-sm btn-info" onclick="previewTemplate(<?php echo $template['id']; ?>)">
                                                                <i class="fas fa-eye"></i>
                                                            </button>
                                                            <button class="btn btn-sm btn-warning" onclick="editTemplate(<?php echo $template['id']; ?>)">
                                                                <i class="fas fa-edit"></i>
                                                            </button>
                                                            <button class="btn btn-sm btn-secondary" onclick="toggleTemplate(<?php echo $template['id']; ?>)">
                                                                <i class="fas fa-power-off"></i>
                                                            </button>
                                                            <button class="btn btn-sm btn-danger" onclick="deleteTemplate(<?php echo $template['id']; ?>)">
                                                                <i class="fas fa-trash"></i>
                                                            </button>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                                
                                <!-- Email Logs Tab -->
                                <div class="tab-pane" id="logs">
                                    <h4>Email Logs</h4>
                                    <div class="table-responsive">
                                        <table class="table table-striped" id="logsTable">
                                            <thead>
                                                <tr>
                                                    <th>ID</th>
                                                    <th>To</th>
                                                    <th>Subject</th>
                                                    <th>Template</th>
                                                    <th>Status</th>
                                                    <th>Sent At</th>
                                                    <th>Actions</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php foreach ($emailLogs as $log): ?>
                                                <tr>
                                                    <td><?php echo $log['id']; ?></td>
                                                    <td><?php echo htmlspecialchars($log['to_email']); ?></td>
                                                    <td><?php echo htmlspecialchars(substr($log['subject'], 0, 50)); ?></td>
                                                    <td><?php echo $log['template_used'] ? htmlspecialchars($log['template_used']) : '-'; ?></td>
                                                    <td><span class="queue-status <?php echo $log['status']; ?>"><?php echo ucfirst($log['status']); ?></span></td>
                                                    <td><?php echo $log['sent_at'] ? date('Y-m-d H:i', strtotime($log['sent_at'])) : '-'; ?></td>
                                                    <td>
                                                        <button class="btn btn-sm btn-info" onclick="viewLogDetails(<?php echo $log['id']; ?>)">
                                                            <i class="fas fa-eye"></i>
                                                        </button>
                                                    </td>
                                                </tr>
                                                <?php endforeach; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                </div>
                                
                                <!-- Settings Tab -->
                                <div class="tab-pane" id="settings">
                                    <h4>Email Configuration</h4>
                                    <form id="emailSettingsForm">
                                        <div class="row">
                                            <div class="col-md-6">
                                                <div class="form-group">
                                                    <label>SMTP Host</label>
                                                    <input type="text" class="form-control" name="smtp_host" value="<?php echo htmlspecialchars($emailConfig->get('smtp_host')); ?>">
                                                </div>
                                                <div class="form-group">
                                                    <label>SMTP Port</label>
                                                    <input type="number" class="form-control" name="smtp_port" value="<?php echo $emailConfig->get('smtp_port'); ?>">
                                                </div>
                                                <div class="form-group">
                                                    <label>SMTP Username</label>
                                                    <input type="text" class="form-control" name="smtp_username" value="<?php echo htmlspecialchars($emailConfig->get('smtp_username')); ?>">
                                                </div>
                                                <div class="form-group">
                                                    <label>SMTP Password</label>
                                                    <input type="password" class="form-control" name="smtp_password" value="<?php echo htmlspecialchars($emailConfig->get('smtp_password')); ?>">
                                                </div>
                                            </div>
                                            <div class="col-md-6">
                                                <div class="form-group">
                                                    <label>From Email</label>
                                                    <input type="email" class="form-control" name="from_email" value="<?php echo htmlspecialchars($emailConfig->get('from_email')); ?>">
                                                </div>
                                                <div class="form-group">
                                                    <label>From Name</label>
                                                    <input type="text" class="form-control" name="from_name" value="<?php echo htmlspecialchars($emailConfig->get('from_name')); ?>">
                                                </div>
                                                <div class="form-group">
                                                    <label>Encryption</label>
                                                    <select class="form-control" name="smtp_encryption">
                                                        <option value="tls" <?php echo $emailConfig->get('smtp_encryption') === 'tls' ? 'selected' : ''; ?>>TLS</option>
                                                        <option value="ssl" <?php echo $emailConfig->get('smtp_encryption') === 'ssl' ? 'selected' : ''; ?>>SSL</option>
                                                        <option value="" <?php echo $emailConfig->get('smtp_encryption') === '' ? 'selected' : ''; ?>>None</option>
                                                    </select>
                                                </div>
                                                <div class="form-group">
                                                    <label>
                                                        <input type="checkbox" name="email_enabled" value="1" <?php echo $emailConfig->get('email_enabled') ? 'checked' : ''; ?>>
                                                        Enable Email Service
                                                    </label>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="form-group">
                                            <button type="submit" class="btn btn-primary">Save Configuration</button>
                                            <button type="button" class="btn btn-secondary" onclick="testConfiguration()">Test Configuration</button>
                                        </div>
                                    </form>
                                </div>
                                
                                <!-- Test Email Tab -->
                                <div class="tab-pane" id="test">
                                    <h4>Send Test Email</h4>
                                    <form id="testEmailForm">
                                        <div class="form-group">
                                            <label>To Email</label>
                                            <input type="email" class="form-control" name="to" required>
                                        </div>
                                        <div class="form-group">
                                            <label>Subject</label>
                                            <input type="text" class="form-control" name="subject" value="Test Email from KSO System">
                                        </div>
                                        <div class="form-group">
                                            <label>Message</label>
                                            <textarea class="form-control" name="body" rows="5">This is a test email sent from the KSO Chandigarh email system.</textarea>
                                        </div>
                                        <button type="submit" class="btn btn-primary">Send Test Email</button>
                                    </form>
                                </div>
                                
                            </div>
                        </div>
                    </div>
                </div>
            </section>
        </div>
        
        <?php include '../templates/footer.php'; ?>
    </div>
    
    <!-- Template Modal -->
    <div class="modal fade" id="templateModal">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h4 class="modal-title">Email Template</h4>
                    <button type="button" class="close" data-dismiss="modal">&times;</button>
                </div>
                <div class="modal-body">
                    <form id="templateForm">
                        <input type="hidden" name="template_id" id="template_id">
                        <div class="form-group">
                            <label>Template Name</label>
                            <input type="text" class="form-control" name="name" id="template_name" required>
                        </div>
                        <div class="form-group">
                            <label>Subject</label>
                            <input type="text" class="form-control" name="subject" id="template_subject" required>
                        </div>
                        <div class="form-group">
                            <label>Category</label>
                            <select class="form-control" name="category" id="template_category">
                                <option value="general">General</option>
                                <option value="membership">Membership</option>
                                <option value="events">Events</option>
                                <option value="finance">Finance</option>
                                <option value="security">Security</option>
                            </select>
                        </div>
                        <div class="form-group">
                            <label>Description</label>
                            <textarea class="form-control" name="description" id="template_description"></textarea>
                        </div>
                        <div class="form-group">
                            <label>Body</label>
                            <textarea id="template_body" name="body"></textarea>
                        </div>
                        <div class="form-group">
                            <button type="button" class="btn btn-info" onclick="previewCurrentTemplate()">Preview</button>
                            <button type="submit" class="btn btn-primary">Save Template</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Preview Modal -->
    <div class="modal fade" id="previewModal">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h4 class="modal-title">Email Preview</h4>
                    <button type="button" class="close" data-dismiss="modal">&times;</button>
                </div>
                <div class="modal-body">
                    <h5>Subject: <span id="preview_subject"></span></h5>
                    <hr>
                    <div id="preview_body" class="email-preview"></div>
                </div>
            </div>
        </div>
    </div>
    
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/4.6.0/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/admin-lte/3.2.0/js/adminlte.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/select2/4.0.13/js/select2.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.2/codemirror.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.2/mode/xml/xml.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.2/mode/htmlmixed/htmlmixed.min.js"></script>
    
    <script>
        let bodyEditor;
        
        $(document).ready(function() {
            // Initialize DataTables
            $('#queueTable, #logsTable').DataTable({
                responsive: true,
                order: [[0, 'desc']]
            });
            
            // Initialize Select2
            $('select').select2();
            
            // Initialize CodeMirror
            bodyEditor = CodeMirror.fromTextArea(document.getElementById('template_body'), {
                mode: 'htmlmixed',
                theme: 'monokai',
                lineNumbers: true,
                autoCloseTags: true,
                matchBrackets: true
            });
            
            // Form submissions
            $('#emailSettingsForm').on('submit', function(e) {
                e.preventDefault();
                updateEmailConfig();
            });
            
            $('#testEmailForm').on('submit', function(e) {
                e.preventDefault();
                sendTestEmail();
            });
            
            $('#templateForm').on('submit', function(e) {
                e.preventDefault();
                saveTemplate();
            });
            
            // Auto-refresh queue status every 30 seconds
            setInterval(refreshQueueStatus, 30000);
        });
        
        function processQueue() {
            $.post('', {action: 'process_queue', limit: 10}, function(response) {
                if (response.processed > 0) {
                    toastr.success(`Processed ${response.processed} emails`);
                    location.reload();
                } else {
                    toastr.info('No emails to process');
                }
            });
        }
        
        function retryFailed() {
            $.post('', {action: 'retry_failed', limit: 5}, function(response) {
                if (response.length > 0) {
                    toastr.success(`Retrying ${response.length} failed emails`);
                    location.reload();
                } else {
                    toastr.info('No failed emails to retry');
                }
            });
        }
        
        function pauseQueue() {
            $.post('', {action: 'pause_queue'}, function(response) {
                if (response.success) {
                    toastr.success(response.message);
                    location.reload();
                } else {
                    toastr.error(response.message);
                }
            });
        }
        
        function resumeQueue() {
            $.post('', {action: 'resume_queue'}, function(response) {
                if (response.success) {
                    toastr.success(response.message);
                    location.reload();
                } else {
                    toastr.error(response.message);
                }
            });
        }
        
        function clearQueue() {
            if (confirm('Are you sure you want to clear the email queue?')) {
                $.post('', {action: 'clear_queue'}, function(response) {
                    if (response.success) {
                        toastr.success(response.message);
                        location.reload();
                    } else {
                        toastr.error(response.message);
                    }
                });
            }
        }
        
        function updateEmailConfig() {
            const formData = $('#emailSettingsForm').serialize();
            $.post('', formData + '&action=update_config', function(response) {
                if (response.success) {
                    toastr.success('Configuration updated successfully');
                } else {
                    toastr.error('Failed to update configuration');
                }
            });
        }
        
        function testConfiguration() {
            const testEmail = prompt('Enter test email address:');
            if (testEmail) {
                $.post('', {action: 'test_config', test_email: testEmail}, function(response) {
                    if (response.success) {
                        toastr.success(response.message);
                    } else {
                        toastr.error(response.message);
                    }
                });
            }
        }
        
        function sendTestEmail() {
            const formData = $('#testEmailForm').serialize();
            $.post('', formData + '&action=send_test_email', function(response) {
                if (response.success) {
                    toastr.success('Test email sent successfully');
                    $('#testEmailForm')[0].reset();
                } else {
                    toastr.error('Failed to send test email: ' + response.message);
                }
            });
        }
        
        function showTemplateModal(templateId = null) {
            if (templateId) {
                // Edit existing template
                // You would fetch template data here
            } else {
                // New template
                $('#templateModal .modal-title').text('New Email Template');
                $('#templateForm')[0].reset();
                $('#template_id').val('');
                if (bodyEditor) {
                    bodyEditor.setValue('');
                }
            }
            $('#templateModal').modal('show');
        }
        
        function saveTemplate() {
            const formData = new FormData($('#templateForm')[0]);
            formData.append('action', $('#template_id').val() ? 'update_template' : 'create_template');
            formData.set('body', bodyEditor.getValue());
            
            $.ajax({
                url: '',
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                success: function(response) {
                    if (response.success) {
                        toastr.success(response.message);
                        $('#templateModal').modal('hide');
                        location.reload();
                    } else {
                        toastr.error(response.message);
                    }
                }
            });
        }
        
        function previewCurrentTemplate() {
            const subject = $('#template_subject').val();
            const body = bodyEditor.getValue();
            
            $.post('', {
                action: 'preview_template',
                subject: subject,
                body: body
            }, function(response) {
                if (response.success) {
                    $('#preview_subject').text(response.preview.subject);
                    $('#preview_body').html(response.preview.body);
                    $('#previewModal').modal('show');
                }
            });
        }
        
        function createDefaultTemplates() {
            $.post('', {action: 'create_defaults'}, function(response) {
                let successCount = 0;
                response.forEach(function(result) {
                    if (result.success) successCount++;
                });
                toastr.success(`Created ${successCount} default templates`);
                location.reload();
            });
        }
        
        function deleteTemplate(id) {
            if (confirm('Are you sure you want to delete this template?')) {
                $.post('', {action: 'delete_template', id: id}, function(response) {
                    if (response.success) {
                        toastr.success(response.message);
                        location.reload();
                    } else {
                        toastr.error(response.message);
                    }
                });
            }
        }
        
        function toggleTemplate(id) {
            $.post('', {action: 'toggle_template', id: id}, function(response) {
                if (response.success) {
                    toastr.success(response.message);
                    location.reload();
                } else {
                    toastr.error(response.message);
                }
            });
        }
        
        function refreshQueueStatus() {
            // Refresh queue table and stats
            location.reload();
        }
    </script>
</body>
</html>
