/*!
 * KSO Chandigarh Management System - Custom JavaScript Framework
 * Copyright (c) 2025 Kuki Students' Organization Chandigarh
 * Licensed under MIT License
 * 
 * Enhanced JavaScript functionality for KSO Chandigarh Management System
 * Built on modern ES6+ standards with progressive enhancement
 * 
 * Version: 1.0.0
 * Author: KSO Chandigarh Development Team
 * Website: https://kso-chandigarh.org
 */

// ========================================================================
// KSO CHANDIGARH GLOBAL NAMESPACE
// ========================================================================

window.KSO = window.KSO || {};

// KSO Configuration
KSO.config = {
    version: '1.0.0',
    organizationName: 'Kuki Students\' Organization Chandigarh',
    shortName: 'KSO Chandigarh',
    website: 'https://kso-chandigarh.org',
    apiBaseUrl: '../api',
    assetsPath: '../assets',
    animationDuration: 300,
    sessionTimeout: 30, // minutes
    debugMode: false
};

// KSO Brand Colors (matching CSS variables)
KSO.colors = {
    primary: '#1e3a8a',
    primaryDark: '#1e40af',
    primaryLight: '#3b82f6',
    secondary: '#dc2626',
    secondaryDark: '#b91c1c',
    secondaryLight: '#ef4444',
    success: '#059669',
    warning: '#d97706',
    info: '#0891b2',
    danger: '#dc2626'
};

// ========================================================================
// KSO CORE UTILITIES
// ========================================================================

KSO.utils = {
    /**
     * Debounce function to limit function calls
     * @param {Function} func - Function to debounce
     * @param {number} wait - Wait time in milliseconds
     * @param {boolean} immediate - Execute immediately
     * @returns {Function} Debounced function
     */
    debounce: function(func, wait, immediate) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                timeout = null;
                if (!immediate) func(...args);
            };
            const callNow = immediate && !timeout;
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
            if (callNow) func(...args);
        };
    },

    /**
     * Throttle function to limit function calls
     * @param {Function} func - Function to throttle
     * @param {number} limit - Time limit in milliseconds
     * @returns {Function} Throttled function
     */
    throttle: function(func, limit) {
        let inThrottle;
        return function(...args) {
            if (!inThrottle) {
                func.apply(this, args);
                inThrottle = true;
                setTimeout(() => inThrottle = false, limit);
            }
        };
    },

    /**
     * Format date for display
     * @param {Date|string} date - Date to format
     * @param {string} format - Format type ('short', 'long', 'time')
     * @returns {string} Formatted date string
     */
    formatDate: function(date, format = 'short') {
        const d = new Date(date);
        const options = {
            short: { year: 'numeric', month: 'short', day: 'numeric' },
            long: { year: 'numeric', month: 'long', day: 'numeric', weekday: 'long' },
            time: { hour: '2-digit', minute: '2-digit', second: '2-digit' },
            datetime: { year: 'numeric', month: 'short', day: 'numeric', hour: '2-digit', minute: '2-digit' }
        };
        return d.toLocaleDateString('en-IN', options[format] || options.short);
    },

    /**
     * Format currency for display
     * @param {number} amount - Amount to format
     * @param {string} currency - Currency code
     * @returns {string} Formatted currency string
     */
    formatCurrency: function(amount, currency = 'INR') {
        return new Intl.NumberFormat('en-IN', {
            style: 'currency',
            currency: currency,
            minimumFractionDigits: 0,
            maximumFractionDigits: 2
        }).format(amount);
    },

    /**
     * Generate unique ID
     * @returns {string} Unique ID
     */
    generateId: function() {
        return 'kso_' + Math.random().toString(36).substr(2, 9) + '_' + Date.now().toString(36);
    },

    /**
     * Validate email format
     * @param {string} email - Email to validate
     * @returns {boolean} Is valid email
     */
    validateEmail: function(email) {
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return emailRegex.test(email);
    },

    /**
     * Validate Indian phone number
     * @param {string} phone - Phone number to validate
     * @returns {boolean} Is valid phone number
     */
    validatePhone: function(phone) {
        const phoneRegex = /^[+]?[91]?[789]\d{9}$/;
        return phoneRegex.test(phone.replace(/\s+/g, ''));
    },

    /**
     * Sanitize HTML content
     * @param {string} html - HTML content to sanitize
     * @returns {string} Sanitized HTML
     */
    sanitizeHtml: function(html) {
        const div = document.createElement('div');
        div.textContent = html;
        return div.innerHTML;
    },

    /**
     * Copy text to clipboard
     * @param {string} text - Text to copy
     * @returns {Promise<boolean>} Success status
     */
    copyToClipboard: async function(text) {
        try {
            await navigator.clipboard.writeText(text);
            return true;
        } catch (err) {
            // Fallback for older browsers
            const textArea = document.createElement('textarea');
            textArea.value = text;
            textArea.style.position = 'fixed';
            textArea.style.opacity = '0';
            document.body.appendChild(textArea);
            textArea.focus();
            textArea.select();
            const successful = document.execCommand('copy');
            document.body.removeChild(textArea);
            return successful;
        }
    }
};

// ========================================================================
// KSO NOTIFICATION SYSTEM
// ========================================================================

KSO.notifications = {
    container: null,
    queue: [],
    maxNotifications: 5,

    /**
     * Initialize notification system
     */
    init: function() {
        this.createContainer();
    },

    /**
     * Create notification container
     */
    createContainer: function() {
        if (this.container) return;

        this.container = document.createElement('div');
        this.container.className = 'kso-notifications-container';
        this.container.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            z-index: 9999;
            max-width: 400px;
            pointer-events: none;
        `;
        document.body.appendChild(this.container);
    },

    /**
     * Show notification
     * @param {string} message - Notification message
     * @param {string} type - Notification type (success, error, warning, info)
     * @param {number} duration - Display duration in milliseconds
     * @param {Object} options - Additional options
     */
    show: function(message, type = 'info', duration = 5000, options = {}) {
        const notification = this.createNotification(message, type, duration, options);
        this.addToQueue(notification);
        this.displayNotification(notification);
    },

    /**
     * Show success notification
     * @param {string} message - Success message
     * @param {number} duration - Display duration
     * @param {Object} options - Additional options
     */
    success: function(message, duration = 4000, options = {}) {
        this.show(message, 'success', duration, options);
    },

    /**
     * Show error notification
     * @param {string} message - Error message
     * @param {number} duration - Display duration
     * @param {Object} options - Additional options
     */
    error: function(message, duration = 6000, options = {}) {
        this.show(message, 'error', duration, options);
    },

    /**
     * Show warning notification
     * @param {string} message - Warning message
     * @param {number} duration - Display duration
     * @param {Object} options - Additional options
     */
    warning: function(message, duration = 5000, options = {}) {
        this.show(message, 'warning', duration, options);
    },

    /**
     * Show info notification
     * @param {string} message - Info message
     * @param {number} duration - Display duration
     * @param {Object} options - Additional options
     */
    info: function(message, duration = 4000, options = {}) {
        this.show(message, 'info', duration, options);
    },

    /**
     * Create notification element
     * @param {string} message - Notification message
     * @param {string} type - Notification type
     * @param {number} duration - Display duration
     * @param {Object} options - Additional options
     * @returns {HTMLElement} Notification element
     */
    createNotification: function(message, type, duration, options) {
        const notification = document.createElement('div');
        const id = KSO.utils.generateId();
        
        notification.id = id;
        notification.className = `kso-notification kso-notification-${type}`;
        notification.style.cssText = `
            background: white;
            border-radius: 8px;
            padding: 16px;
            margin-bottom: 12px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
            border-left: 4px solid ${this.getTypeColor(type)};
            transform: translateX(100%);
            opacity: 0;
            transition: all 0.3s ease;
            pointer-events: auto;
            cursor: pointer;
            position: relative;
        `;

        const icon = this.getTypeIcon(type);
        const closeBtn = '<span class="kso-notification-close" style="position: absolute; top: 8px; right: 12px; cursor: pointer; color: #666; font-size: 16px; line-height: 1;">&times;</span>';
        
        notification.innerHTML = `
            <div style="display: flex; align-items: start; gap: 12px;">
                <div style="color: ${this.getTypeColor(type)}; font-size: 18px; margin-top: 2px;">${icon}</div>
                <div style="flex: 1; font-size: 14px; line-height: 1.4; color: #333;">${message}</div>
            </div>
            ${closeBtn}
        `;

        // Add click handlers
        notification.addEventListener('click', () => this.removeNotification(id));
        
        // Auto-remove after duration
        if (duration > 0) {
            setTimeout(() => this.removeNotification(id), duration);
        }

        return notification;
    },

    /**
     * Get color for notification type
     * @param {string} type - Notification type
     * @returns {string} Color code
     */
    getTypeColor: function(type) {
        const colors = {
            success: KSO.colors.success,
            error: KSO.colors.danger,
            warning: KSO.colors.warning,
            info: KSO.colors.info
        };
        return colors[type] || colors.info;
    },

    /**
     * Get icon for notification type
     * @param {string} type - Notification type
     * @returns {string} Icon HTML
     */
    getTypeIcon: function(type) {
        const icons = {
            success: '✓',
            error: '✕',
            warning: '⚠',
            info: 'ℹ'
        };
        return icons[type] || icons.info;
    },

    /**
     * Add notification to queue
     * @param {HTMLElement} notification - Notification element
     */
    addToQueue: function(notification) {
        this.queue.push(notification);
        
        // Remove oldest if queue exceeds maximum
        if (this.queue.length > this.maxNotifications) {
            const oldest = this.queue.shift();
            this.removeNotification(oldest.id);
        }
    },

    /**
     * Display notification with animation
     * @param {HTMLElement} notification - Notification element
     */
    displayNotification: function(notification) {
        this.container.appendChild(notification);
        
        // Trigger animation
        requestAnimationFrame(() => {
            notification.style.transform = 'translateX(0)';
            notification.style.opacity = '1';
        });
    },

    /**
     * Remove notification
     * @param {string} id - Notification ID
     */
    removeNotification: function(id) {
        const notification = document.getElementById(id);
        if (!notification) return;

        notification.style.transform = 'translateX(100%)';
        notification.style.opacity = '0';
        
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
            this.queue = this.queue.filter(n => n.id !== id);
        }, 300);
    }
};

// ========================================================================
// KSO LOADING SYSTEM
// ========================================================================

KSO.loading = {
    overlay: null,
    activeLoaders: new Set(),

    /**
     * Show loading overlay
     * @param {string} message - Loading message
     * @param {string} target - Target element selector (optional)
     */
    show: function(message = 'Loading...', target = null) {
        const loaderId = KSO.utils.generateId();
        this.activeLoaders.add(loaderId);

        if (target) {
            this.showTargetLoader(target, message, loaderId);
        } else {
            this.showGlobalLoader(message, loaderId);
        }

        return loaderId;
    },

    /**
     * Hide loading overlay
     * @param {string} loaderId - Loader ID to hide
     */
    hide: function(loaderId = null) {
        if (loaderId) {
            this.activeLoaders.delete(loaderId);
            this.hideSpecificLoader(loaderId);
        } else {
            this.activeLoaders.clear();
            this.hideAllLoaders();
        }
    },

    /**
     * Show global loading overlay
     * @param {string} message - Loading message
     * @param {string} loaderId - Loader ID
     */
    showGlobalLoader: function(message, loaderId) {
        if (!this.overlay) {
            this.overlay = document.createElement('div');
            this.overlay.className = 'kso-loading-overlay';
            this.overlay.style.cssText = `
                position: fixed;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background: rgba(0, 0, 0, 0.5);
                display: flex;
                align-items: center;
                justify-content: center;
                z-index: 10000;
                backdrop-filter: blur(2px);
            `;
            document.body.appendChild(this.overlay);
        }

        this.overlay.innerHTML = `
            <div class="kso-loading-content" style="
                background: white;
                border-radius: 8px;
                padding: 24px;
                text-align: center;
                box-shadow: 0 4px 12px rgba(0, 0, 0, 0.2);
                max-width: 300px;
            ">
                <div class="kso-spinner" style="
                    width: 32px;
                    height: 32px;
                    border: 3px solid #f3f4f6;
                    border-top: 3px solid ${KSO.colors.primary};
                    border-radius: 50%;
                    animation: kso-spin 1s linear infinite;
                    margin: 0 auto 16px;
                "></div>
                <div style="color: #333; font-size: 14px; font-weight: 500;">${message}</div>
            </div>
        `;

        this.overlay.style.display = 'flex';
        this.addSpinnerAnimation();
    },

    /**
     * Show target-specific loader
     * @param {string} target - Target element selector
     * @param {string} message - Loading message
     * @param {string} loaderId - Loader ID
     */
    showTargetLoader: function(target, message, loaderId) {
        const targetEl = document.querySelector(target);
        if (!targetEl) return;

        const loader = document.createElement('div');
        loader.className = 'kso-target-loader';
        loader.dataset.loaderId = loaderId;
        loader.style.cssText = `
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(255, 255, 255, 0.9);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 1000;
            border-radius: inherit;
        `;

        loader.innerHTML = `
            <div style="text-align: center;">
                <div class="kso-spinner" style="
                    width: 24px;
                    height: 24px;
                    border: 2px solid #f3f4f6;
                    border-top: 2px solid ${KSO.colors.primary};
                    border-radius: 50%;
                    animation: kso-spin 1s linear infinite;
                    margin: 0 auto 12px;
                "></div>
                <div style="color: #666; font-size: 12px;">${message}</div>
            </div>
        `;

        targetEl.style.position = 'relative';
        targetEl.appendChild(loader);
        this.addSpinnerAnimation();
    },

    /**
     * Hide specific loader
     * @param {string} loaderId - Loader ID
     */
    hideSpecificLoader: function(loaderId) {
        const loader = document.querySelector(`[data-loader-id="${loaderId}"]`);
        if (loader) {
            loader.remove();
        }
    },

    /**
     * Hide all loaders
     */
    hideAllLoaders: function() {
        if (this.overlay) {
            this.overlay.style.display = 'none';
        }
        
        document.querySelectorAll('.kso-target-loader').forEach(loader => {
            loader.remove();
        });
    },

    /**
     * Add spinner animation CSS
     */
    addSpinnerAnimation: function() {
        if (document.querySelector('#kso-spinner-style')) return;

        const style = document.createElement('style');
        style.id = 'kso-spinner-style';
        style.textContent = `
            @keyframes kso-spin {
                0% { transform: rotate(0deg); }
                100% { transform: rotate(360deg); }
            }
        `;
        document.head.appendChild(style);
    }
};

// ========================================================================
// KSO FORM ENHANCEMENTS
// ========================================================================

KSO.forms = {
    /**
     * Initialize form enhancements
     */
    init: function() {
        this.setupFormValidation();
        this.setupFormSubmission();
        this.setupFileUploads();
    },

    /**
     * Setup form validation
     */
    setupFormValidation: function() {
        document.querySelectorAll('form[data-kso-validate]').forEach(form => {
            form.addEventListener('submit', (e) => {
                if (!this.validateForm(form)) {
                    e.preventDefault();
                }
            });
        });
    },

    /**
     * Setup form submission handling
     */
    setupFormSubmission: function() {
        document.querySelectorAll('form[data-kso-submit]').forEach(form => {
            form.addEventListener('submit', (e) => {
                e.preventDefault();
                this.handleFormSubmit(form);
            });
        });
    },

    /**
     * Setup file upload enhancements
     */
    setupFileUploads: function() {
        document.querySelectorAll('input[type="file"][data-kso-upload]').forEach(input => {
            this.enhanceFileInput(input);
        });
    },

    /**
     * Validate form
     * @param {HTMLFormElement} form - Form to validate
     * @returns {boolean} Is form valid
     */
    validateForm: function(form) {
        let isValid = true;
        const fields = form.querySelectorAll('[data-kso-required], [data-kso-validate]');
        
        fields.forEach(field => {
            const errors = this.validateField(field);
            if (errors.length > 0) {
                this.showFieldErrors(field, errors);
                isValid = false;
            } else {
                this.clearFieldErrors(field);
            }
        });

        return isValid;
    },

    /**
     * Validate individual field
     * @param {HTMLElement} field - Field to validate
     * @returns {Array} Array of error messages
     */
    validateField: function(field) {
        const errors = [];
        const value = field.value.trim();
        const rules = field.dataset.ksoValidate ? field.dataset.ksoValidate.split('|') : [];
        
        // Required validation
        if (field.hasAttribute('data-kso-required') && !value) {
            errors.push('This field is required');
        }

        // Skip other validations if field is empty and not required
        if (!value && !field.hasAttribute('data-kso-required')) {
            return errors;
        }

        // Rule-based validation
        rules.forEach(rule => {
            const [ruleName, ruleValue] = rule.split(':');
            
            switch (ruleName) {
                case 'email':
                    if (!KSO.utils.validateEmail(value)) {
                        errors.push('Please enter a valid email address');
                    }
                    break;
                case 'phone':
                    if (!KSO.utils.validatePhone(value)) {
                        errors.push('Please enter a valid phone number');
                    }
                    break;
                case 'min':
                    if (value.length < parseInt(ruleValue)) {
                        errors.push(`Minimum ${ruleValue} characters required`);
                    }
                    break;
                case 'max':
                    if (value.length > parseInt(ruleValue)) {
                        errors.push(`Maximum ${ruleValue} characters allowed`);
                    }
                    break;
                case 'number':
                    if (!/^\d+$/.test(value)) {
                        errors.push('Please enter a valid number');
                    }
                    break;
            }
        });

        return errors;
    },

    /**
     * Show field errors
     * @param {HTMLElement} field - Field element
     * @param {Array} errors - Array of error messages
     */
    showFieldErrors: function(field, errors) {
        this.clearFieldErrors(field);
        
        field.classList.add('is-invalid');
        
        const errorDiv = document.createElement('div');
        errorDiv.className = 'kso-field-errors invalid-feedback';
        errorDiv.innerHTML = errors.map(error => `<div>${error}</div>`).join('');
        
        field.parentNode.appendChild(errorDiv);
    },

    /**
     * Clear field errors
     * @param {HTMLElement} field - Field element
     */
    clearFieldErrors: function(field) {
        field.classList.remove('is-invalid');
        const errorDiv = field.parentNode.querySelector('.kso-field-errors');
        if (errorDiv) {
            errorDiv.remove();
        }
    },

    /**
     * Handle form submission
     * @param {HTMLFormElement} form - Form to submit
     */
    handleFormSubmit: function(form) {
        const url = form.getAttribute('action') || form.dataset.ksoSubmit;
        const method = form.getAttribute('method') || 'POST';
        const formData = new FormData(form);
        
        // Show loading
        const loaderId = KSO.loading.show('Submitting form...');
        
        // Disable form
        const submitBtn = form.querySelector('button[type="submit"]');
        if (submitBtn) {
            submitBtn.disabled = true;
        }
        
        fetch(url, {
            method: method,
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            KSO.loading.hide(loaderId);
            
            if (data.success) {
                KSO.notifications.success(data.message || 'Form submitted successfully');
                form.reset();
            } else {
                KSO.notifications.error(data.message || 'Form submission failed');
            }
        })
        .catch(error => {
            KSO.loading.hide(loaderId);
            KSO.notifications.error('An error occurred while submitting the form');
            console.error('Form submission error:', error);
        })
        .finally(() => {
            if (submitBtn) {
                submitBtn.disabled = false;
            }
        });
    },

    /**
     * Enhance file input
     * @param {HTMLInputElement} input - File input element
     */
    enhanceFileInput: function(input) {
        const wrapper = document.createElement('div');
        wrapper.className = 'kso-file-upload-wrapper';
        
        input.parentNode.insertBefore(wrapper, input);
        wrapper.appendChild(input);
        
        const preview = document.createElement('div');
        preview.className = 'kso-file-preview';
        wrapper.appendChild(preview);
        
        input.addEventListener('change', (e) => {
            this.handleFileChange(e, preview);
        });
    },

    /**
     * Handle file input change
     * @param {Event} event - Change event
     * @param {HTMLElement} preview - Preview element
     */
    handleFileChange: function(event, preview) {
        const files = event.target.files;
        preview.innerHTML = '';
        
        Array.from(files).forEach(file => {
            const item = document.createElement('div');
            item.className = 'kso-file-item';
            item.innerHTML = `
                <div class="file-info">
                    <span class="file-name">${file.name}</span>
                    <span class="file-size">${this.formatFileSize(file.size)}</span>
                </div>
            `;
            preview.appendChild(item);
        });
    },

    /**
     * Format file size
     * @param {number} bytes - File size in bytes
     * @returns {string} Formatted file size
     */
    formatFileSize: function(bytes) {
        if (bytes === 0) return '0 Bytes';
        const k = 1024;
        const sizes = ['Bytes', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
    }
};

// ========================================================================
// KSO ENHANCED UI COMPONENTS
// ========================================================================

KSO.ui = {
    /**
     * Initialize UI enhancements
     */
    init: function() {
        this.setupTooltips();
        this.setupModals();
        this.setupTabs();
        this.setupAccordions();
        this.setupDataTables();
    },

    /**
     * Setup tooltips
     */
    setupTooltips: function() {
        document.querySelectorAll('[data-kso-tooltip]').forEach(element => {
            element.addEventListener('mouseenter', (e) => this.showTooltip(e));
            element.addEventListener('mouseleave', (e) => this.hideTooltip(e));
        });
    },

    /**
     * Show tooltip
     * @param {Event} event - Mouse event
     */
    showTooltip: function(event) {
        const element = event.target;
        const text = element.dataset.ksoTooltip;
        
        const tooltip = document.createElement('div');
        tooltip.className = 'kso-tooltip';
        tooltip.textContent = text;
        tooltip.style.cssText = `
            position: absolute;
            background: #333;
            color: white;
            padding: 8px 12px;
            border-radius: 4px;
            font-size: 12px;
            z-index: 10000;
            white-space: nowrap;
            opacity: 0;
            transition: opacity 0.2s;
            pointer-events: none;
        `;
        
        document.body.appendChild(tooltip);
        
        const rect = element.getBoundingClientRect();
        tooltip.style.left = (rect.left + rect.width / 2 - tooltip.offsetWidth / 2) + 'px';
        tooltip.style.top = (rect.top - tooltip.offsetHeight - 8) + 'px';
        
        element._tooltip = tooltip;
        
        setTimeout(() => {
            tooltip.style.opacity = '1';
        }, 10);
    },

    /**
     * Hide tooltip
     * @param {Event} event - Mouse event
     */
    hideTooltip: function(event) {
        const element = event.target;
        if (element._tooltip) {
            element._tooltip.remove();
            delete element._tooltip;
        }
    },

    /**
     * Setup modals
     */
    setupModals: function() {
        document.querySelectorAll('[data-kso-modal]').forEach(trigger => {
            trigger.addEventListener('click', (e) => {
                e.preventDefault();
                const modalId = trigger.dataset.ksoModal;
                this.openModal(modalId);
            });
        });
    },

    /**
     * Open modal
     * @param {string} modalId - Modal ID
     */
    openModal: function(modalId) {
        const modal = document.getElementById(modalId);
        if (!modal) return;
        
        modal.style.display = 'block';
        modal.classList.add('kso-modal-active');
        
        // Close on backdrop click
        modal.addEventListener('click', (e) => {
            if (e.target === modal) {
                this.closeModal(modalId);
            }
        });
        
        // Close on escape key
        document.addEventListener('keydown', (e) => {
            if (e.key === 'Escape') {
                this.closeModal(modalId);
            }
        });
    },

    /**
     * Close modal
     * @param {string} modalId - Modal ID
     */
    closeModal: function(modalId) {
        const modal = document.getElementById(modalId);
        if (!modal) return;
        
        modal.style.display = 'none';
        modal.classList.remove('kso-modal-active');
    },

    /**
     * Setup tabs
     */
    setupTabs: function() {
        document.querySelectorAll('[data-kso-tab]').forEach(tab => {
            tab.addEventListener('click', (e) => {
                e.preventDefault();
                const tabId = tab.dataset.ksoTab;
                this.showTab(tabId, tab);
            });
        });
    },

    /**
     * Show tab
     * @param {string} tabId - Tab ID
     * @param {HTMLElement} tabTrigger - Tab trigger element
     */
    showTab: function(tabId, tabTrigger) {
        const tabContent = document.getElementById(tabId);
        if (!tabContent) return;
        
        // Hide all tabs in the same group
        const tabGroup = tabTrigger.closest('[data-kso-tab-group]');
        if (tabGroup) {
            tabGroup.querySelectorAll('[data-kso-tab]').forEach(t => t.classList.remove('active'));
            tabGroup.querySelectorAll('.kso-tab-content').forEach(c => c.style.display = 'none');
        }
        
        // Show selected tab
        tabTrigger.classList.add('active');
        tabContent.style.display = 'block';
    },

    /**
     * Setup accordions
     */
    setupAccordions: function() {
        document.querySelectorAll('[data-kso-accordion]').forEach(accordion => {
            accordion.addEventListener('click', (e) => {
                e.preventDefault();
                const contentId = accordion.dataset.ksoAccordion;
                this.toggleAccordion(contentId, accordion);
            });
        });
    },

    /**
     * Toggle accordion
     * @param {string} contentId - Content ID
     * @param {HTMLElement} trigger - Accordion trigger
     */
    toggleAccordion: function(contentId, trigger) {
        const content = document.getElementById(contentId);
        if (!content) return;
        
        const isOpen = content.style.display === 'block';
        
        if (isOpen) {
            content.style.display = 'none';
            trigger.classList.remove('active');
        } else {
            content.style.display = 'block';
            trigger.classList.add('active');
        }
    },

    /**
     * Setup enhanced data tables
     */
    setupDataTables: function() {
        document.querySelectorAll('[data-kso-table]').forEach(table => {
            this.enhanceTable(table);
        });
    },

    /**
     * Enhance table with sorting and filtering
     * @param {HTMLTableElement} table - Table element
     */
    enhanceTable: function(table) {
        // Add sorting to headers
        const headers = table.querySelectorAll('th[data-sortable]');
        headers.forEach(header => {
            header.style.cursor = 'pointer';
            header.addEventListener('click', () => {
                this.sortTable(table, header);
            });
        });
        
        // Add search functionality
        const searchInput = table.dataset.ksoSearch;
        if (searchInput) {
            const input = document.querySelector(searchInput);
            if (input) {
                input.addEventListener('input', KSO.utils.debounce(() => {
                    this.filterTable(table, input.value);
                }, 300));
            }
        }
    },

    /**
     * Sort table
     * @param {HTMLTableElement} table - Table element
     * @param {HTMLElement} header - Header element
     */
    sortTable: function(table, header) {
        const tbody = table.querySelector('tbody');
        const rows = Array.from(tbody.querySelectorAll('tr'));
        const columnIndex = Array.from(header.parentNode.children).indexOf(header);
        const isAscending = header.classList.contains('sort-asc');
        
        // Remove sort classes from all headers
        header.parentNode.querySelectorAll('th').forEach(th => {
            th.classList.remove('sort-asc', 'sort-desc');
        });
        
        // Add sort class to current header
        header.classList.add(isAscending ? 'sort-desc' : 'sort-asc');
        
        // Sort rows
        rows.sort((a, b) => {
            const aValue = a.children[columnIndex].textContent.trim();
            const bValue = b.children[columnIndex].textContent.trim();
            
            let comparison = 0;
            if (aValue > bValue) comparison = 1;
            else if (aValue < bValue) comparison = -1;
            
            return isAscending ? -comparison : comparison;
        });
        
        // Re-append sorted rows
        rows.forEach(row => tbody.appendChild(row));
    },

    /**
     * Filter table
     * @param {HTMLTableElement} table - Table element
     * @param {string} query - Search query
     */
    filterTable: function(table, query) {
        const tbody = table.querySelector('tbody');
        const rows = tbody.querySelectorAll('tr');
        const lowerQuery = query.toLowerCase();
        
        rows.forEach(row => {
            const text = row.textContent.toLowerCase();
            const matches = text.includes(lowerQuery);
            row.style.display = matches ? '' : 'none';
        });
    }
};

// ========================================================================
// KSO INITIALIZATION
// ========================================================================

KSO.init = function() {
    // Initialize all components
    KSO.notifications.init();
    KSO.forms.init();
    KSO.ui.init();
    
    // Log initialization
    if (KSO.config.debugMode) {
        console.log(`${KSO.config.shortName} JavaScript Framework v${KSO.config.version} initialized`);
    }
    
    // Trigger custom init event
    document.dispatchEvent(new CustomEvent('kso:init', {
        detail: { version: KSO.config.version }
    }));
};

// ========================================================================
// AUTO-INITIALIZATION
// ========================================================================

// Initialize when DOM is ready
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', KSO.init);
} else {
    KSO.init();
}

// Export for module systems
if (typeof module !== 'undefined' && module.exports) {
    module.exports = KSO;
}
