/**
 * Simplified Authentication Service for KSO Chandigarh Management System
 */

class AuthService {
    constructor() {
        this.baseUrl = window.CONFIG ? CONFIG.getApiUrl(CONFIG.ENDPOINTS.AUTH) : '/api/auth';
        this.currentAdmin = null;
        this.isAuthenticated = false;
        
        // Try to restore from localStorage first to avoid redirect loop
        const storedAuth = localStorage.getItem('kso_auth_state');
        if (storedAuth) {
            try {
                const authData = JSON.parse(storedAuth);
                const timestamp = authData.timestamp || 0;
                const now = Date.now();
                const fourHoursInMs = 4 * 60 * 60 * 1000;
                
                // Only use stored auth if it's less than 4 hours old
                if (now - timestamp < fourHoursInMs) {
                    this.currentAdmin = authData.admin;
                    this.isAuthenticated = true;
                }
            } catch (e) {
                // Ignore parse errors
            }
        }
        
        // Initialize auth state after a small delay
        setTimeout(() => this.init(), 100);
    }
    
    async init() {
        // Check if user is already authenticated
        await this.checkSession();
    }
    
    /**
     * Login with username/email and password
     */
    async login(username, password) {
        try {
            const response = await fetch(`${this.baseUrl}/login.php`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    username: username,
                    password: password
                })
            });
            
            const result = await response.json();
            
            if (result.success) {
                this.currentAdmin = result.data.admin;
                this.isAuthenticated = true;
                
                // Store auth state in localStorage for persistence
                localStorage.setItem('kso_auth_state', JSON.stringify({
                    admin: this.currentAdmin,
                    timestamp: Date.now()
                }));
                
                return result;
            } else {
                throw new Error(result.message || 'Login failed');
            }
        } catch (error) {
            console.error('Login error:', error);
            throw error;
        }
    }
    
    /**
     * Logout current user
     */
    async logout() {
        try {
            const response = await fetch(`${this.baseUrl}/logout.php`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                }
            });
            
            const result = await response.json();
            
            // Clear auth state regardless of server response
            this.clearAuthState();
            
            return { success: true };
        } catch (error) {
            console.error('Logout error:', error);
            // Clear auth state even if logout request fails
            this.clearAuthState();
            return { success: true };
        }
    }
    
    /**
     * Check current session status
     */
    async checkSession() {
        try {
            const response = await fetch(`${this.baseUrl}/session.php`, {
                method: 'GET',
                headers: {
                    'Content-Type': 'application/json',
                },
                credentials: 'include' // Include cookies in the request
            });
            
            const result = await response.json();
            
            if (result.success && result.data && result.data.logged_in) {
                this.currentAdmin = result.data.admin;
                this.isAuthenticated = true;
                
                // Update localStorage
                localStorage.setItem('kso_auth_state', JSON.stringify({
                    admin: this.currentAdmin,
                    timestamp: Date.now()
                }));
                
                return { success: true, admin: this.currentAdmin };
            } else {
                this.clearAuthState();
                return { success: false };
            }
        } catch (error) {
            console.error('Session check error:', error);
            this.clearAuthState();
            return { success: false };
        }
    }
    
    /**
     * Clear authentication state
     */
    clearAuthState() {
        this.currentAdmin = null;
        this.isAuthenticated = false;
        localStorage.removeItem('kso_auth_state');
        localStorage.removeItem('adminUsername');
        
        // Also clear session cookie by setting expired date
        document.cookie = "KSO_CHANDIGARH=; expires=Thu, 01 Jan 1970 00:00:00 UTC; path=/;";
    }
    
    /**
     * Get current admin information
     */
    getCurrentAdmin() {
        return this.currentAdmin;
    }
    
    /**
     * Check if user is logged in
     */
    isLoggedIn() {
        return this.isAuthenticated;
    }
    
    /**
     * Check if user has specific role
     */
    hasRole(role) {
        return this.currentAdmin && this.currentAdmin.role === role;
    }
    
    /**
     * Redirect to login page
     */
    redirectToLogin() {
        // Prevent redirect loops
        if (document.cookie.includes('KSO_REDIRECT_PREVENTION')) {
            console.log('Preventing redirect loop');
            return;
        }
        
        // Set a cookie to prevent redirect loops
        document.cookie = "KSO_REDIRECT_PREVENTION=true; path=/; max-age=5";
        
        // Always redirect to the main login page in the root directory
        const rootLoginPath = '../../login.html';
        
        // Force redirection to main login page
        window.location.href = rootLoginPath;
    }
    
    /**
     * Redirect to dashboard
     */
    redirectToDashboard() {
        const currentPath = window.location.pathname;
        if (currentPath.includes('/frontend/pages/')) {
            window.location.href = 'dashboard.html';
        } else {
            window.location.href = 'frontend/pages/dashboard.html';
        }
    }
}

// Create global auth service instance
window.authService = new AuthService();

// Export for use in other files
if (typeof module !== 'undefined' && module.exports) {
    module.exports = AuthService;
}
