/**
 * KSO Component Loader System
 * JavaScript-based include system for modular HTML components
 * Similar to AdminLTE's approach but for static HTML
 */

class ComponentLoader {
    constructor() {
        this.components = new Map();
        this.loadedComponents = new Set();
        this.basePath = this.getBasePath();
    }

    /**
     * Calculate the base path for includes based on current page depth
     */
    getBasePath() {
        const pathname = window.location.pathname;
        const pathSegments = pathname.split('/').filter(segment => segment !== '');
        
        // Remove the filename from the path
        const depth = pathSegments.length > 1 ? pathSegments.length - 1 : 0;
        
        // Calculate relative path to frontend root
        if (pathname.includes('/frontend/')) {
            return depth === 0 ? './' : '../'.repeat(depth);
        } else {
            return './frontend/';
        }
    }

    /**
     * Load a component from the includes directory
     */
    async loadComponent(componentName) {
        if (this.loadedComponents.has(componentName)) {
            return this.components.get(componentName);
        }

        try {
            const response = await fetch(`${this.basePath}include/${componentName}.html`);
            if (!response.ok) {
                throw new Error(`Failed to load component: ${componentName}`);
            }
            
            const html = await response.text();
            this.components.set(componentName, html);
            this.loadedComponents.add(componentName);
            return html;
        } catch (error) {
            console.error(`Error loading component ${componentName}:`, error);
            return `<!-- Error loading ${componentName} -->`;
        }
    }

    /**
     * Load all components and inject them into the page
     */
    async loadAllComponents() {
        const componentElements = document.querySelectorAll('[data-include]');
        
        for (const element of componentElements) {
            const componentName = element.getAttribute('data-include');
            const html = await this.loadComponent(componentName);
            element.innerHTML = html;
            element.removeAttribute('data-include');
        }

        // Fire custom event when all components are loaded
        document.dispatchEvent(new CustomEvent('componentsLoaded'));
    }

    /**
     * Initialize the component loader
     */
    init() {
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', () => this.loadAllComponents());
        } else {
            this.loadAllComponents();
        }
    }
}

// Global instance
window.ComponentLoader = new ComponentLoader();

// Auto-initialize
window.ComponentLoader.init();

// Utility function for backwards compatibility
function loadComponent(elementId, componentName) {
    window.ComponentLoader.loadComponent(componentName).then(html => {
        const element = document.getElementById(elementId);
        if (element) {
            element.innerHTML = html;
        }
    });
}
