/**
 * KSO Chandigarh Configuration
 * Centralized configuration for API endpoints and system settings
 */

// Detect the current environment and set appropriate base URL
const CONFIG = {
    // API Configuration - handles both local and production environments
    API_BASE_URL: (() => {
        // Get the base path from current location
        const path = window.location.pathname;
        const basePath = path.includes('/KSOCHD/') ? '/KSOCHD' : '';
        return window.location.origin + basePath + '/api';
    })(),
    
    // API Endpoints
    ENDPOINTS: {
        AUTH: '/auth',
        MEMBERS: '/members',
        EVENTS: '/events',
        FINANCE: '/finance',
        DOCUMENTS: '/documents',
        NOTIFICATIONS: '/notifications',
        REPORTS: '/reports',
        SETTINGS: '/settings'
    },
    
    // Specific API Actions
    API_ACTIONS: {
        // Auth actions (using direct PHP files)
        LOGIN: '/auth/login.php',
        LOGOUT: '/auth/logout.php',
        SESSION: '/auth/session.php',
        
        // Member actions
        MEMBER_LIST: '/members/index.php',
        MEMBER_DETAILS: '/members/member.php',
        MEMBER_APPROVE: '/members/approve.php',
        MEMBER_REJECT: '/members/reject.php',
        MEMBER_STATS: '/members/stats.php',
        MEMBER_ASSIGN: '/members/assign.php',
        
        // Event actions
        EVENT_LIST: '/events/index.php',
        EVENT_DETAILS: '/events/event.php',
        EVENT_REGISTRATIONS: '/events/registrations.php',
        EVENT_STATS: '/events/stats.php',
        
        // Finance actions
        FINANCE_LIST: '/finance/index.php',
        FINANCE_STATS: '/finance/stats.php'
    },
    
    // Application Settings
    APP: {
        NAME: 'KSO Chandigarh Management System',
        VERSION: '2.0.0',
        SESSION_TIMEOUT: 30, // minutes
        AUTO_LOGOUT_WARNING: 5, // minutes before timeout
        PAGE_SIZE: 20, // default pagination
        MAX_UPLOAD_SIZE: 5 * 1024 * 1024, // 5MB
        ALLOWED_IMAGE_TYPES: ['jpg', 'jpeg', 'png', 'gif', 'webp'],
        ALLOWED_DOCUMENT_TYPES: ['pdf', 'doc', 'docx', 'xls', 'xlsx']
    },
    
    // UI Configuration
    UI: {
        THEME: 'light',
        SIDEBAR_COLLAPSED: false,
        NOTIFICATIONS_POSITION: 'topright',
        ANIMATION_DURATION: 300,
        TOAST_DURATION: 3000
    },
    
    // Development/Debug Settings
    DEBUG: {
        ENABLED: window.location.hostname === 'localhost' || window.location.hostname === '127.0.0.1',
        LOG_LEVEL: 'info', // 'debug', 'info', 'warn', 'error'
        SHOW_SQL_QUERIES: false,
        PERFORMANCE_MONITORING: true
    }
};

// Helper functions
CONFIG.getApiUrl = function(endpoint) {
    return this.API_BASE_URL + endpoint;
};

CONFIG.getFullApiUrl = function(action) {
    return this.API_BASE_URL + this.API_ACTIONS[action];
};

CONFIG.isDebug = function() {
    return this.DEBUG.ENABLED;
};

CONFIG.log = function(message, level = 'info') {
    if (this.DEBUG.ENABLED) {
        console[level](`[KSO] ${message}`);
    }
};

// Export for use in other files
window.CONFIG = CONFIG;

// Initialize configuration
document.addEventListener('DOMContentLoaded', function() {
    CONFIG.log('KSO Chandigarh Management System Configuration Loaded', 'info');
    
    // Set up global error handling
    window.addEventListener('error', function(event) {
        CONFIG.log(`Global Error: ${event.error.message}`, 'error');
    });
    
    // Set up unhandled promise rejection handling
    window.addEventListener('unhandledrejection', function(event) {
        CONFIG.log(`Unhandled Promise Rejection: ${event.reason}`, 'error');
    });
});
