/**
 * KSO Chandigarh Dashboard JavaScript
 * Handles dashboard functionality and data loading
 */

// Redirect prevention flag - prevents redirect loops
let redirectAttempted = false;

// Initialize dashboard when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    // Prevent redirect loops
    if (redirectAttempted) {
        console.log("Preventing redirect loop");
        return;
    }
    
    // Add a small delay to ensure authService is fully initialized
    setTimeout(function() {
        // Check authentication first - if not authenticated, this will redirect to login
        try {
            if (typeof requireAuth === 'function') {
                redirectAttempted = true; // Set flag before attempting redirect
                if (!requireAuth()) return; // Stop if not authenticated
                redirectAttempted = false; // Reset flag if auth successful
            } else {
                console.error('requireAuth function not found - redirecting to login');
                window.location.href = '../../login.html';
                return;
            }
        } catch (e) {
            console.error('Authentication check failed:', e);
            window.location.href = '../../login.html';
            return;
        }
        
        // Initialize dashboard
        initializeDashboard();
    }, 200);
});

/**
 * Initialize dashboard components
 */
function initializeDashboard() {
    // Load statistics
    loadDashboardStats();
    
    // Load recent activity
    loadRecentActivity();
    
    // Update last update time
    updateLastUpdateTime();
    
    // Setup refresh functionality
    setupRefreshHandler();
    
    // Setup logout handler
    setupLogoutHandler();
    
    // Setup sidebar toggle
    setupSidebarToggle();
}

/**
 * Load dashboard statistics
 */
async function loadDashboardStats() {
    try {
        // Load actual stats from API
        const apiService = new ApiService();
        const response = await apiService.request('dashboard.php');
        
        if (response.success) {
            const stats = response.data;
            
            // Update stat displays with real data
            updateStatDisplay('totalMembers', stats.total_members || 0);
            updateStatDisplay('totalEvents', stats.total_events || 0);
            updateStatDisplay('pendingApprovals', stats.pending_approvals || 0);
            updateStatDisplay('monthlyRevenue', stats.monthly_revenue || '₹0');
        }
    } catch (error) {
        console.error('Error loading dashboard stats:', error);
        
        // Fallback to placeholder data
        const stats = {
            totalMembers: 150,
            totalEvents: 25,
            pendingApprovals: 8,
            monthlyRevenue: '₹25,000'
        };
        
        updateStatDisplay('totalMembers', stats.totalMembers);
        updateStatDisplay('totalEvents', stats.totalEvents);
        updateStatDisplay('pendingApprovals', stats.pendingApprovals);
        updateStatDisplay('monthlyRevenue', stats.monthlyRevenue);
    }
}

/**
 * Update individual stat display
 */
function updateStatDisplay(elementId, value) {
    const element = document.getElementById(elementId);
    if (element) {
        element.textContent = value;
    }
}

/**
 * Load recent activity
 */
function loadRecentActivity() {
    const activityContainer = document.getElementById('recentActivity');
    if (!activityContainer) return;
    
    // Simulate recent activity data
    const activities = [
        {
            time: '2 mins ago',
            icon: 'bi-person-plus',
            color: 'success',
            text: 'New member registration: John Doe'
        },
        {
            time: '15 mins ago',
            icon: 'bi-calendar-event',
            color: 'info',
            text: 'Event "Cultural Night" was approved'
        },
        {
            time: '1 hour ago',
            icon: 'bi-credit-card',
            color: 'warning',
            text: 'Payment received from membership fees'
        },
        {
            time: '2 hours ago',
            icon: 'bi-file-earmark-text',
            color: 'primary',
            text: 'Monthly report generated'
        }
    ];
    
    // Generate activity HTML
    let activityHTML = '';
    activities.forEach(activity => {
        activityHTML += `
            <div class="timeline-item">
                <div class="timeline-marker">
                    <i class="${activity.icon} text-${activity.color}"></i>
                </div>
                <div class="timeline-content">
                    <p class="timeline-text">${activity.text}</p>
                    <small class="text-muted">${activity.time}</small>
                </div>
            </div>
        `;
    });
    
    activityContainer.innerHTML = activityHTML;
}

/**
 * Update last update time
 */
function updateLastUpdateTime() {
    const lastUpdateElement = document.getElementById('lastUpdate');
    if (lastUpdateElement) {
        const now = new Date();
        const timeString = now.toLocaleTimeString();
        lastUpdateElement.textContent = timeString;
    }
}

/**
 * Setup refresh handler
 */
function setupRefreshHandler() {
    const refreshBtn = document.getElementById('refreshBtn');
    if (refreshBtn) {
        refreshBtn.addEventListener('click', function() {
            // Add loading state
            const icon = refreshBtn.querySelector('i');
            icon.classList.add('fa-spin');
            
            // Reload data
            setTimeout(() => {
                loadDashboardStats();
                loadRecentActivity();
                updateLastUpdateTime();
                
                // Remove loading state
                icon.classList.remove('fa-spin');
                
                // Show success message
                showNotification('Dashboard refreshed successfully!', 'success');
            }, 1000);
        });
    }
}

/**
 * Setup logout handler
 */
function setupLogoutHandler() {
    const logoutBtn = document.getElementById('logoutBtn');
    if (logoutBtn) {
        logoutBtn.addEventListener('click', async function(e) {
            e.preventDefault();
            
            try {
                // Use the auth service to handle logout
                if (window.authService) {
                    await window.authService.logout();
                    window.authService.redirectToLogin();
                } else {
                    // Fallback if auth service is not available
                    localStorage.removeItem('kso_auth');
                    sessionStorage.removeItem('kso_auth');
                    window.location.href = '../../login.html'; // Relative path to root login page
                }
            } catch (error) {
                console.error('Logout error:', error);
                // Force redirect even if there's an error
                window.location.href = '../../login.html';
            }
        });
    }
}

/**
 * Setup sidebar toggle
 */
function setupSidebarToggle() {
    const sidebarToggle = document.querySelector('[data-lte-toggle="sidebar"]');
    if (sidebarToggle) {
        sidebarToggle.addEventListener('click', function(e) {
            e.preventDefault();
            document.body.classList.toggle('sidebar-collapse');
        });
    }
}

/**
 * Show notification
 */
function showNotification(message, type = 'info') {
    // Create notification element
    const notification = document.createElement('div');
    notification.className = `alert alert-${type} alert-dismissible fade show position-fixed`;
    notification.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px;';
    notification.innerHTML = `
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    
    // Add to page
    document.body.appendChild(notification);
    
    // Auto-remove after 5 seconds
    setTimeout(() => {
        if (notification.parentNode) {
            notification.parentNode.removeChild(notification);
        }
    }, 5000);
}

/**
 * Check authentication status
 */
function checkAuth() {
    const auth = localStorage.getItem('kso_auth') || sessionStorage.getItem('kso_auth');
    if (!auth) {
        window.location.href = 'login.html';
        return false;
    }
    return true;
}

// Check authentication on page load
checkAuth();
