// Documents Management JavaScript
class DocumentsManager {
    constructor() {
        this.currentPage = 1;
        this.itemsPerPage = 10;
        this.documents = [];
        this.filteredDocuments = [];
        this.currentView = 'list';
        this.uploadedFiles = [];
    }

    // Initialize upload functionality
    initializeUpload() {
        const uploadArea = document.getElementById('uploadArea');
        const fileInput = document.getElementById('fileInput');
        
        // Drag and drop functionality
        uploadArea.addEventListener('dragover', (e) => {
            e.preventDefault();
            uploadArea.classList.add('dragover');
        });
        
        uploadArea.addEventListener('dragleave', (e) => {
            e.preventDefault();
            uploadArea.classList.remove('dragover');
        });
        
        uploadArea.addEventListener('drop', (e) => {
            e.preventDefault();
            uploadArea.classList.remove('dragover');
            this.handleFiles(e.dataTransfer.files);
        });
        
        uploadArea.addEventListener('click', () => {
            fileInput.click();
        });
        
        fileInput.addEventListener('change', (e) => {
            this.handleFiles(e.target.files);
        });
    }

    // Handle file upload
    async handleFiles(files) {
        const validTypes = [
            'application/pdf',
            'application/msword',
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'image/jpeg',
            'image/jpg',
            'image/png',
            'image/gif',
            'application/zip',
            'application/x-rar-compressed'
        ];
        
        const maxSize = 10 * 1024 * 1024; // 10MB
        const validFiles = [];
        
        for (let file of files) {
            if (!validTypes.includes(file.type)) {
                showToast('error', `File type not supported: ${file.name}`);
                continue;
            }
            
            if (file.size > maxSize) {
                showToast('error', `File too large: ${file.name} (max 10MB)`);
                continue;
            }
            
            validFiles.push(file);
        }
        
        if (validFiles.length > 0) {
            await this.uploadFiles(validFiles);
        }
    }

    // Upload files to server
    async uploadFiles(files) {
        const progressContainer = document.getElementById('uploadProgress');
        const progressBar = progressContainer.querySelector('.progress-bar');
        
        progressContainer.style.display = 'block';
        
        for (let i = 0; i < files.length; i++) {
            const file = files[i];
            const formData = new FormData();
            formData.append('file', file);
            formData.append('category', 'Other');
            formData.append('description', '');
            formData.append('access_level', 'public');
            
            try {
                const progress = ((i + 1) / files.length) * 100;
                progressBar.style.width = progress + '%';
                
                const response = await apiService.request('/api/documents/upload.php', {
                    method: 'POST',
                    body: formData
                });
                
                if (response.success) {
                    showToast('success', `Uploaded: ${file.name}`);
                } else {
                    showToast('error', `Failed to upload: ${file.name}`);
                }
            } catch (error) {
                console.error('Upload error:', error);
                showToast('error', `Error uploading: ${file.name}`);
            }
        }
        
        progressContainer.style.display = 'none';
        progressBar.style.width = '0%';
        
        // Refresh documents list
        await this.loadDocuments();
        await this.loadStats();
    }

    // Load documents from API
    async loadDocuments() {
        try {
            const response = await apiService.get('/api/documents/index.php');
            if (response.success) {
                this.documents = response.data;
                this.filteredDocuments = [...this.documents];
                this.renderDocuments();
            } else {
                showToast('error', 'Failed to load documents');
            }
        } catch (error) {
            console.error('Error loading documents:', error);
            showToast('error', 'Error loading documents');
        }
    }

    // Load statistics
    async loadStats() {
        try {
            const response = await apiService.get('/api/documents/stats.php');
            if (response.success) {
                const stats = response.data;
                document.getElementById('totalDocuments').textContent = stats.total_documents || 0;
                document.getElementById('officialDocuments').textContent = stats.official_documents || 0;
                document.getElementById('memberDocuments').textContent = stats.member_documents || 0;
                document.getElementById('totalSize').textContent = this.formatFileSize(stats.total_size || 0);
            }
        } catch (error) {
            console.error('Error loading stats:', error);
        }
    }

    // Filter documents
    filterDocuments() {
        const category = document.getElementById('categoryFilter').value;
        const type = document.getElementById('typeFilter').value;
        const search = document.getElementById('searchInput').value.toLowerCase();
        
        this.filteredDocuments = this.documents.filter(doc => {
            const matchCategory = !category || doc.category === category;
            const matchType = !type || this.getFileType(doc.file_name) === type;
            const matchSearch = !search || 
                doc.title.toLowerCase().includes(search) ||
                doc.description.toLowerCase().includes(search) ||
                doc.file_name.toLowerCase().includes(search);
            
            return matchCategory && matchType && matchSearch;
        });
        
        this.currentPage = 1;
        this.renderDocuments();
    }

    // Get file type from filename
    getFileType(filename) {
        const ext = filename.split('.').pop().toLowerCase();
        switch (ext) {
            case 'pdf':
                return 'pdf';
            case 'doc':
            case 'docx':
                return 'doc';
            case 'jpg':
            case 'jpeg':
            case 'png':
            case 'gif':
                return 'image';
            case 'zip':
            case 'rar':
                return 'archive';
            default:
                return 'other';
        }
    }

    // Get file icon
    getFileIcon(filename) {
        const type = this.getFileType(filename);
        switch (type) {
            case 'pdf':
                return '<i class="bi bi-file-pdf file-icon pdf-icon"></i>';
            case 'doc':
                return '<i class="bi bi-file-word file-icon doc-icon"></i>';
            case 'image':
                return '<i class="bi bi-file-image file-icon img-icon"></i>';
            case 'archive':
                return '<i class="bi bi-file-zip file-icon zip-icon"></i>';
            default:
                return '<i class="bi bi-file-earmark file-icon"></i>';
        }
    }

    // Render documents
    renderDocuments() {
        if (this.currentView === 'grid') {
            this.renderGridView();
        } else {
            this.renderListView();
        }
        this.renderPagination();
    }

    // Render grid view
    renderGridView() {
        const container = document.getElementById('documentGrid');
        const startIndex = (this.currentPage - 1) * this.itemsPerPage;
        const endIndex = startIndex + this.itemsPerPage;
        const pageDocuments = this.filteredDocuments.slice(startIndex, endIndex);
        
        container.innerHTML = '';
        
        pageDocuments.forEach(doc => {
            const card = document.createElement('div');
            card.className = 'col-md-3 col-sm-6 mb-3';
            card.innerHTML = `
                <div class="card document-card h-100">
                    <div class="card-body text-center">
                        ${this.getFileIcon(doc.file_name)}
                        <h6 class="card-title mt-2">${doc.title}</h6>
                        <p class="card-text text-muted small">${doc.description || 'No description'}</p>
                        <span class="badge badge-primary category-badge">${doc.category}</span>
                        <div class="mt-3">
                            <button class="btn btn-sm btn-outline-primary" onclick="documentsManager.downloadDocument(${doc.id})">
                                <i class="bi bi-download"></i>
                            </button>
                            <button class="btn btn-sm btn-outline-info" onclick="documentsManager.viewDocument(${doc.id})">
                                <i class="bi bi-eye"></i>
                            </button>
                            <button class="btn btn-sm btn-outline-warning" onclick="documentsManager.editDocument(${doc.id})">
                                <i class="bi bi-pencil"></i>
                            </button>
                        </div>
                    </div>
                    <div class="card-footer small text-muted">
                        <div>Size: ${this.formatFileSize(doc.file_size)}</div>
                        <div>By: ${doc.uploaded_by_name}</div>
                        <div>${this.formatDate(doc.created_at)}</div>
                    </div>
                </div>
            `;
            container.appendChild(card);
        });
    }

    // Render list view
    renderListView() {
        const tbody = document.getElementById('documentsTableBody');
        const startIndex = (this.currentPage - 1) * this.itemsPerPage;
        const endIndex = startIndex + this.itemsPerPage;
        const pageDocuments = this.filteredDocuments.slice(startIndex, endIndex);
        
        tbody.innerHTML = '';
        
        pageDocuments.forEach(doc => {
            const row = document.createElement('tr');
            row.innerHTML = `
                <td>
                    ${this.isImage(doc.file_name) ? 
                        `<img src="${doc.file_path}" class="document-preview" alt="Preview">` :
                        this.getFileIcon(doc.file_name)
                    }
                </td>
                <td>
                    <strong>${doc.title}</strong><br>
                    <small class="text-muted">${doc.file_name}</small>
                </td>
                <td><span class="badge badge-primary">${doc.category}</span></td>
                <td>${this.getFileType(doc.file_name).toUpperCase()}</td>
                <td>${this.formatFileSize(doc.file_size)}</td>
                <td>${doc.uploaded_by_name}</td>
                <td>${this.formatDate(doc.created_at)}</td>
                <td>
                    <div class="btn-group" role="group">
                        <button class="btn btn-sm btn-outline-primary" onclick="documentsManager.downloadDocument(${doc.id})" title="Download">
                            <i class="bi bi-download"></i>
                        </button>
                        <button class="btn btn-sm btn-outline-info" onclick="documentsManager.viewDocument(${doc.id})" title="View">
                            <i class="bi bi-eye"></i>
                        </button>
                        <button class="btn btn-sm btn-outline-warning" onclick="documentsManager.editDocument(${doc.id})" title="Edit">
                            <i class="bi bi-pencil"></i>
                        </button>
                        <button class="btn btn-sm btn-outline-danger" onclick="documentsManager.deleteDocumentConfirm(${doc.id})" title="Delete">
                            <i class="bi bi-trash"></i>
                        </button>
                    </div>
                </td>
            `;
            tbody.appendChild(row);
        });
    }

    // Check if file is an image
    isImage(filename) {
        const imageTypes = ['jpg', 'jpeg', 'png', 'gif'];
        const ext = filename.split('.').pop().toLowerCase();
        return imageTypes.includes(ext);
    }

    // Render pagination
    renderPagination() {
        const totalPages = Math.ceil(this.filteredDocuments.length / this.itemsPerPage);
        const pagination = document.getElementById('pagination');
        
        // Update showing info
        const startIndex = (this.currentPage - 1) * this.itemsPerPage + 1;
        const endIndex = Math.min(this.currentPage * this.itemsPerPage, this.filteredDocuments.length);
        
        document.getElementById('showingStart').textContent = this.filteredDocuments.length > 0 ? startIndex : 0;
        document.getElementById('showingEnd').textContent = endIndex;
        document.getElementById('totalRecords').textContent = this.filteredDocuments.length;
        
        pagination.innerHTML = '';
        
        if (totalPages <= 1) return;
        
        // Previous button
        const prevLi = document.createElement('li');
        prevLi.className = `page-item ${this.currentPage === 1 ? 'disabled' : ''}`;
        prevLi.innerHTML = `<a class="page-link" href="#" onclick="documentsManager.changePage(${this.currentPage - 1})">Previous</a>`;
        pagination.appendChild(prevLi);
        
        // Page numbers
        const startPage = Math.max(1, this.currentPage - 2);
        const endPage = Math.min(totalPages, this.currentPage + 2);
        
        for (let i = startPage; i <= endPage; i++) {
            const pageLi = document.createElement('li');
            pageLi.className = `page-item ${i === this.currentPage ? 'active' : ''}`;
            pageLi.innerHTML = `<a class="page-link" href="#" onclick="documentsManager.changePage(${i})">${i}</a>`;
            pagination.appendChild(pageLi);
        }
        
        // Next button
        const nextLi = document.createElement('li');
        nextLi.className = `page-item ${this.currentPage === totalPages ? 'disabled' : ''}`;
        nextLi.innerHTML = `<a class="page-link" href="#" onclick="documentsManager.changePage(${this.currentPage + 1})">Next</a>`;
        pagination.appendChild(nextLi);
    }

    // Change page
    changePage(page) {
        if (page < 1 || page > Math.ceil(this.filteredDocuments.length / this.itemsPerPage)) return;
        this.currentPage = page;
        this.renderDocuments();
    }

    // View document
    async viewDocument(id) {
        const doc = this.documents.find(d => d.id === id);
        if (!doc) return;
        
        // Open document in new tab
        window.open(doc.file_path, '_blank');
    }

    // Download document
    async downloadDocument(id) {
        const doc = this.documents.find(d => d.id === id);
        if (!doc) return;
        
        // Create download link
        const link = document.createElement('a');
        link.href = doc.file_path;
        link.download = doc.file_name;
        link.click();
    }

    // Edit document
    async editDocument(id) {
        const doc = this.documents.find(d => d.id === id);
        if (!doc) return;
        
        // Populate form
        document.getElementById('documentId').value = doc.id;
        document.getElementById('documentTitle').value = doc.title;
        document.getElementById('documentCategory').value = doc.category;
        document.getElementById('documentDescription').value = doc.description || '';
        document.getElementById('documentTags').value = doc.tags || '';
        document.getElementById('documentAccess').value = doc.access_level;
        
        // Show modal
        $('#documentModal').modal('show');
    }

    // Save document
    async saveDocument() {
        const id = document.getElementById('documentId').value;
        const data = {
            title: document.getElementById('documentTitle').value,
            category: document.getElementById('documentCategory').value,
            description: document.getElementById('documentDescription').value,
            tags: document.getElementById('documentTags').value,
            access_level: document.getElementById('documentAccess').value
        };
        
        try {
            const response = await apiService.put(`/api/documents/document.php?id=${id}`, data);
            if (response.success) {
                showToast('success', 'Document updated successfully');
                $('#documentModal').modal('hide');
                await this.loadDocuments();
            } else {
                showToast('error', response.message || 'Failed to update document');
            }
        } catch (error) {
            console.error('Error updating document:', error);
            showToast('error', 'Error updating document');
        }
    }

    // Delete document confirmation
    deleteDocumentConfirm(id) {
        if (confirm('Are you sure you want to delete this document?')) {
            this.deleteDocument(id);
        }
    }

    // Delete document
    async deleteDocument(id = null) {
        if (!id) {
            id = document.getElementById('documentId').value;
        }
        
        try {
            const response = await apiService.delete(`/api/documents/document.php?id=${id}`);
            if (response.success) {
                showToast('success', 'Document deleted successfully');
                $('#documentModal').modal('hide');
                await this.loadDocuments();
                await this.loadStats();
            } else {
                showToast('error', response.message || 'Failed to delete document');
            }
        } catch (error) {
            console.error('Error deleting document:', error);
            showToast('error', 'Error deleting document');
        }
    }

    // Utility: Format file size
    formatFileSize(bytes) {
        if (bytes === 0) return '0 Bytes';
        const k = 1024;
        const sizes = ['Bytes', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
    }

    // Utility: Format date
    formatDate(dateString) {
        const date = new Date(dateString);
        return date.toLocaleDateString() + ' ' + date.toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'});
    }
}

// Global functions
function loadDocuments() {
    documentsManager.loadDocuments();
}

function loadStats() {
    documentsManager.loadStats();
}

function filterDocuments() {
    documentsManager.filterDocuments();
}

function initializeUpload() {
    documentsManager.initializeUpload();
}

function saveDocument() {
    documentsManager.saveDocument();
}

function deleteDocument() {
    documentsManager.deleteDocument();
}

// Initialize
const documentsManager = new DocumentsManager();

// Search on enter key
document.addEventListener('DOMContentLoaded', function() {
    const searchInput = document.getElementById('searchInput');
    if (searchInput) {
        searchInput.addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                filterDocuments();
            }
        });
    }
});
