/**
 * Enhanced Photo Upload System
 * Handles chunked uploads, real-time progress, drag & drop, and optimization
 */

class EnhancedPhotoUpload {
    constructor(options = {}) {
        this.options = {
            dropZone: options.dropZone || '#photo-drop-zone',
            fileInput: options.fileInput || '#photo-file-input',
            progressContainer: options.progressContainer || '#upload-progress',
            previewContainer: options.previewContainer || '#photo-preview',
            chunkSize: options.chunkSize || 5 * 1024 * 1024, // 5MB chunks
            maxFileSize: options.maxFileSize || 50 * 1024 * 1024, // 50MB max
            allowedTypes: options.allowedTypes || ['image/jpeg', 'image/png', 'image/gif', 'image/webp'],
            apiBaseUrl: options.apiBaseUrl || '/api/photos/enhanced.php',
            onUploadStart: options.onUploadStart || (() => {}),
            onUploadProgress: options.onUploadProgress || (() => {}),
            onUploadComplete: options.onUploadComplete || (() => {}),
            onUploadError: options.onUploadError || (() => {}),
            onPreviewGenerated: options.onPreviewGenerated || (() => {}),
            ...options
        };
        
        this.currentUploads = new Map();
        this.init();
    }
    
    init() {
        this.setupDropZone();
        this.setupFileInput();
        this.setupEventListeners();
    }
    
    /**
     * Setup drag and drop functionality
     */
    setupDropZone() {
        const dropZone = document.querySelector(this.options.dropZone);
        if (!dropZone) return;
        
        // Prevent default drag behaviors
        ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
            dropZone.addEventListener(eventName, this.preventDefaults, false);
            document.body.addEventListener(eventName, this.preventDefaults, false);
        });
        
        // Highlight drop zone when item is dragged over it
        ['dragenter', 'dragover'].forEach(eventName => {
            dropZone.addEventListener(eventName, () => this.highlight(dropZone), false);
        });
        
        ['dragleave', 'drop'].forEach(eventName => {
            dropZone.addEventListener(eventName, () => this.unhighlight(dropZone), false);
        });
        
        // Handle dropped files
        dropZone.addEventListener('drop', (e) => this.handleDrop(e), false);
        
        // Click to select files
        dropZone.addEventListener('click', () => {
            document.querySelector(this.options.fileInput)?.click();
        });
    }
    
    /**
     * Setup file input
     */
    setupFileInput() {
        const fileInput = document.querySelector(this.options.fileInput);
        if (!fileInput) return;
        
        fileInput.addEventListener('change', (e) => {
            this.handleFiles(e.target.files);
        });
    }
    
    /**
     * Setup additional event listeners
     */
    setupEventListeners() {
        // Window unload warning for active uploads
        window.addEventListener('beforeunload', (e) => {
            if (this.currentUploads.size > 0) {
                e.preventDefault();
                e.returnValue = 'You have uploads in progress. Are you sure you want to leave?';
                return e.returnValue;
            }
        });
    }
    
    preventDefaults(e) {
        e.preventDefault();
        e.stopPropagation();
    }
    
    highlight(element) {
        element.classList.add('drag-over');
    }
    
    unhighlight(element) {
        element.classList.remove('drag-over');
    }
    
    handleDrop(e) {
        const dt = e.dataTransfer;
        const files = dt.files;
        this.handleFiles(files);
    }
    
    /**
     * Handle selected/dropped files
     */
    handleFiles(files) {
        Array.from(files).forEach(file => {
            if (this.validateFile(file)) {
                this.uploadFile(file);
            }
        });
    }
    
    /**
     * Validate file before upload
     */
    validateFile(file) {
        // Check file type
        if (!this.options.allowedTypes.includes(file.type)) {
            this.showError(`Invalid file type: ${file.type}. Allowed types: ${this.options.allowedTypes.join(', ')}`);
            return false;
        }
        
        // Check file size
        if (file.size > this.options.maxFileSize) {
            this.showError(`File too large: ${this.formatFileSize(file.size)}. Maximum allowed: ${this.formatFileSize(this.options.maxFileSize)}`);
            return false;
        }
        
        if (file.size < 1024) {
            this.showError('File too small (minimum 1KB)');
            return false;
        }
        
        return true;
    }
    
    /**
     * Upload file (chunked or traditional)
     */
    async uploadFile(file) {
        const fileId = this.generateFileId();
        const shouldUseChunked = file.size > this.options.chunkSize;
        
        // Generate preview immediately
        this.generatePreview(file, fileId);
        
        // Create upload session
        const uploadSession = {
            fileId,
            file,
            progress: 0,
            status: 'uploading',
            startTime: Date.now(),
            chunks: shouldUseChunked ? Math.ceil(file.size / this.options.chunkSize) : 1,
            uploadedChunks: 0
        };
        
        this.currentUploads.set(fileId, uploadSession);
        this.options.onUploadStart(uploadSession);
        this.createProgressIndicator(uploadSession);
        
        try {
            let result;
            if (shouldUseChunked) {
                result = await this.uploadChunked(uploadSession);
            } else {
                result = await this.uploadTraditional(uploadSession);
            }
            
            uploadSession.status = 'completed';
            uploadSession.result = result;
            this.options.onUploadComplete(uploadSession);
            this.updateProgressIndicator(uploadSession);
            
        } catch (error) {
            uploadSession.status = 'error';
            uploadSession.error = error.message;
            this.options.onUploadError(error, uploadSession);
            this.updateProgressIndicator(uploadSession);
        } finally {
            // Clean up after delay
            setTimeout(() => {
                this.currentUploads.delete(fileId);
            }, 5000);
        }
    }
    
    /**
     * Upload file in chunks
     */
    async uploadChunked(uploadSession) {
        const { file, fileId, chunks } = uploadSession;
        const chunkSize = this.options.chunkSize;
        
        for (let chunkIndex = 0; chunkIndex < chunks; chunkIndex++) {
            const start = chunkIndex * chunkSize;
            const end = Math.min(start + chunkSize, file.size);
            const chunk = file.slice(start, end);
            
            // Calculate chunk hash for integrity check
            const chunkHash = await this.calculateHash(chunk);
            
            // Upload chunk
            await this.uploadChunk(uploadSession, chunkIndex, chunk, chunkHash);
            
            uploadSession.uploadedChunks = chunkIndex + 1;
            uploadSession.progress = (uploadSession.uploadedChunks / chunks) * 100;
            
            this.options.onUploadProgress(uploadSession);
            this.updateProgressIndicator(uploadSession);
        }
        
        // Complete upload
        return await this.completeUpload(uploadSession);
    }
    
    /**
     * Upload single chunk
     */
    async uploadChunk(uploadSession, chunkIndex, chunk, chunkHash) {
        const { fileId, file, chunks } = uploadSession;
        
        const response = await fetch(`${this.options.apiBaseUrl}?path=upload-chunk`, {
            method: 'POST',
            headers: {
                'X-Chunk-Index': chunkIndex.toString(),
                'X-Total-Chunks': chunks.toString(),
                'X-File-Id': fileId,
                'X-Original-Name': file.name,
                'X-Total-Size': file.size.toString(),
                'X-Chunk-Hash': chunkHash,
                'X-CSRF-Token': this.getCSRFToken()
            },
            body: chunk
        });
        
        if (!response.ok) {
            throw new Error(`Chunk upload failed: ${response.statusText}`);
        }
        
        const result = await response.json();
        if (!result.success) {
            throw new Error(result.error || 'Chunk upload failed');
        }
        
        return result;
    }
    
    /**
     * Complete chunked upload
     */
    async completeUpload(uploadSession) {
        const { fileId } = uploadSession;
        const memberId = this.options.memberId || 0;
        
        const response = await fetch(`${this.options.apiBaseUrl}?path=upload-complete`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-Token': this.getCSRFToken()
            },
            body: JSON.stringify({
                file_id: fileId,
                member_id: memberId
            })
        });
        
        if (!response.ok) {
            throw new Error(`Upload completion failed: ${response.statusText}`);
        }
        
        const result = await response.json();
        if (!result.success) {
            throw new Error(result.error || 'Upload completion failed');
        }
        
        return result;
    }
    
    /**
     * Traditional upload for small files
     */
    async uploadTraditional(uploadSession) {
        const { file } = uploadSession;
        const memberId = this.options.memberId || 0;
        
        const formData = new FormData();
        formData.append('photo', file);
        formData.append('member_id', memberId);
        formData.append('csrf_token', this.getCSRFToken());
        
        const xhr = new XMLHttpRequest();
        
        // Track upload progress
        xhr.upload.addEventListener('progress', (e) => {
            if (e.lengthComputable) {
                uploadSession.progress = (e.loaded / e.total) * 100;
                this.options.onUploadProgress(uploadSession);
                this.updateProgressIndicator(uploadSession);
            }
        });
        
        return new Promise((resolve, reject) => {
            xhr.onload = () => {
                if (xhr.status === 200) {
                    try {
                        const result = JSON.parse(xhr.responseText);
                        if (result.success) {
                            resolve(result);
                        } else {
                            reject(new Error(result.error || 'Upload failed'));
                        }
                    } catch (e) {
                        reject(new Error('Invalid response format'));
                    }
                } else {
                    reject(new Error(`Upload failed: ${xhr.statusText}`));
                }
            };
            
            xhr.onerror = () => reject(new Error('Network error during upload'));
            
            xhr.open('POST', `${this.options.apiBaseUrl}?path=upload-traditional`);
            xhr.send(formData);
        });
    }
    
    /**
     * Generate file preview
     */
    generatePreview(file, fileId) {
        const reader = new FileReader();
        
        reader.onload = (e) => {
            const preview = {
                fileId,
                fileName: file.name,
                fileSize: file.size,
                dataUrl: e.target.result,
                file
            };
            
            this.options.onPreviewGenerated(preview);
            this.showPreview(preview);
        };
        
        reader.readAsDataURL(file);
    }
    
    /**
     * Show preview in container
     */
    showPreview(preview) {
        const container = document.querySelector(this.options.previewContainer);
        if (!container) return;
        
        const previewElement = document.createElement('div');
        previewElement.className = 'photo-preview-item';
        previewElement.setAttribute('data-file-id', preview.fileId);
        
        previewElement.innerHTML = `
            <div class="preview-image">
                <img src="${preview.dataUrl}" alt="${preview.fileName}" loading="lazy">
                <div class="preview-overlay">
                    <div class="preview-info">
                        <span class="file-name">${preview.fileName}</span>
                        <span class="file-size">${this.formatFileSize(preview.fileSize)}</span>
                    </div>
                </div>
            </div>
            <div class="preview-actions">
                <button type="button" class="btn-cancel" onclick="photoUpload.cancelUpload('${preview.fileId}')">
                    <i class="fas fa-times"></i> Cancel
                </button>
            </div>
        `;
        
        container.appendChild(previewElement);
    }
    
    /**
     * Create progress indicator
     */
    createProgressIndicator(uploadSession) {
        const container = document.querySelector(this.options.progressContainer);
        if (!container) return;
        
        const progressElement = document.createElement('div');
        progressElement.className = 'upload-progress-item';
        progressElement.setAttribute('data-file-id', uploadSession.fileId);
        
        progressElement.innerHTML = `
            <div class="progress-info">
                <span class="file-name">${uploadSession.file.name}</span>
                <span class="progress-text">0%</span>
            </div>
            <div class="progress-bar">
                <div class="progress-fill" style="width: 0%"></div>
            </div>
            <div class="progress-stats">
                <span class="upload-speed">-</span>
                <span class="time-remaining">-</span>
            </div>
        `;
        
        container.appendChild(progressElement);
    }
    
    /**
     * Update progress indicator
     */
    updateProgressIndicator(uploadSession) {
        const element = document.querySelector(`[data-file-id="${uploadSession.fileId}"]`);
        if (!element) return;
        
        const progressFill = element.querySelector('.progress-fill');
        const progressText = element.querySelector('.progress-text');
        const uploadSpeed = element.querySelector('.upload-speed');
        const timeRemaining = element.querySelector('.time-remaining');
        
        if (progressFill) progressFill.style.width = `${uploadSession.progress}%`;
        if (progressText) progressText.textContent = `${Math.round(uploadSession.progress)}%`;
        
        // Calculate upload speed and time remaining
        const elapsed = Date.now() - uploadSession.startTime;
        const uploaded = (uploadSession.progress / 100) * uploadSession.file.size;
        const speed = uploaded / (elapsed / 1000); // bytes per second
        const remaining = (uploadSession.file.size - uploaded) / speed; // seconds
        
        if (uploadSpeed && speed > 0) {
            uploadSpeed.textContent = this.formatFileSize(speed) + '/s';
        }
        
        if (timeRemaining && remaining > 0 && uploadSession.progress < 100) {
            timeRemaining.textContent = this.formatTime(remaining);
        }
        
        // Update status classes
        element.className = `upload-progress-item status-${uploadSession.status}`;
        
        if (uploadSession.status === 'completed') {
            setTimeout(() => {
                element.classList.add('fade-out');
                setTimeout(() => element.remove(), 500);
            }, 2000);
        }
    }
    
    /**
     * Cancel upload
     */
    async cancelUpload(fileId) {
        const uploadSession = this.currentUploads.get(fileId);
        if (!uploadSession) return;
        
        uploadSession.status = 'cancelled';
        
        try {
            await fetch(`${this.options.apiBaseUrl}?path=cancel-upload&session_id=${fileId}`, {
                method: 'DELETE',
                headers: {
                    'X-CSRF-Token': this.getCSRFToken()
                }
            });
        } catch (error) {
            console.error('Error cancelling upload:', error);
        }
        
        // Remove from UI
        const previewElement = document.querySelector(`[data-file-id="${fileId}"]`);
        if (previewElement) previewElement.remove();
        
        this.currentUploads.delete(fileId);
    }
    
    /**
     * Utility functions
     */
    generateFileId() {
        return 'upload_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
    }
    
    async calculateHash(data) {
        const buffer = await data.arrayBuffer();
        const hashBuffer = await crypto.subtle.digest('SHA-256', buffer);
        const hashArray = Array.from(new Uint8Array(hashBuffer));
        return hashArray.map(b => b.toString(16).padStart(2, '0')).join('');
    }
    
    formatFileSize(bytes) {
        if (bytes === 0) return '0 B';
        const k = 1024;
        const sizes = ['B', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
    }
    
    formatTime(seconds) {
        if (seconds < 60) return `${Math.round(seconds)}s`;
        if (seconds < 3600) return `${Math.round(seconds / 60)}m ${Math.round(seconds % 60)}s`;
        return `${Math.round(seconds / 3600)}h ${Math.round((seconds % 3600) / 60)}m`;
    }
    
    getCSRFToken() {
        return document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '';
    }
    
    showError(message) {
        if (this.options.onUploadError) {
            this.options.onUploadError(new Error(message));
        } else {
            console.error('Photo Upload Error:', message);
            alert(message);
        }
    }
}

/**
 * Real-time Member Status Tracker
 */
class MemberStatusTracker {
    constructor(options = {}) {
        this.options = {
            apiBaseUrl: options.apiBaseUrl || '/api/photos/enhanced.php',
            updateInterval: options.updateInterval || 30000, // 30 seconds
            onStatusUpdate: options.onStatusUpdate || (() => {}),
            onOnlineCountUpdate: options.onOnlineCountUpdate || (() => {}),
            ...options
        };
        
        this.isTracking = false;
        this.intervalId = null;
        this.lastOnlineCount = 0;
    }
    
    start() {
        if (this.isTracking) return;
        
        this.isTracking = true;
        this.updateStatus(); // Initial update
        
        this.intervalId = setInterval(() => {
            this.updateStatus();
        }, this.options.updateInterval);
        
        // Update activity on user interaction
        this.trackUserActivity();
    }
    
    stop() {
        this.isTracking = false;
        if (this.intervalId) {
            clearInterval(this.intervalId);
            this.intervalId = null;
        }
    }
    
    async updateStatus() {
        try {
            // Update current member's activity
            if (this.options.memberId) {
                await this.updateMemberActivity();
            }
            
            // Get online members
            const response = await fetch(`${this.options.apiBaseUrl}?path=online-members&limit=100`);
            if (response.ok) {
                const data = await response.json();
                if (data.success) {
                    this.options.onStatusUpdate(data.members);
                    
                    if (data.online_count !== this.lastOnlineCount) {
                        this.lastOnlineCount = data.online_count;
                        this.options.onOnlineCountUpdate(data.online_count, data.total_active);
                    }
                    
                    this.updateMemberStatusDisplay(data.members);
                }
            }
        } catch (error) {
            console.error('Error updating member status:', error);
        }
    }
    
    async updateMemberActivity() {
        if (!this.options.memberId) return;
        
        try {
            await fetch(`${this.options.apiBaseUrl}?path=update-activity`, {
                method: 'PUT',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-Token': this.getCSRFToken()
                },
                body: JSON.stringify({
                    member_id: this.options.memberId,
                    activity_type: 'page_view',
                    details: {
                        page: window.location.pathname,
                        timestamp: Date.now()
                    }
                })
            });
        } catch (error) {
            console.error('Error updating member activity:', error);
        }
    }
    
    trackUserActivity() {
        let activityTimeout;
        
        const activity = () => {
            clearTimeout(activityTimeout);
            activityTimeout = setTimeout(() => {
                this.updateMemberActivity();
            }, 1000); // Debounce to 1 second
        };
        
        ['mousedown', 'mousemove', 'keypress', 'scroll', 'touchstart'].forEach(event => {
            document.addEventListener(event, activity, true);
        });
    }
    
    updateMemberStatusDisplay(members) {
        // Update member status indicators in the UI
        members.forEach(member => {
            const statusElements = document.querySelectorAll(`[data-member-id="${member.id}"] .member-status`);
            statusElements.forEach(element => {
                element.className = `member-status status-${member.status_label}`;
                element.setAttribute('title', this.getStatusTitle(member));
                
                // Update status dot color
                const dot = element.querySelector('.status-dot');
                if (dot) {
                    dot.className = `status-dot ${member.status_label}`;
                }
                
                // Update last seen text
                const lastSeen = element.querySelector('.last-seen');
                if (lastSeen) {
                    lastSeen.textContent = this.formatLastSeen(member);
                }
            });
        });
        
        // Update online counter
        const onlineCounter = document.querySelector('.online-members-count');
        if (onlineCounter) {
            onlineCounter.textContent = this.lastOnlineCount;
        }
    }
    
    getStatusTitle(member) {
        switch (member.status_label) {
            case 'online':
                return 'Online now';
            case 'recently_active':
                return `Active ${member.minutes_since_activity} minutes ago`;
            case 'away':
                return `Away - last seen ${member.minutes_since_activity} minutes ago`;
            default:
                return 'Offline';
        }
    }
    
    formatLastSeen(member) {
        if (member.status_label === 'online') {
            return 'Online now';
        }
        
        const minutes = member.minutes_since_activity;
        if (minutes < 60) {
            return `${minutes}m ago`;
        } else if (minutes < 1440) {
            return `${Math.floor(minutes / 60)}h ago`;
        } else {
            return `${Math.floor(minutes / 1440)}d ago`;
        }
    }
    
    getCSRFToken() {
        return document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '';
    }
}

/**
 * Lazy Loading for Images
 */
class LazyImageLoader {
    constructor(options = {}) {
        this.options = {
            selector: options.selector || '.lazy-image',
            rootMargin: options.rootMargin || '50px',
            threshold: options.threshold || 0.1,
            ...options
        };
        
        this.observer = null;
        this.init();
    }
    
    init() {
        if ('IntersectionObserver' in window) {
            this.observer = new IntersectionObserver(
                this.handleIntersection.bind(this),
                {
                    rootMargin: this.options.rootMargin,
                    threshold: this.options.threshold
                }
            );
            
            this.observeImages();
        } else {
            // Fallback for older browsers
            this.loadAllImages();
        }
    }
    
    observeImages() {
        const images = document.querySelectorAll(this.options.selector);
        images.forEach(img => this.observer.observe(img));
    }
    
    handleIntersection(entries) {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                this.loadImage(entry.target);
                this.observer.unobserve(entry.target);
            }
        });
    }
    
    loadImage(img) {
        const src = img.dataset.src;
        const srcset = img.dataset.srcset;
        
        if (src) {
            img.src = src;
        }
        
        if (srcset) {
            img.srcset = srcset;
        }
        
        img.classList.remove('lazy-image');
        img.classList.add('lazy-loaded');
        
        // Support for WebP
        if (img.dataset.webp && this.supportsWebP()) {
            img.src = img.dataset.webp;
        }
    }
    
    loadAllImages() {
        const images = document.querySelectorAll(this.options.selector);
        images.forEach(img => this.loadImage(img));
    }
    
    supportsWebP() {
        const canvas = document.createElement('canvas');
        canvas.width = 1;
        canvas.height = 1;
        return canvas.toDataURL('image/webp').indexOf('data:image/webp') === 0;
    }
    
    refresh() {
        if (this.observer) {
            this.observeImages();
        }
    }
}

// Global instances
let photoUpload, memberStatusTracker, lazyImageLoader;

// Initialize when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    // Initialize photo upload if drop zone exists
    if (document.querySelector('#photo-drop-zone')) {
        photoUpload = new EnhancedPhotoUpload({
            onUploadStart: (session) => {
                console.log('Upload started:', session.file.name);
            },
            onUploadProgress: (session) => {
                console.log(`Upload progress: ${session.progress}%`);
            },
            onUploadComplete: (session) => {
                console.log('Upload completed:', session.result);
                showNotification('Photo uploaded successfully!', 'success');
            },
            onUploadError: (error, session) => {
                console.error('Upload error:', error);
                showNotification(`Upload failed: ${error.message}`, 'error');
            },
            onPreviewGenerated: (preview) => {
                console.log('Preview generated:', preview.fileName);
            }
        });
    }
    
    // Initialize member status tracker
    const memberId = document.body.dataset.memberId;
    if (memberId) {
        memberStatusTracker = new MemberStatusTracker({
            memberId: parseInt(memberId),
            onStatusUpdate: (members) => {
                console.log(`Status update: ${members.length} members`);
            },
            onOnlineCountUpdate: (onlineCount, totalActive) => {
                console.log(`Online: ${onlineCount}/${totalActive}`);
            }
        });
        
        memberStatusTracker.start();
    }
    
    // Initialize lazy image loading
    lazyImageLoader = new LazyImageLoader();
});

// Helper function for notifications
function showNotification(message, type = 'info') {
    // You can integrate with your notification system here
    console.log(`[${type.toUpperCase()}] ${message}`);
}

// Export for global use
window.EnhancedPhotoUpload = EnhancedPhotoUpload;
window.MemberStatusTracker = MemberStatusTracker;
window.LazyImageLoader = LazyImageLoader;
