// Event Registrations Management JavaScript
class EventRegistrationsManager {
    constructor() {
        this.currentPage = 1;
        this.itemsPerPage = 10;
        this.totalItems = 0;
        this.currentFilters = {};
        this.currentSort = { field: 'registration_date', order: 'desc' };
        this.registrations = [];
        this.events = [];
        this.members = [];
    }

    // Load all event registrations with pagination and filters
    async loadEventRegistrations(page = 1) {
        try {
            showLoading();
            this.currentPage = page;
            
            const params = new URLSearchParams({
                page: page,
                limit: this.itemsPerPage,
                sort_field: this.currentSort.field,
                sort_order: this.currentSort.order,
                ...this.currentFilters
            });

            const response = await fetch(`../../api/events/registrations.php?${params}`);
            const data = await response.json();
            
            if (data.success) {
                this.registrations = data.data.registrations;
                this.totalItems = data.data.total;
                this.renderRegistrationsTable();
                this.renderPagination();
            } else {
                showError('Error loading registrations: ' + data.message);
            }
        } catch (error) {
            console.error('Error loading registrations:', error);
            showError('Failed to load registrations');
        } finally {
            hideLoading();
        }
    }

    // Load statistics
    async loadStats() {
        try {
            const response = await fetch('../../api/events/registrations.php?action=stats');
            const data = await response.json();
            
            if (data.success) {
                document.getElementById('totalRegistrations').textContent = data.data.total || 0;
                document.getElementById('pendingRegistrations').textContent = data.data.pending || 0;
                document.getElementById('confirmedRegistrations').textContent = data.data.confirmed || 0;
                document.getElementById('cancelledRegistrations').textContent = data.data.cancelled || 0;
            }
        } catch (error) {
            console.error('Error loading stats:', error);
        }
    }

    // Load events for dropdown
    async loadEvents() {
        try {
            const response = await fetch('../../api/events/index.php?limit=1000');
            const data = await response.json();
            
            if (data.success) {
                this.events = data.data.events;
                this.populateEventSelect();
            }
        } catch (error) {
            console.error('Error loading events:', error);
        }
    }

    // Load members for dropdown
    async loadMembers() {
        try {
            const response = await fetch('../../api/members/index.php?status=approved&limit=1000');
            const data = await response.json();
            
            if (data.success) {
                this.members = data.data.members;
                this.populateMemberSelect();
            }
        } catch (error) {
            console.error('Error loading members:', error);
        }
    }

    // Populate event select dropdown
    populateEventSelect() {
        const selects = document.querySelectorAll('#regEvent, #filterEvent');
        selects.forEach(select => {
            const isFilter = select.id === 'filterEvent';
            select.innerHTML = isFilter ? '<option value="">All Events</option>' : '<option value="">Choose event...</option>';
            
            this.events.forEach(event => {
                const option = document.createElement('option');
                option.value = event.id;
                option.textContent = `${event.title} (${this.formatDate(event.event_date)})`;
                select.appendChild(option);
            });
        });
    }

    // Populate member select dropdown
    populateMemberSelect() {
        const select = document.getElementById('regMember');
        select.innerHTML = '<option value="">Choose member...</option>';
        
        this.members.forEach(member => {
            const option = document.createElement('option');
            option.value = member.id;
            option.textContent = `${member.full_name} (KSO-${member.kso_id})`;
            select.appendChild(option);
        });
    }

    // Render registrations table
    renderRegistrationsTable() {
        const tbody = document.getElementById('registrationsTableBody');
        tbody.innerHTML = '';

        if (this.registrations.length === 0) {
            tbody.innerHTML = `
                <tr>
                    <td colspan="7" class="text-center py-4">
                        <div class="text-muted">
                            <i class="bi bi-calendar-check fs-1"></i>
                            <p class="mt-2">No registrations found</p>
                        </div>
                    </td>
                </tr>
            `;
            return;
        }

        this.registrations.forEach(registration => {
            const row = this.createRegistrationRow(registration);
            tbody.appendChild(row);
        });

        // Update showing info
        const start = (this.currentPage - 1) * this.itemsPerPage + 1;
        const end = Math.min(this.currentPage * this.itemsPerPage, this.totalItems);
        document.getElementById('showingStart').textContent = start;
        document.getElementById('showingEnd').textContent = end;
        document.getElementById('totalRecords').textContent = this.totalItems;
    }

    // Create a table row for a registration
    createRegistrationRow(registration) {
        const row = document.createElement('tr');
        
        const statusBadge = this.getStatusBadge(registration.status);
        
        row.innerHTML = `
            <td>
                <input type="checkbox" class="form-check-input" value="${registration.id}">
            </td>
            <td>
                <div class="d-flex align-items-center">
                    <img src="../assets/img/user-placeholder.png" class="rounded-circle me-2" width="32" height="32" alt="Photo">
                    <div>
                        <div class="fw-bold">${registration.member_name}</div>
                        <small class="text-muted">KSO-${registration.member_kso_id}</small>
                    </div>
                </div>
            </td>
            <td>
                <div>
                    <div class="fw-bold">${registration.event_title}</div>
                    <small class="text-muted">${this.formatDate(registration.event_date)}</small>
                </div>
            </td>
            <td>${this.formatDateTime(registration.registration_date)}</td>
            <td>
                <span class="badge bg-info">${registration.attendee_count || 1}</span>
            </td>
            <td>${statusBadge}</td>
            <td>
                <div class="btn-group" role="group">
                    <button type="button" class="btn btn-sm btn-outline-primary" onclick="eventRegistrationsManager.viewRegistration(${registration.id})" title="View">
                        <i class="bi bi-eye"></i>
                    </button>
                    <button type="button" class="btn btn-sm btn-outline-success" onclick="eventRegistrationsManager.editRegistration(${registration.id})" title="Edit">
                        <i class="bi bi-pencil"></i>
                    </button>
                    ${registration.status === 'pending' ? `
                        <button type="button" class="btn btn-sm btn-outline-success" onclick="eventRegistrationsManager.confirmRegistration(${registration.id})" title="Confirm">
                            <i class="bi bi-check"></i>
                        </button>
                    ` : ''}
                    ${registration.status !== 'cancelled' ? `
                        <button type="button" class="btn btn-sm btn-outline-danger" onclick="eventRegistrationsManager.cancelRegistration(${registration.id})" title="Cancel">
                            <i class="bi bi-x"></i>
                        </button>
                    ` : ''}
                </div>
            </td>
        `;
        
        return row;
    }

    // Get status badge HTML
    getStatusBadge(status) {
        const badges = {
            'pending': '<span class="badge bg-warning">Pending</span>',
            'confirmed': '<span class="badge bg-success">Confirmed</span>',
            'cancelled': '<span class="badge bg-danger">Cancelled</span>',
            'attended': '<span class="badge bg-info">Attended</span>',
            'no_show': '<span class="badge bg-secondary">No Show</span>'
        };
        return badges[status] || '<span class="badge bg-secondary">Unknown</span>';
    }

    // Create new registration
    async createRegistration() {
        try {
            const form = document.getElementById('registrationForm');
            
            // Get form data
            const data = {
                event_id: document.getElementById('regEvent').value,
                member_id: document.getElementById('regMember').value,
                attendee_count: document.getElementById('regAttendeeCount').value || 1,
                special_requirements: document.getElementById('regSpecialRequirements').value,
                notes: document.getElementById('regNotes').value
            };

            // Validate
            if (!data.event_id || !data.member_id) {
                showError('Please fill in all required fields');
                return;
            }

            showLoading();
            const response = await fetch('../../api/events/registrations.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(data)
            });

            const result = await response.json();
            
            if (result.success) {
                showSuccess('Registration created successfully');
                bootstrap.Modal.getInstance(document.getElementById('newRegistrationModal')).hide();
                this.loadEventRegistrations();
                this.loadStats();
                form.reset();
            } else {
                showError('Error creating registration: ' + result.message);
            }
        } catch (error) {
            console.error('Error creating registration:', error);
            showError('Failed to create registration');
        } finally {
            hideLoading();
        }
    }

    // View registration details
    async viewRegistration(registrationId) {
        try {
            const response = await fetch(`../../api/events/registrations.php?id=${registrationId}`);
            const data = await response.json();
            
            if (data.success) {
                this.showRegistrationDetails(data.data);
            } else {
                showError('Error loading registration details: ' + data.message);
            }
        } catch (error) {
            console.error('Error loading registration:', error);
            showError('Failed to load registration details');
        }
    }

    // Show registration details modal
    showRegistrationDetails(registration) {
        const detailsContent = document.getElementById('registrationDetailsContent');
        
        detailsContent.innerHTML = `
            <div class="row">
                <div class="col-md-6">
                    <h6>Registration Information</h6>
                    <table class="table table-sm">
                        <tr><td><strong>Registration ID:</strong></td><td>#${registration.id}</td></tr>
                        <tr><td><strong>Registration Date:</strong></td><td>${this.formatDateTime(registration.registration_date)}</td></tr>
                        <tr><td><strong>Status:</strong></td><td>${this.getStatusBadge(registration.status)}</td></tr>
                        <tr><td><strong>Attendee Count:</strong></td><td>${registration.attendee_count || 1}</td></tr>
                    </table>
                </div>
                <div class="col-md-6">
                    <h6>Member Information</h6>
                    <table class="table table-sm">
                        <tr><td><strong>Name:</strong></td><td>${registration.member_name}</td></tr>
                        <tr><td><strong>KSO ID:</strong></td><td>KSO-${registration.member_kso_id}</td></tr>
                        <tr><td><strong>Email:</strong></td><td>${registration.member_email || 'N/A'}</td></tr>
                        <tr><td><strong>Phone:</strong></td><td>${registration.member_phone || 'N/A'}</td></tr>
                    </table>
                </div>
            </div>
            <div class="row mt-3">
                <div class="col-md-12">
                    <h6>Event Information</h6>
                    <table class="table table-sm">
                        <tr><td><strong>Event:</strong></td><td>${registration.event_title}</td></tr>
                        <tr><td><strong>Date:</strong></td><td>${this.formatDateTime(registration.event_date)}</td></tr>
                        <tr><td><strong>Venue:</strong></td><td>${registration.event_venue || 'N/A'}</td></tr>
                        <tr><td><strong>Fee:</strong></td><td>₹${registration.event_fee || 0}</td></tr>
                    </table>
                </div>
            </div>
            ${registration.special_requirements ? `
                <div class="row mt-3">
                    <div class="col-md-12">
                        <h6>Special Requirements</h6>
                        <p>${registration.special_requirements}</p>
                    </div>
                </div>
            ` : ''}
            ${registration.notes ? `
                <div class="row mt-3">
                    <div class="col-md-12">
                        <h6>Notes</h6>
                        <p>${registration.notes}</p>
                    </div>
                </div>
            ` : ''}
        `;
        
        new bootstrap.Modal(document.getElementById('registrationDetailsModal')).show();
    }

    // Confirm registration
    async confirmRegistration(registrationId) {
        try {
            showLoading();
            
            const response = await fetch(`../../api/events/registrations.php?id=${registrationId}`, {
                method: 'PUT',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({ status: 'confirmed' })
            });

            const result = await response.json();
            
            if (result.success) {
                showSuccess('Registration confirmed successfully');
                this.loadEventRegistrations();
                this.loadStats();
            } else {
                showError('Error confirming registration: ' + result.message);
            }
        } catch (error) {
            console.error('Error confirming registration:', error);
            showError('Failed to confirm registration');
        } finally {
            hideLoading();
        }
    }

    // Cancel registration
    async cancelRegistration(registrationId) {
        if (!confirm('Are you sure you want to cancel this registration?')) {
            return;
        }

        try {
            showLoading();
            
            const response = await fetch(`../../api/events/registrations.php?id=${registrationId}`, {
                method: 'PUT',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({ status: 'cancelled' })
            });

            const result = await response.json();
            
            if (result.success) {
                showSuccess('Registration cancelled successfully');
                this.loadEventRegistrations();
                this.loadStats();
            } else {
                showError('Error cancelling registration: ' + result.message);
            }
        } catch (error) {
            console.error('Error cancelling registration:', error);
            showError('Failed to cancel registration');
        } finally {
            hideLoading();
        }
    }

    // Search registrations
    searchRegistrations() {
        const searchTerm = document.getElementById('searchRegistrations').value.trim();
        if (searchTerm) {
            this.currentFilters.search = searchTerm;
        } else {
            delete this.currentFilters.search;
        }
        this.loadEventRegistrations(1);
    }

    // Filter registrations
    filterRegistrations() {
        const status = document.getElementById('filterStatus').value;
        const event = document.getElementById('filterEvent').value;
        const date = document.getElementById('filterDate').value;

        // Update filters
        if (status) this.currentFilters.status = status;
        else delete this.currentFilters.status;

        if (event) this.currentFilters.event_id = event;
        else delete this.currentFilters.event_id;

        if (date) this.currentFilters.date = date;
        else delete this.currentFilters.date;

        this.loadEventRegistrations(1);
    }

    // Clear all filters
    clearFilters() {
        document.getElementById('searchRegistrations').value = '';
        document.getElementById('filterStatus').value = '';
        document.getElementById('filterEvent').value = '';
        document.getElementById('filterDate').value = '';
        
        this.currentFilters = {};
        this.loadEventRegistrations(1);
    }

    // Get selected registration IDs
    getSelectedRegistrations() {
        const checkboxes = document.querySelectorAll('#registrationsTableBody input[type="checkbox"]:checked');
        return Array.from(checkboxes).map(cb => parseInt(cb.value));
    }

    // Bulk actions
    bulkConfirm() {
        const selected = this.getSelectedRegistrations();
        if (selected.length === 0) {
            showWarning('Please select registrations to confirm');
            return;
        }
        showInfo(`Bulk confirm functionality will be implemented for ${selected.length} registrations`);
    }

    bulkCancel() {
        const selected = this.getSelectedRegistrations();
        if (selected.length === 0) {
            showWarning('Please select registrations to cancel');
            return;
        }
        if (confirm(`Are you sure you want to cancel ${selected.length} registrations?`)) {
            showInfo(`Bulk cancel functionality will be implemented for ${selected.length} registrations`);
        }
    }

    exportRegistrations() {
        showInfo('Export functionality will be implemented');
    }

    // Render pagination
    renderPagination() {
        const totalPages = Math.ceil(this.totalItems / this.itemsPerPage);
        const pagination = document.getElementById('pagination');
        pagination.innerHTML = '';

        if (totalPages <= 1) return;

        // Previous button
        const prevLi = document.createElement('li');
        prevLi.className = `page-item ${this.currentPage === 1 ? 'disabled' : ''}`;
        prevLi.innerHTML = `<a class="page-link" href="#" onclick="eventRegistrationsManager.loadEventRegistrations(${this.currentPage - 1})">Previous</a>`;
        pagination.appendChild(prevLi);

        // Page numbers
        const startPage = Math.max(1, this.currentPage - 2);
        const endPage = Math.min(totalPages, this.currentPage + 2);

        for (let i = startPage; i <= endPage; i++) {
            const li = document.createElement('li');
            li.className = `page-item ${i === this.currentPage ? 'active' : ''}`;
            li.innerHTML = `<a class="page-link" href="#" onclick="eventRegistrationsManager.loadEventRegistrations(${i})">${i}</a>`;
            pagination.appendChild(li);
        }

        // Next button
        const nextLi = document.createElement('li');
        nextLi.className = `page-item ${this.currentPage === totalPages ? 'disabled' : ''}`;
        nextLi.innerHTML = `<a class="page-link" href="#" onclick="eventRegistrationsManager.loadEventRegistrations(${this.currentPage + 1})">Next</a>`;
        pagination.appendChild(nextLi);
    }

    // Utility methods
    formatDate(dateString) {
        if (!dateString) return 'N/A';
        return new Date(dateString).toLocaleDateString();
    }

    formatDateTime(dateString) {
        if (!dateString) return 'N/A';
        return new Date(dateString).toLocaleString();
    }
}

// Global instance
const eventRegistrationsManager = new EventRegistrationsManager();

// Global functions for HTML onclick events
function loadEventRegistrations(page = 1) {
    eventRegistrationsManager.loadEventRegistrations(page);
}

function loadStats() {
    eventRegistrationsManager.loadStats();
}

function loadEvents() {
    eventRegistrationsManager.loadEvents();
}

function loadMembers() {
    eventRegistrationsManager.loadMembers();
}

function createRegistration() {
    eventRegistrationsManager.createRegistration();
}

function searchRegistrations() {
    eventRegistrationsManager.searchRegistrations();
}

function filterRegistrations() {
    eventRegistrationsManager.filterRegistrations();
}

function clearFilters() {
    eventRegistrationsManager.clearFilters();
}

function bulkConfirm() {
    eventRegistrationsManager.bulkConfirm();
}

function bulkCancel() {
    eventRegistrationsManager.bulkCancel();
}

function exportRegistrations() {
    eventRegistrationsManager.exportRegistrations();
}
