// Expenses Management JavaScript
class ExpensesManager {
    constructor() {
        this.currentPage = 1;
        this.itemsPerPage = 10;
        this.expenses = [];
        this.filteredExpenses = [];
        this.selectedExpenses = [];
        this.charts = {};
    }

    // Load expenses from API
    async loadExpenses() {
        try {
            const response = await apiService.get('/api/expenses/index.php');
            if (response.success) {
                this.expenses = response.data;
                this.filteredExpenses = [...this.expenses];
                this.renderExpenses();
            } else {
                showToast('error', 'Failed to load expenses');
            }
        } catch (error) {
            console.error('Error loading expenses:', error);
            showToast('error', 'Error loading expenses');
        }
    }

    // Load statistics
    async loadStats() {
        try {
            const response = await apiService.get('/api/expenses/stats.php');
            if (response.success) {
                const stats = response.data;
                document.getElementById('totalExpenses').textContent = this.formatCurrency(stats.total_expenses || 0);
                document.getElementById('approvedExpenses').textContent = this.formatCurrency(stats.approved_expenses || 0);
                document.getElementById('pendingExpenses').textContent = this.formatCurrency(stats.pending_expenses || 0);
                document.getElementById('rejectedExpenses').textContent = this.formatCurrency(stats.rejected_expenses || 0);
                
                this.updateCharts(stats);
            }
        } catch (error) {
            console.error('Error loading stats:', error);
        }
    }

    // Initialize charts
    initializeCharts() {
        // Monthly Expense Trends Chart
        const expenseCtx = document.getElementById('expenseChart').getContext('2d');
        this.charts.expense = new Chart(expenseCtx, {
            type: 'line',
            data: {
                labels: [],
                datasets: [{
                    label: 'Monthly Expenses',
                    data: [],
                    borderColor: '#007bff',
                    backgroundColor: 'rgba(0, 123, 255, 0.1)',
                    tension: 0.4,
                    fill: true
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            callback: function(value) {
                                return '₹' + value.toLocaleString();
                            }
                        }
                    }
                }
            }
        });

        // Category Breakdown Chart
        const categoryCtx = document.getElementById('categoryChart').getContext('2d');
        this.charts.category = new Chart(categoryCtx, {
            type: 'doughnut',
            data: {
                labels: [],
                datasets: [{
                    data: [],
                    backgroundColor: [
                        '#007bff', '#28a745', '#ffc107', '#dc3545', 
                        '#6f42c1', '#fd7e14', '#20c997', '#6c757d'
                    ]
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'bottom'
                    }
                }
            }
        });
    }

    // Update charts with data
    updateCharts(stats) {
        // Update monthly trends
        if (stats.monthly_trends) {
            this.charts.expense.data.labels = stats.monthly_trends.map(trend => trend.month);
            this.charts.expense.data.datasets[0].data = stats.monthly_trends.map(trend => trend.amount);
            this.charts.expense.update();
        }

        // Update category breakdown
        if (stats.category_breakdown) {
            this.charts.category.data.labels = stats.category_breakdown.map(cat => cat.category);
            this.charts.category.data.datasets[0].data = stats.category_breakdown.map(cat => cat.amount);
            this.charts.category.update();
        }
    }

    // Filter expenses
    filterExpenses() {
        const status = document.getElementById('statusFilter').value;
        const category = document.getElementById('categoryFilter').value;
        const dateFrom = document.getElementById('dateFrom').value;
        const dateTo = document.getElementById('dateTo').value;
        const search = document.getElementById('searchInput').value.toLowerCase();
        
        this.filteredExpenses = this.expenses.filter(expense => {
            const matchStatus = !status || expense.status === status;
            const matchCategory = !category || expense.category === category;
            const matchDateFrom = !dateFrom || new Date(expense.expense_date) >= new Date(dateFrom);
            const matchDateTo = !dateTo || new Date(expense.expense_date) <= new Date(dateTo);
            const matchSearch = !search || 
                expense.description.toLowerCase().includes(search) ||
                expense.notes.toLowerCase().includes(search) ||
                expense.submitted_by_name.toLowerCase().includes(search);
            
            return matchStatus && matchCategory && matchDateFrom && matchDateTo && matchSearch;
        });
        
        this.currentPage = 1;
        this.renderExpenses();
    }

    // Render expenses table
    renderExpenses() {
        const tbody = document.getElementById('expensesTableBody');
        const startIndex = (this.currentPage - 1) * this.itemsPerPage;
        const endIndex = startIndex + this.itemsPerPage;
        const pageExpenses = this.filteredExpenses.slice(startIndex, endIndex);
        
        tbody.innerHTML = '';
        
        if (pageExpenses.length === 0) {
            tbody.innerHTML = `
                <tr>
                    <td colspan="9" class="text-center py-4">
                        <i class="bi bi-receipt" style="font-size: 3rem; color: #6c757d;"></i>
                        <h5 class="mt-3 text-muted">No expenses found</h5>
                        <p class="text-muted">Try adjusting your search criteria or add a new expense.</p>
                    </td>
                </tr>
            `;
            return;
        }
        
        pageExpenses.forEach(expense => {
            const row = document.createElement('tr');
            row.className = `expense-${expense.status}`;
            
            row.innerHTML = `
                <td>
                    <input type="checkbox" class="expense-checkbox" value="${expense.id}" 
                           onchange="expensesManager.toggleExpenseSelection(${expense.id})">
                </td>
                <td>
                    ${expense.receipt_url ? 
                        `<img src="${expense.receipt_url}" class="receipt-preview" 
                              onclick="expensesManager.viewReceipt('${expense.receipt_url}')" 
                              alt="Receipt">` :
                        '<i class="bi bi-file-earmark text-muted"></i>'
                    }
                </td>
                <td>
                    <strong>${expense.description}</strong>
                    ${expense.notes ? `<br><small class="text-muted">${expense.notes}</small>` : ''}
                </td>
                <td>
                    <span class="amount-display text-${this.getAmountColor(expense.amount)}">
                        ${this.formatCurrency(expense.amount)}
                    </span>
                </td>
                <td>
                    <span class="badge category-badge badge-secondary">${expense.category}</span>
                </td>
                <td>
                    ${this.getStatusBadge(expense.status)}
                </td>
                <td>
                    ${expense.submitted_by_name}<br>
                    <small class="text-muted">${expense.submitted_by_role || 'Member'}</small>
                </td>
                <td>
                    ${this.formatDate(expense.expense_date)}<br>
                    <small class="text-muted">Submitted: ${this.formatDate(expense.created_at)}</small>
                </td>
                <td>
                    <div class="btn-group" role="group">
                        <button class="btn btn-sm btn-outline-info" onclick="expensesManager.viewExpense(${expense.id})" title="View Details">
                            <i class="bi bi-eye"></i>
                        </button>
                        ${expense.status === 'draft' ? `
                            <button class="btn btn-sm btn-outline-warning" onclick="expensesManager.editExpense(${expense.id})" title="Edit">
                                <i class="bi bi-pencil"></i>
                            </button>
                        ` : ''}
                        ${expense.status === 'pending' ? `
                            <button class="btn btn-sm btn-outline-success" onclick="expensesManager.approveExpense(${expense.id})" title="Approve">
                                <i class="bi bi-check"></i>
                            </button>
                            <button class="btn btn-sm btn-outline-danger" onclick="expensesManager.rejectExpense(${expense.id})" title="Reject">
                                <i class="bi bi-x"></i>
                            </button>
                        ` : ''}
                        ${expense.status !== 'approved' ? `
                            <button class="btn btn-sm btn-outline-danger" onclick="expensesManager.deleteExpenseConfirm(${expense.id})" title="Delete">
                                <i class="bi bi-trash"></i>
                            </button>
                        ` : ''}
                    </div>
                </td>
            `;
            
            tbody.appendChild(row);
        });
        
        this.renderPagination();
    }

    // Get amount color based on value
    getAmountColor(amount) {
        if (amount > 10000) return 'danger';
        if (amount > 5000) return 'warning';
        return 'primary';
    }

    // Get status badge
    getStatusBadge(status) {
        const badges = {
            draft: '<span class="badge badge-secondary">Draft</span>',
            pending: '<span class="badge badge-warning">Pending</span>',
            approved: '<span class="badge badge-success">Approved</span>',
            rejected: '<span class="badge badge-danger">Rejected</span>'
        };
        return badges[status] || '<span class="badge badge-secondary">Unknown</span>';
    }

    // Render pagination
    renderPagination() {
        const totalPages = Math.ceil(this.filteredExpenses.length / this.itemsPerPage);
        const pagination = document.getElementById('pagination');
        
        // Update showing info
        const startIndex = (this.currentPage - 1) * this.itemsPerPage + 1;
        const endIndex = Math.min(this.currentPage * this.itemsPerPage, this.filteredExpenses.length);
        
        document.getElementById('showingStart').textContent = this.filteredExpenses.length > 0 ? startIndex : 0;
        document.getElementById('showingEnd').textContent = endIndex;
        document.getElementById('totalRecords').textContent = this.filteredExpenses.length;
        
        pagination.innerHTML = '';
        
        if (totalPages <= 1) return;
        
        // Previous button
        const prevLi = document.createElement('li');
        prevLi.className = `page-item ${this.currentPage === 1 ? 'disabled' : ''}`;
        prevLi.innerHTML = `<a class="page-link" href="#" onclick="expensesManager.changePage(${this.currentPage - 1})">Previous</a>`;
        pagination.appendChild(prevLi);
        
        // Page numbers
        const startPage = Math.max(1, this.currentPage - 2);
        const endPage = Math.min(totalPages, this.currentPage + 2);
        
        for (let i = startPage; i <= endPage; i++) {
            const pageLi = document.createElement('li');
            pageLi.className = `page-item ${i === this.currentPage ? 'active' : ''}`;
            pageLi.innerHTML = `<a class="page-link" href="#" onclick="expensesManager.changePage(${i})">${i}</a>`;
            pagination.appendChild(pageLi);
        }
        
        // Next button
        const nextLi = document.createElement('li');
        nextLi.className = `page-item ${this.currentPage === totalPages ? 'disabled' : ''}`;
        nextLi.innerHTML = `<a class="page-link" href="#" onclick="expensesManager.changePage(${this.currentPage + 1})">Next</a>`;
        pagination.appendChild(nextLi);
    }

    // Change page
    changePage(page) {
        if (page < 1 || page > Math.ceil(this.filteredExpenses.length / this.itemsPerPage)) return;
        this.currentPage = page;
        this.renderExpenses();
    }

    // Initialize upload functionality
    initializeUpload() {
        const uploadArea = document.getElementById('receiptUploadArea');
        const fileInput = document.getElementById('receiptInput');
        
        uploadArea.addEventListener('click', () => fileInput.click());
        
        uploadArea.addEventListener('dragover', (e) => {
            e.preventDefault();
            uploadArea.classList.add('dragover');
        });
        
        uploadArea.addEventListener('dragleave', (e) => {
            e.preventDefault();
            uploadArea.classList.remove('dragover');
        });
        
        uploadArea.addEventListener('drop', (e) => {
            e.preventDefault();
            uploadArea.classList.remove('dragover');
            this.handleReceiptUpload(e.dataTransfer.files[0]);
        });
        
        fileInput.addEventListener('change', (e) => {
            this.handleReceiptUpload(e.target.files[0]);
        });
    }

    // Handle receipt upload
    handleReceiptUpload(file) {
        if (!file) return;
        
        const validTypes = ['image/jpeg', 'image/jpg', 'image/png', 'application/pdf'];
        const maxSize = 5 * 1024 * 1024; // 5MB
        
        if (!validTypes.includes(file.type)) {
            showToast('error', 'Please upload only images (JPG, PNG) or PDF files');
            return;
        }
        
        if (file.size > maxSize) {
            showToast('error', 'File size should be less than 5MB');
            return;
        }
        
        // Show preview
        const preview = document.getElementById('receiptPreview');
        preview.style.display = 'block';
        
        if (file.type.startsWith('image/')) {
            const reader = new FileReader();
            reader.onload = (e) => {
                preview.innerHTML = `
                    <img src="${e.target.result}" class="img-thumbnail" style="max-height: 150px;">
                    <p class="mt-2 mb-0"><strong>File:</strong> ${file.name}</p>
                    <p class="mb-0"><strong>Size:</strong> ${this.formatFileSize(file.size)}</p>
                `;
            };
            reader.readAsDataURL(file);
        } else {
            preview.innerHTML = `
                <div class="text-center p-3 border rounded">
                    <i class="bi bi-file-pdf" style="font-size: 3rem; color: #dc3545;"></i>
                    <p class="mt-2 mb-0"><strong>File:</strong> ${file.name}</p>
                    <p class="mb-0"><strong>Size:</strong> ${this.formatFileSize(file.size)}</p>
                </div>
            `;
        }
        
        // Store file for later upload
        this.currentReceiptFile = file;
    }

    // Save expense
    async saveExpense() {
        if (!this.validateExpenseForm()) return;
        
        const formData = new FormData();
        const expenseId = document.getElementById('expenseId').value;
        
        // Add form data
        formData.append('description', document.getElementById('expenseDescription').value);
        formData.append('amount', document.getElementById('expenseAmount').value);
        formData.append('category', document.getElementById('expenseCategory').value);
        formData.append('expense_date', document.getElementById('expenseDate').value);
        formData.append('notes', document.getElementById('expenseNotes').value);
        formData.append('status', 'pending');
        
        // Add receipt file if uploaded
        if (this.currentReceiptFile) {
            formData.append('receipt', this.currentReceiptFile);
        }
        
        try {
            let response;
            if (expenseId) {
                response = await apiService.request(`/api/expenses/expense.php?id=${expenseId}`, {
                    method: 'PUT',
                    body: formData
                });
            } else {
                response = await apiService.request('/api/expenses/index.php', {
                    method: 'POST',
                    body: formData
                });
            }
            
            if (response.success) {
                showToast('success', expenseId ? 'Expense updated successfully' : 'Expense submitted for approval');
                $('#expenseModal').modal('hide');
                await this.loadExpenses();
                await this.loadStats();
                this.clearForm();
            } else {
                showToast('error', response.message || 'Failed to save expense');
            }
        } catch (error) {
            console.error('Error saving expense:', error);
            showToast('error', 'Error saving expense');
        }
    }

    // Save as draft
    async saveDraft() {
        if (!this.validateExpenseForm(false)) return;
        
        const formData = new FormData();
        
        // Add form data
        formData.append('description', document.getElementById('expenseDescription').value);
        formData.append('amount', document.getElementById('expenseAmount').value);
        formData.append('category', document.getElementById('expenseCategory').value);
        formData.append('expense_date', document.getElementById('expenseDate').value);
        formData.append('notes', document.getElementById('expenseNotes').value);
        formData.append('status', 'draft');
        
        // Add receipt file if uploaded
        if (this.currentReceiptFile) {
            formData.append('receipt', this.currentReceiptFile);
        }
        
        try {
            const response = await apiService.request('/api/expenses/draft.php', {
                method: 'POST',
                body: formData
            });
            
            if (response.success) {
                showToast('success', 'Expense saved as draft');
                $('#expenseModal').modal('hide');
                await this.loadExpenses();
                this.clearForm();
            } else {
                showToast('error', response.message || 'Failed to save draft');
            }
        } catch (error) {
            console.error('Error saving draft:', error);
            showToast('error', 'Error saving draft');
        }
    }

    // Validate expense form
    validateExpenseForm(requireAll = true) {
        const description = document.getElementById('expenseDescription').value.trim();
        const amount = document.getElementById('expenseAmount').value;
        const category = document.getElementById('expenseCategory').value;
        const date = document.getElementById('expenseDate').value;
        
        if (!description) {
            showToast('error', 'Please enter expense description');
            return false;
        }
        
        if (requireAll) {
            if (!amount || amount <= 0) {
                showToast('error', 'Please enter a valid amount');
                return false;
            }
            
            if (!category) {
                showToast('error', 'Please select a category');
                return false;
            }
            
            if (!date) {
                showToast('error', 'Please select expense date');
                return false;
            }
        }
        
        return true;
    }

    // Clear form
    clearForm() {
        document.getElementById('expenseForm').reset();
        document.getElementById('expenseId').value = '';
        document.getElementById('receiptPreview').style.display = 'none';
        this.currentReceiptFile = null;
    }

    // View expense details
    async viewExpense(id) {
        try {
            const response = await apiService.get(`/api/expenses/expense.php?id=${id}`);
            if (response.success) {
                this.showExpenseDetails(response.data);
            } else {
                showToast('error', 'Failed to load expense details');
            }
        } catch (error) {
            console.error('Error loading expense:', error);
            showToast('error', 'Error loading expense details');
        }
    }

    // Show expense details in modal
    showExpenseDetails(expense) {
        const modalBody = document.getElementById('expenseDetailBody');
        const actions = document.getElementById('expenseActions');
        
        modalBody.innerHTML = `
            <div class="row">
                <div class="col-md-8">
                    <h5>${expense.description}</h5>
                    <div class="row">
                        <div class="col-md-6">
                            <p><strong>Amount:</strong> <span class="text-primary">${this.formatCurrency(expense.amount)}</span></p>
                            <p><strong>Category:</strong> ${expense.category}</p>
                            <p><strong>Date:</strong> ${this.formatDate(expense.expense_date)}</p>
                            <p><strong>Status:</strong> ${this.getStatusBadge(expense.status)}</p>
                        </div>
                        <div class="col-md-6">
                            <p><strong>Submitted By:</strong> ${expense.submitted_by_name}</p>
                            <p><strong>Submitted On:</strong> ${this.formatDate(expense.created_at)}</p>
                            ${expense.approved_by_name ? `<p><strong>Approved By:</strong> ${expense.approved_by_name}</p>` : ''}
                            ${expense.approved_at ? `<p><strong>Approved On:</strong> ${this.formatDate(expense.approved_at)}</p>` : ''}
                        </div>
                    </div>
                    ${expense.notes ? `
                        <div class="mt-3">
                            <strong>Notes:</strong>
                            <p class="text-muted">${expense.notes}</p>
                        </div>
                    ` : ''}
                    ${expense.rejection_reason ? `
                        <div class="mt-3">
                            <strong>Rejection Reason:</strong>
                            <p class="text-danger">${expense.rejection_reason}</p>
                        </div>
                    ` : ''}
                </div>
                <div class="col-md-4">
                    ${expense.receipt_url ? `
                        <div class="text-center">
                            <strong>Receipt:</strong>
                            <div class="mt-2">
                                ${expense.receipt_url.toLowerCase().includes('.pdf') ?
                                    `<div class="border p-3 rounded">
                                        <i class="bi bi-file-pdf" style="font-size: 3rem; color: #dc3545;"></i>
                                        <p class="mt-2 mb-0">PDF Receipt</p>
                                        <button class="btn btn-sm btn-outline-primary mt-2" onclick="expensesManager.viewReceipt('${expense.receipt_url}')">
                                            <i class="bi bi-eye mr-1"></i>View
                                        </button>
                                    </div>` :
                                    `<img src="${expense.receipt_url}" class="img-thumbnail" style="max-width: 100%; max-height: 200px; cursor: pointer;" onclick="expensesManager.viewReceipt('${expense.receipt_url}')">`
                                }
                            </div>
                        </div>
                    ` : '<p class="text-muted">No receipt uploaded</p>'}
                </div>
            </div>
        `;
        
        // Setup action buttons based on status
        actions.innerHTML = '';
        if (expense.status === 'pending') {
            actions.innerHTML = `
                <button type="button" class="btn btn-success" onclick="expensesManager.approveExpense(${expense.id})">
                    <i class="bi bi-check mr-2"></i>Approve
                </button>
                <button type="button" class="btn btn-danger" onclick="expensesManager.rejectExpense(${expense.id})">
                    <i class="bi bi-x mr-2"></i>Reject
                </button>
            `;
        } else if (expense.status === 'draft') {
            actions.innerHTML = `
                <button type="button" class="btn btn-warning" onclick="expensesManager.editExpense(${expense.id}); $('#expenseDetailModal').modal('hide');">
                    <i class="bi bi-pencil mr-2"></i>Edit
                </button>
                <button type="button" class="btn btn-primary" onclick="expensesManager.submitExpense(${expense.id})">
                    <i class="bi bi-send mr-2"></i>Submit for Approval
                </button>
            `;
        }
        
        $('#expenseDetailModal').modal('show');
    }

    // Edit expense
    async editExpense(id) {
        const expense = this.expenses.find(e => e.id === id);
        if (!expense) return;
        
        // Populate form
        document.getElementById('expenseId').value = expense.id;
        document.getElementById('expenseDescription').value = expense.description;
        document.getElementById('expenseAmount').value = expense.amount;
        document.getElementById('expenseCategory').value = expense.category;
        document.getElementById('expenseDate').value = expense.expense_date;
        document.getElementById('expenseNotes').value = expense.notes || '';
        
        // Show receipt preview if exists
        if (expense.receipt_url) {
            const preview = document.getElementById('receiptPreview');
            preview.style.display = 'block';
            
            if (expense.receipt_url.toLowerCase().includes('.pdf')) {
                preview.innerHTML = `
                    <div class="text-center p-3 border rounded">
                        <i class="bi bi-file-pdf" style="font-size: 3rem; color: #dc3545;"></i>
                        <p class="mt-2 mb-0"><strong>Current Receipt:</strong> PDF File</p>
                        <button class="btn btn-sm btn-outline-primary mt-2" onclick="expensesManager.viewReceipt('${expense.receipt_url}')">
                            <i class="bi bi-eye mr-1"></i>View Current
                        </button>
                    </div>
                `;
            } else {
                preview.innerHTML = `
                    <img src="${expense.receipt_url}" class="img-thumbnail" style="max-height: 150px;">
                    <p class="mt-2 mb-0"><strong>Current Receipt</strong></p>
                `;
            }
        }
        
        $('#expenseModal').modal('show');
    }

    // Approve expense
    async approveExpense(id) {
        if (!confirm('Are you sure you want to approve this expense?')) return;
        
        try {
            const response = await apiService.post(`/api/expenses/approve.php?id=${id}`);
            if (response.success) {
                showToast('success', 'Expense approved successfully');
                $('#expenseDetailModal').modal('hide');
                await this.loadExpenses();
                await this.loadStats();
            } else {
                showToast('error', response.message || 'Failed to approve expense');
            }
        } catch (error) {
            console.error('Error approving expense:', error);
            showToast('error', 'Error approving expense');
        }
    }

    // Reject expense
    async rejectExpense(id) {
        const reason = prompt('Please provide a reason for rejection:');
        if (!reason) return;
        
        try {
            const response = await apiService.post(`/api/expenses/reject.php?id=${id}`, {
                rejection_reason: reason
            });
            if (response.success) {
                showToast('success', 'Expense rejected');
                $('#expenseDetailModal').modal('hide');
                await this.loadExpenses();
                await this.loadStats();
            } else {
                showToast('error', response.message || 'Failed to reject expense');
            }
        } catch (error) {
            console.error('Error rejecting expense:', error);
            showToast('error', 'Error rejecting expense');
        }
    }

    // Submit expense for approval
    async submitExpense(id) {
        try {
            const response = await apiService.post(`/api/expenses/submit.php?id=${id}`);
            if (response.success) {
                showToast('success', 'Expense submitted for approval');
                $('#expenseDetailModal').modal('hide');
                await this.loadExpenses();
                await this.loadStats();
            } else {
                showToast('error', response.message || 'Failed to submit expense');
            }
        } catch (error) {
            console.error('Error submitting expense:', error);
            showToast('error', 'Error submitting expense');
        }
    }

    // Delete expense confirmation
    deleteExpenseConfirm(id) {
        if (confirm('Are you sure you want to delete this expense?')) {
            this.deleteExpense(id);
        }
    }

    // Delete expense
    async deleteExpense(id) {
        try {
            const response = await apiService.delete(`/api/expenses/expense.php?id=${id}`);
            if (response.success) {
                showToast('success', 'Expense deleted successfully');
                await this.loadExpenses();
                await this.loadStats();
            } else {
                showToast('error', response.message || 'Failed to delete expense');
            }
        } catch (error) {
            console.error('Error deleting expense:', error);
            showToast('error', 'Error deleting expense');
        }
    }

    // View receipt
    viewReceipt(url) {
        const viewer = document.getElementById('receiptViewer');
        
        if (url.toLowerCase().includes('.pdf')) {
            viewer.innerHTML = `<iframe src="${url}" width="100%" height="500px"></iframe>`;
        } else {
            viewer.innerHTML = `<img src="${url}" class="img-fluid" alt="Receipt">`;
        }
        
        $('#receiptModal').modal('show');
    }

    // Toggle expense selection
    toggleExpenseSelection(id) {
        const checkbox = document.querySelector(`input[value="${id}"]`);
        if (checkbox.checked) {
            if (!this.selectedExpenses.includes(id)) {
                this.selectedExpenses.push(id);
            }
        } else {
            this.selectedExpenses = this.selectedExpenses.filter(expenseId => expenseId !== id);
        }
        
        this.updateSelectAllCheckbox();
    }

    // Toggle select all
    toggleSelectAll() {
        const selectAllCheckbox = document.getElementById('selectAll');
        const checkboxes = document.querySelectorAll('.expense-checkbox');
        
        checkboxes.forEach(checkbox => {
            checkbox.checked = selectAllCheckbox.checked;
            const id = parseInt(checkbox.value);
            
            if (selectAllCheckbox.checked) {
                if (!this.selectedExpenses.includes(id)) {
                    this.selectedExpenses.push(id);
                }
            } else {
                this.selectedExpenses = this.selectedExpenses.filter(expenseId => expenseId !== id);
            }
        });
    }

    // Update select all checkbox state
    updateSelectAllCheckbox() {
        const selectAllCheckbox = document.getElementById('selectAll');
        const checkboxes = document.querySelectorAll('.expense-checkbox');
        const checkedBoxes = document.querySelectorAll('.expense-checkbox:checked');
        
        selectAllCheckbox.checked = checkboxes.length > 0 && checkboxes.length === checkedBoxes.length;
        selectAllCheckbox.indeterminate = checkedBoxes.length > 0 && checkboxes.length !== checkedBoxes.length;
    }

    // Approve multiple expenses
    async approveMultiple() {
        if (this.selectedExpenses.length === 0) {
            showToast('warning', 'Please select expenses to approve');
            return;
        }
        
        const pendingExpenses = this.selectedExpenses.filter(id => {
            const expense = this.expenses.find(e => e.id === id);
            return expense && expense.status === 'pending';
        });
        
        if (pendingExpenses.length === 0) {
            showToast('warning', 'No pending expenses selected');
            return;
        }
        
        if (!confirm(`Are you sure you want to approve ${pendingExpenses.length} expense(s)?`)) return;
        
        try {
            const response = await apiService.post('/api/expenses/approve-multiple.php', {
                expense_ids: pendingExpenses
            });
            
            if (response.success) {
                showToast('success', `${pendingExpenses.length} expense(s) approved successfully`);
                this.selectedExpenses = [];
                document.getElementById('selectAll').checked = false;
                await this.loadExpenses();
                await this.loadStats();
            } else {
                showToast('error', response.message || 'Failed to approve expenses');
            }
        } catch (error) {
            console.error('Error approving expenses:', error);
            showToast('error', 'Error approving expenses');
        }
    }

    // Export expenses
    async exportExpenses() {
        try {
            const filters = {
                status: document.getElementById('statusFilter').value,
                category: document.getElementById('categoryFilter').value,
                date_from: document.getElementById('dateFrom').value,
                date_to: document.getElementById('dateTo').value,
                search: document.getElementById('searchInput').value
            };
            
            const queryString = new URLSearchParams(filters).toString();
            const url = `/api/expenses/export.php?${queryString}`;
            
            // Create download link
            const link = document.createElement('a');
            link.href = url;
            link.download = `expenses_${new Date().toISOString().split('T')[0]}.csv`;
            link.click();
            
            showToast('success', 'Expenses export started');
        } catch (error) {
            console.error('Error exporting expenses:', error);
            showToast('error', 'Error exporting expenses');
        }
    }

    // Utility: Format currency
    formatCurrency(amount) {
        return '₹' + parseFloat(amount).toLocaleString('en-IN', {
            minimumFractionDigits: 2,
            maximumFractionDigits: 2
        });
    }

    // Utility: Format file size
    formatFileSize(bytes) {
        if (bytes === 0) return '0 Bytes';
        const k = 1024;
        const sizes = ['Bytes', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
    }

    // Utility: Format date
    formatDate(dateString) {
        const date = new Date(dateString);
        return date.toLocaleDateString('en-IN');
    }
}

// Global functions
function loadExpenses() {
    expensesManager.loadExpenses();
}

function loadStats() {
    expensesManager.loadStats();
}

function initializeCharts() {
    expensesManager.initializeCharts();
}

function initializeUpload() {
    expensesManager.initializeUpload();
}

function filterExpenses() {
    expensesManager.filterExpenses();
}

function saveExpense() {
    expensesManager.saveExpense();
}

function saveDraft() {
    expensesManager.saveDraft();
}

// Initialize
const expensesManager = new ExpensesManager();

// Search on enter key
document.addEventListener('DOMContentLoaded', function() {
    const searchInput = document.getElementById('searchInput');
    if (searchInput) {
        searchInput.addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                filterExpenses();
            }
        });
    }
});
