// Finance Management JavaScript Module
class FinanceManager {
    constructor() {
        this.apiBaseUrl = window.CONFIG ? CONFIG.getApiUrl(CONFIG.ENDPOINTS.FINANCE) : '/api/finance';
        this.currentPage = 1;
        this.itemsPerPage = 20;
        this.totalPages = 1;
        this.transactions = [];
        
        this.init();
    }
    
    init() {
        this.bindEvents();
        this.loadTransactions();
        this.loadStatistics();
    }
    
    bindEvents() {
        // Add transaction form
        const addForm = document.getElementById('addTransactionForm');
        if (addForm) {
            addForm.addEventListener('submit', (e) => this.handleAddTransaction(e));
        }
        
        // Edit transaction form
        const editForm = document.getElementById('editTransactionForm');
        if (editForm) {
            editForm.addEventListener('submit', (e) => this.handleEditTransaction(e));
        }
        
        // Search and filters
        const searchInput = document.getElementById('transactionSearch');
        if (searchInput) {
            searchInput.addEventListener('input', debounce(() => this.handleSearch(), 500));
        }
        
        const typeFilter = document.getElementById('transactionTypeFilter');
        if (typeFilter) {
            typeFilter.addEventListener('change', () => this.handleFilter());
        }
        
        const dateFromFilter = document.getElementById('dateFromFilter');
        if (dateFromFilter) {
            dateFromFilter.addEventListener('change', () => this.handleFilter());
        }
        
        const dateToFilter = document.getElementById('dateToFilter');
        if (dateToFilter) {
            dateToFilter.addEventListener('change', () => this.handleFilter());
        }
        
        // Pagination
        const prevBtn = document.getElementById('prevPage');
        const nextBtn = document.getElementById('nextPage');
        
        if (prevBtn) prevBtn.addEventListener('click', () => this.previousPage());
        if (nextBtn) nextBtn.addEventListener('click', () => this.nextPage());
        
        // Export buttons
        const exportBtn = document.getElementById('exportTransactions');
        if (exportBtn) {
            exportBtn.addEventListener('click', () => this.exportTransactions());
        }
        
        // Advanced Filtering
        const filterForm = document.getElementById('financeFilterForm');
        if (filterForm) {
            filterForm.addEventListener('submit', (e) => this.handleFilterForm(e));
        }
        // Export
        const exportExcelBtn = document.getElementById('exportExcelBtn');
        if (exportExcelBtn) exportExcelBtn.addEventListener('click', () => this.exportToExcel());
        const exportPdfBtn = document.getElementById('exportPdfBtn');
        if (exportPdfBtn) exportPdfBtn.addEventListener('click', () => this.exportToPdf());
        // Audit Trail
        const auditTrailModal = document.getElementById('auditTrailModal');
        if (auditTrailModal) {
            auditTrailModal.addEventListener('show.bs.modal', () => this.loadAuditTrail());
        }
        // Recurring Transactions, Budget, Attachments, Approval, Notifications, Role-based Access
        // (Scaffolded for future implementation)
        
        // Filter/search
        document.getElementById('filterForm')?.addEventListener('submit', e => {
            e.preventDefault();
            this.loadTransactions();
        });
        // Export
        document.getElementById('exportExcelBtn')?.addEventListener('click', () => this.export('excel'));
        document.getElementById('exportPdfBtn')?.addEventListener('click', () => this.export('pdf'));
        // Recurring (scaffold)
        document.getElementById('addRecurringBtn')?.addEventListener('click', () => this.openRecurringModal());
        // Budgeting
        document.getElementById('budgetForm')?.addEventListener('submit', e => {
            e.preventDefault();
            this.saveBudget();
        });
        // Audit trail
        document.getElementById('auditTrailBtn')?.addEventListener('click', () => this.openAuditTrail());
        // Attachment upload (in modals)
        document.querySelectorAll('.attachment-input').forEach(input => {
            input.addEventListener('change', e => this.uploadAttachment(e.target));
        });
        // Approval workflow, notifications, role-based access (scaffold)
    }
    
    async loadTransactions(page = 1) {
        try {
            showLoading();
            
            const params = new URLSearchParams({
                page: page,
                limit: this.itemsPerPage
            });
            
            // Add filters
            const search = document.getElementById('transactionSearch')?.value;
            const type = document.getElementById('transactionTypeFilter')?.value;
            const dateFrom = document.getElementById('dateFromFilter')?.value;
            const dateTo = document.getElementById('dateToFilter')?.value;
            
            if (search) params.append('search', search);
            if (type) params.append('type', type);
            if (dateFrom) params.append('date_from', dateFrom);
            if (dateTo) params.append('date_to', dateTo);
            
            const response = await fetch(`${this.apiBaseUrl}/index.php?${params}`);
            const result = await response.json();
            
            if (result.success) {
                this.transactions = result.data;
                this.currentPage = result.pagination.page;
                this.totalPages = result.pagination.pages;
                this.displayTransactions();
                this.updatePagination();
            } else {
                showError('Failed to load transactions: ' + result.error);
            }
        } catch (error) {
            showError('Error loading transactions: ' + error.message);
        } finally {
            hideLoading();
        }
    }
    
    async loadStatistics() {
        try {
            const response = await fetch(`${this.apiBaseUrl}/stats.php`);
            const result = await response.json();
            
            if (result.success) {
                this.displayStatistics(result.data);
            } else {
                console.error('Failed to load statistics:', result.error);
            }
        } catch (error) {
            console.error('Error loading statistics:', error);
        }
    }
    
    displayStatistics(stats) {
        // Update statistics cards
        const totalIncomeEl = document.getElementById('totalIncome');
        const totalExpensesEl = document.getElementById('totalExpenses');
        const netBalanceEl = document.getElementById('netBalance');
        const monthlyIncomeEl = document.getElementById('monthlyIncome');
        const monthlyExpensesEl = document.getElementById('monthlyExpenses');
        const transactionsCountEl = document.getElementById('transactionsCount');
        
        if (totalIncomeEl) totalIncomeEl.textContent = formatCurrency(stats.total_income || 0);
        if (totalExpensesEl) totalExpensesEl.textContent = formatCurrency(stats.total_expenses || 0);
        if (netBalanceEl) {
            netBalanceEl.textContent = formatCurrency(stats.net_balance || 0);
            netBalanceEl.className = stats.net_balance >= 0 ? 'text-success' : 'text-danger';
        }
        if (monthlyIncomeEl) monthlyIncomeEl.textContent = formatCurrency(stats.monthly_income || 0);
        if (monthlyExpensesEl) monthlyExpensesEl.textContent = formatCurrency(stats.monthly_expenses || 0);
        if (transactionsCountEl) transactionsCountEl.textContent = stats.transactions_count || 0;
        
        // Update charts if available
        this.updateCharts(stats);
    }
    
    updateCharts(stats) {
        // Income vs Expenses Chart
        if (window.Chart && stats.income_vs_expenses_chart) {
            const ctx = document.getElementById('incomeExpensesChart');
            if (ctx) {
                this.renderIncomeExpensesChart(ctx, stats.income_vs_expenses_chart);
            }
        }
        
        // Monthly trends
        if (stats.monthly_trends) {
            this.updateTrendsDisplay(stats.monthly_trends);
        }
    }
    
    renderIncomeExpensesChart(ctx, data) {
        const labels = data.map(item => {
            const date = new Date(item.month + '-01');
            return date.toLocaleDateString('en-US', { month: 'short', year: 'numeric' });
        });
        const incomeData = data.map(item => item.income || 0);
        const expenseData = data.map(item => item.expenses || 0);
        
        new Chart(ctx, {
            type: 'line',
            data: {
                labels: labels,
                datasets: [{
                    label: 'Income',
                    data: incomeData,
                    borderColor: 'rgb(75, 192, 192)',
                    backgroundColor: 'rgba(75, 192, 192, 0.2)',
                    tension: 0.1
                }, {
                    label: 'Expenses',
                    data: expenseData,
                    borderColor: 'rgb(255, 99, 132)',
                    backgroundColor: 'rgba(255, 99, 132, 0.2)',
                    tension: 0.1
                }]
            },
            options: {
                responsive: true,
                interaction: {
                    mode: 'index',
                    intersect: false,
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            callback: function(value) {
                                return formatCurrency(value);
                            }
                        }
                    }
                }
            }
        });
    }
    
    displayTransactions() {
        const tbody = document.getElementById('transactionsTableBody');
        if (!tbody) return;
        
        tbody.innerHTML = '';
        
        if (this.transactions.length === 0) {
            tbody.innerHTML = '<tr><td colspan="7" class="text-center">No transactions found</td></tr>';
            return;
        }
        
        this.transactions.forEach(transaction => {
            const row = this.createTransactionRow(transaction);
            tbody.appendChild(row);
        });
    }
    
    createTransactionRow(transaction) {
        const row = document.createElement('tr');
        
        const typeClass = transaction.transaction_type === 'income' ? 'success' : 'danger';
        const typeIcon = transaction.transaction_type === 'income' ? 'bi-arrow-up' : 'bi-arrow-down';
        
        row.innerHTML = `
            <td>
                <span class="badge bg-${typeClass}">
                    <i class="bi ${typeIcon}"></i> ${transaction.transaction_type}
                </span>
            </td>
            <td>${formatCurrency(transaction.amount)}</td>
            <td>${escapeHtml(transaction.description)}</td>
            <td>${transaction.category || '-'}</td>
            <td>${formatDate(transaction.transaction_date)}</td>
            <td>${transaction.reference_number || '-'}</td>
            <td>
                <div class="btn-group btn-group-sm">
                    <button class="btn btn-outline-primary" onclick="financeManager.editTransaction(${transaction.id})" title="Edit">
                        <i class="bi bi-pencil"></i>
                    </button>
                    <button class="btn btn-outline-danger" onclick="financeManager.deleteTransaction(${transaction.id})" title="Delete">
                        <i class="bi bi-trash"></i>
                    </button>
                </div>
            </td>
        `;
        
        return row;
    }
    
    async handleAddTransaction(event) {
        event.preventDefault();
        
        const formData = new FormData(event.target);
        const transactionData = {
            transaction_type: formData.get('transaction_type'),
            amount: parseFloat(formData.get('amount')),
            description: formData.get('description'),
            transaction_date: formData.get('transaction_date'),
            category: formData.get('category'),
            reference_number: formData.get('reference_number'),
            notes: formData.get('notes')
        };
        
        try {
            showLoading();
            
            const response = await fetch(`${this.apiBaseUrl}/index.php`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(transactionData)
            });
            
            const result = await response.json();
            
            if (result.success) {
                showSuccess('Transaction added successfully');
                event.target.reset();
                
                // Close modal if it exists
                const modal = bootstrap.Modal.getInstance(document.getElementById('addTransactionModal'));
                if (modal) modal.hide();
                
                this.loadTransactions(this.currentPage);
                this.loadStatistics();
            } else {
                showError('Failed to add transaction: ' + result.error);
            }
        } catch (error) {
            showError('Error adding transaction: ' + error.message);
        } finally {
            hideLoading();
        }
    }
    
    async editTransaction(id) {
        try {
            const response = await fetch(`${this.apiBaseUrl}/index.php?id=${id}`);
            const result = await response.json();
            
            if (result.success) {
                this.populateEditForm(result.data);
                const modal = new bootstrap.Modal(document.getElementById('editTransactionModal'));
                modal.show();
            } else {
                showError('Failed to load transaction details: ' + result.error);
            }
        } catch (error) {
            showError('Error loading transaction: ' + error.message);
        }
    }
    
    populateEditForm(transaction) {
        const form = document.getElementById('editTransactionForm');
        if (!form) return;
        
        form.dataset.transactionId = transaction.id;
        form.querySelector('[name="transaction_type"]').value = transaction.transaction_type;
        form.querySelector('[name="amount"]').value = transaction.amount;
        form.querySelector('[name="description"]').value = transaction.description;
        form.querySelector('[name="transaction_date"]').value = transaction.transaction_date;
        form.querySelector('[name="category"]').value = transaction.category || '';
        form.querySelector('[name="reference_number"]').value = transaction.reference_number || '';
        form.querySelector('[name="notes"]').value = transaction.notes || '';
    }
    
    async handleEditTransaction(event) {
        event.preventDefault();
        
        const transactionId = event.target.dataset.transactionId;
        const formData = new FormData(event.target);
        const transactionData = {
            transaction_type: formData.get('transaction_type'),
            amount: parseFloat(formData.get('amount')),
            description: formData.get('description'),
            transaction_date: formData.get('transaction_date'),
            category: formData.get('category'),
            reference_number: formData.get('reference_number'),
            notes: formData.get('notes')
        };
        
        try {
            showLoading();
            
            const response = await fetch(`${this.apiBaseUrl}/index.php?id=${transactionId}`, {
                method: 'PUT',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(transactionData)
            });
            
            const result = await response.json();
            
            if (result.success) {
                showSuccess('Transaction updated successfully');
                
                const modal = bootstrap.Modal.getInstance(document.getElementById('editTransactionModal'));
                if (modal) modal.hide();
                
                this.loadTransactions(this.currentPage);
                this.loadStatistics();
            } else {
                showError('Failed to update transaction: ' + result.error);
            }
        } catch (error) {
            showError('Error updating transaction: ' + error.message);
        } finally {
            hideLoading();
        }
    }
    
    async deleteTransaction(id) {
        const reason = prompt('Please provide a reason for deleting this transaction:');
        if (!reason || reason.trim().length < 5) {
            alert('A valid reason (at least 5 characters) is required to delete a transaction.');
            return;
        }
        if (!confirm('Are you sure you want to delete this transaction?')) {
            return;
        }
        try {
            showLoading();
            const response = await fetch(`${this.apiBaseUrl}/index.php?id=${id}`, {
                method: 'DELETE',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ reason })
            });
            const result = await response.json();
            if (result.success) {
                showSuccess('Transaction deleted successfully');
                this.loadTransactions(this.currentPage);
                this.loadStatistics();
            } else {
                showError('Failed to delete transaction: ' + result.error);
            }
        } catch (error) {
            showError('Error deleting transaction: ' + error.message);
        } finally {
            hideLoading();
        }
    }
    
    handleSearch() {
        this.currentPage = 1;
        this.loadTransactions(1);
    }
    
    handleFilter() {
        this.currentPage = 1;
        this.loadTransactions(1);
    }
    
    handleFilterForm(e) {
        e.preventDefault();
        // TODO: Collect filter values and reload transactions
        this.loadTransactions(1);
    }
    
    exportToExcel() {
        // TODO: Implement export to Excel
        alert('Export to Excel coming soon!');
    }
    
    exportToPdf() {
        // TODO: Implement export to PDF
        alert('Export to PDF coming soon!');
    }
    
    async exportTransactions() {
        try {
            const params = new URLSearchParams();
            
            // Add current filters
            const search = document.getElementById('transactionSearch')?.value;
            const type = document.getElementById('transactionTypeFilter')?.value;
            const dateFrom = document.getElementById('dateFromFilter')?.value;
            const dateTo = document.getElementById('dateToFilter')?.value;
            
            if (search) params.append('search', search);
            if (type) params.append('type', type);
            if (dateFrom) params.append('date_from', dateFrom);
            if (dateTo) params.append('date_to', dateTo);
            
            params.append('format', 'csv');
            
            // Create download link
            const url = `${this.apiBaseUrl}/index.php?${params}`;
            const link = document.createElement('a');
            link.href = url;
            link.download = `transactions_${new Date().toISOString().split('T')[0]}.csv`;
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
            
            showSuccess('Export started successfully');
        } catch (error) {
            showError('Error exporting transactions: ' + error.message);
        }
    }
    
    loadAuditTrail() {
        // TODO: Load audit trail data and render in #auditTrailContent
        document.getElementById('auditTrailContent').innerHTML = 'Audit trail feature coming soon!';
    }
    
    openAuditTrail() {
        // Placeholder for audit trail modal logic
        alert('Audit trail feature coming soon!');
    }
    
    previousPage() {
        if (this.currentPage > 1) {
            this.currentPage--;
            this.loadTransactions(this.currentPage);
        }
    }
    
    nextPage() {
        if (this.currentPage < this.totalPages) {
            this.currentPage++;
            this.loadTransactions(this.currentPage);
        }
    }
    
    updatePagination() {
        const pageInfo = document.getElementById('pageInfo');
        const prevBtn = document.getElementById('prevPage');
        const nextBtn = document.getElementById('nextPage');
        
        if (pageInfo) {
            pageInfo.textContent = `Page ${this.currentPage} of ${this.totalPages}`;
        }
        
        if (prevBtn) {
            prevBtn.disabled = this.currentPage <= 1;
        }
        
        if (nextBtn) {
            nextBtn.disabled = this.currentPage >= this.totalPages;
        }
    }
}

// Utility functions
function formatCurrency(amount) {
    return new Intl.NumberFormat('en-IN', {
        style: 'currency',
        currency: 'INR'
    }).format(amount);
}

function formatDate(dateString) {
    return new Date(dateString).toLocaleDateString('en-IN');
}

function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

// Initialize when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    window.financeManager = new FinanceManager();
});
