// ID Cards Management JavaScript
class IDCardsManager {
    constructor() {
        this.currentPage = 1;
        this.itemsPerPage = 10;
        this.totalItems = 0;
        this.currentFilters = {};
        this.currentSort = { field: 'issue_date', order: 'desc' };
        this.cards = [];
        this.members = [];
    }

    // Load all ID cards with pagination and filters
    async loadIDCards(page = 1) {
        try {
            showLoading();
            this.currentPage = page;
            
            const params = new URLSearchParams({
                page: page,
                limit: this.itemsPerPage,
                sort_field: this.currentSort.field,
                sort_order: this.currentSort.order,
                ...this.currentFilters
            });

            const response = await fetch(`../../api/id-cards/index.php?${params}`);
            const data = await response.json();
            
            if (data.success) {
                this.cards = data.data.cards;
                this.totalItems = data.data.total;
                this.renderCardsTable();
                this.renderPagination();
            } else {
                showError('Error loading ID cards: ' + data.message);
            }
        } catch (error) {
            console.error('Error loading ID cards:', error);
            showError('Failed to load ID cards');
        } finally {
            hideLoading();
        }
    }

    // Load statistics
    async loadStats() {
        try {
            const response = await fetch('../../api/id-cards/stats.php');
            const data = await response.json();
            
            if (data.success) {
                document.getElementById('totalCards').textContent = data.data.total || 0;
                document.getElementById('activeCards').textContent = data.data.active || 0;
                document.getElementById('expiringCards').textContent = data.data.expiring_soon || 0;
                document.getElementById('expiredCards').textContent = data.data.expired || 0;
            }
        } catch (error) {
            console.error('Error loading stats:', error);
        }
    }

    // Load members for dropdown
    async loadMembers() {
        try {
            const response = await fetch('../../api/members/index.php?status=approved&limit=1000');
            const data = await response.json();
            
            if (data.success) {
                this.members = data.data.members;
                this.populateMemberSelect();
            }
        } catch (error) {
            console.error('Error loading members:', error);
        }
    }

    // Populate member select dropdown
    populateMemberSelect() {
        const select = document.getElementById('cardMember');
        select.innerHTML = '<option value="">Choose member...</option>';
        
        this.members.forEach(member => {
            const option = document.createElement('option');
            option.value = member.id;
            option.textContent = `${member.full_name} (KSO-${member.kso_id})`;
            select.appendChild(option);
        });
    }

    // Render cards table
    renderCardsTable() {
        const tbody = document.getElementById('cardsTableBody');
        tbody.innerHTML = '';

        if (this.cards.length === 0) {
            tbody.innerHTML = `
                <tr>
                    <td colspan="8" class="text-center py-4">
                        <div class="text-muted">
                            <i class="bi bi-card-heading fs-1"></i>
                            <p class="mt-2">No ID cards found</p>
                        </div>
                    </td>
                </tr>
            `;
            return;
        }

        this.cards.forEach(card => {
            const row = this.createCardRow(card);
            tbody.appendChild(row);
        });

        // Update showing info
        const start = (this.currentPage - 1) * this.itemsPerPage + 1;
        const end = Math.min(this.currentPage * this.itemsPerPage, this.totalItems);
        document.getElementById('showingStart').textContent = start;
        document.getElementById('showingEnd').textContent = end;
        document.getElementById('totalRecords').textContent = this.totalItems;
    }

    // Create a table row for a card
    createCardRow(card) {
        const row = document.createElement('tr');
        row.className = `card-status-${card.status}`;
        
        const statusBadge = this.getStatusBadge(card.status);
        const typeBadge = this.getTypeBadge(card.card_type);
        
        row.innerHTML = `
            <td>
                <input type="checkbox" class="form-check-input" value="${card.id}">
            </td>
            <td>
                <strong>${card.card_number}</strong>
            </td>
            <td>
                <div class="d-flex align-items-center">
                    <img src="../assets/img/user-placeholder.png" class="rounded-circle me-2" width="32" height="32" alt="Photo">
                    <div>
                        <div class="fw-bold">${card.member_name}</div>
                        <small class="text-muted">KSO-${card.member_kso_id}</small>
                    </div>
                </div>
            </td>
            <td>${typeBadge}</td>
            <td>${this.formatDate(card.issue_date)}</td>
            <td>${this.formatDate(card.expiry_date)}</td>
            <td>${statusBadge}</td>
            <td>
                <div class="btn-group" role="group">
                    <button type="button" class="btn btn-sm btn-outline-primary" onclick="idCardsManager.viewCard(${card.id})" title="View">
                        <i class="bi bi-eye"></i>
                    </button>
                    <button type="button" class="btn btn-sm btn-outline-info" onclick="idCardsManager.printCard(${card.id})" title="Print">
                        <i class="bi bi-printer"></i>
                    </button>
                    <button type="button" class="btn btn-sm btn-outline-success" onclick="idCardsManager.downloadCard(${card.id})" title="Download">
                        <i class="bi bi-download"></i>
                    </button>
                    ${card.status === 'active' ? `
                        <div class="btn-group" role="group">
                            <button type="button" class="btn btn-sm btn-outline-secondary dropdown-toggle" data-bs-toggle="dropdown">
                                <i class="bi bi-three-dots"></i>
                            </button>
                            <ul class="dropdown-menu">
                                <li><a class="dropdown-item" href="#" onclick="idCardsManager.editCard(${card.id})">
                                    <i class="bi bi-pencil"></i> Edit
                                </a></li>
                                <li><a class="dropdown-item" href="#" onclick="idCardsManager.replaceCard(${card.id})">
                                    <i class="bi bi-arrow-repeat"></i> Replace
                                </a></li>
                                <li><a class="dropdown-item" href="#" onclick="idCardsManager.cancelCard(${card.id})">
                                    <i class="bi bi-x-circle"></i> Cancel
                                </a></li>
                            </ul>
                        </div>
                    ` : ''}
                </div>
            </td>
        `;
        
        return row;
    }

    // Get status badge HTML
    getStatusBadge(status) {
        const badges = {
            'active': '<span class="badge bg-success">Active</span>',
            'expired': '<span class="badge bg-danger">Expired</span>',
            'cancelled': '<span class="badge bg-secondary">Cancelled</span>',
            'replaced': '<span class="badge bg-warning">Replaced</span>'
        };
        return badges[status] || '<span class="badge bg-secondary">Unknown</span>';
    }

    // Get type badge HTML
    getTypeBadge(type) {
        const badges = {
            'individual': '<span class="badge bg-primary">Individual</span>',
            'family_head': '<span class="badge bg-info">Family Head</span>',
            'dependent': '<span class="badge bg-warning">Dependent</span>'
        };
        return badges[type] || '<span class="badge bg-secondary">Unknown</span>';
    }

    // Generate new ID card
    async generateCard() {
        try {
            const form = document.getElementById('generateCardForm');
            const formData = new FormData(form);
            
            // Convert FormData to JSON
            const data = {
                member_id: document.getElementById('cardMember').value,
                card_type: document.getElementById('cardType').value,
                issue_date: document.getElementById('cardIssueDate').value,
                expiry_date: document.getElementById('cardExpiryDate').value,
                notes: document.getElementById('cardNotes').value,
                generate_qr: document.getElementById('generateQR').checked
            };

            // Validate
            if (!data.member_id || !data.card_type || !data.issue_date || !data.expiry_date) {
                showError('Please fill in all required fields');
                return;
            }

            showLoading();
            const response = await fetch('../../api/id-cards/index.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(data)
            });

            const result = await response.json();
            
            if (result.success) {
                showSuccess('ID card generated successfully');
                bootstrap.Modal.getInstance(document.getElementById('generateCardModal')).hide();
                this.loadIDCards();
                this.loadStats();
                form.reset();
            } else {
                showError('Error generating card: ' + result.message);
            }
        } catch (error) {
            console.error('Error generating card:', error);
            showError('Failed to generate card');
        } finally {
            hideLoading();
        }
    }

    // View card details
    async viewCard(cardId) {
        try {
            const response = await fetch(`../../api/id-cards/card.php?id=${cardId}`);
            const data = await response.json();
            
            if (data.success) {
                this.showCardPreview(data.data);
            } else {
                showError('Error loading card details: ' + data.message);
            }
        } catch (error) {
            console.error('Error loading card:', error);
            showError('Failed to load card details');
        }
    }

    // Show card preview modal
    showCardPreview(card) {
        const previewContent = document.getElementById('cardPreviewContent');
        const member = this.members.find(m => m.id === card.member_id);
        
        previewContent.innerHTML = `
            <div class="id-card-preview">
                <div class="id-card-header">
                    <h5 class="mb-1">KSO CHANDIGARH</h5>
                    <small>Membership Card</small>
                </div>
                <div class="row">
                    <div class="col-3">
                        <div class="id-card-photo">
                            <i class="bi bi-person fs-1"></i>
                        </div>
                        <div class="qr-code-placeholder">
                            <i class="bi bi-qr-code fs-3"></i>
                        </div>
                    </div>
                    <div class="col-9">
                        <h6 class="mb-1">${card.member_name}</h6>
                        <p class="mb-1"><small>KSO ID: ${card.member_kso_id}</small></p>
                        <p class="mb-1"><small>Card #: ${card.card_number}</small></p>
                        <p class="mb-1"><small>Type: ${card.card_type.replace('_', ' ').toUpperCase()}</small></p>
                        <p class="mb-1"><small>Valid: ${this.formatDate(card.issue_date)} - ${this.formatDate(card.expiry_date)}</small></p>
                        <p class="mb-0"><small>Status: ${card.status.toUpperCase()}</small></p>
                    </div>
                </div>
            </div>
            <div class="mt-3">
                <h6>Card Details</h6>
                <div class="row">
                    <div class="col-md-6">
                        <p><strong>Issue Date:</strong> ${this.formatDate(card.issue_date)}</p>
                        <p><strong>Expiry Date:</strong> ${this.formatDate(card.expiry_date)}</p>
                        <p><strong>Status:</strong> ${this.getStatusBadge(card.status)}</p>
                    </div>
                    <div class="col-md-6">
                        <p><strong>Generated By:</strong> ${card.generated_by_name || 'System'}</p>
                        <p><strong>Created:</strong> ${this.formatDateTime(card.created_at)}</p>
                        ${card.notes ? `<p><strong>Notes:</strong> ${card.notes}</p>` : ''}
                    </div>
                </div>
            </div>
        `;
        
        new bootstrap.Modal(document.getElementById('cardPreviewModal')).show();
    }

    // Print card
    printCard(cardId) {
        // This would integrate with a printing service or generate PDF
        console.log('Printing card:', cardId);
        showInfo('Print functionality will be implemented with PDF generation');
    }

    // Download card
    downloadCard(cardId) {
        // This would generate and download PDF
        console.log('Downloading card:', cardId);
        showInfo('Download functionality will be implemented with PDF generation');
    }

    // Search cards
    searchCards() {
        const searchTerm = document.getElementById('searchCards').value.trim();
        if (searchTerm) {
            this.currentFilters.search = searchTerm;
        } else {
            delete this.currentFilters.search;
        }
        this.loadIDCards(1);
    }

    // Filter cards
    filterCards() {
        const status = document.getElementById('filterStatus').value;
        const type = document.getElementById('filterType').value;
        const expiryDate = document.getElementById('filterExpiryDate').value;

        // Update filters
        if (status) this.currentFilters.status = status;
        else delete this.currentFilters.status;

        if (type) this.currentFilters.card_type = type;
        else delete this.currentFilters.card_type;

        if (expiryDate) this.currentFilters.expiry_date = expiryDate;
        else delete this.currentFilters.expiry_date;

        this.loadIDCards(1);
    }

    // Clear all filters
    clearFilters() {
        document.getElementById('searchCards').value = '';
        document.getElementById('filterStatus').value = '';
        document.getElementById('filterType').value = '';
        document.getElementById('filterExpiryDate').value = '';
        
        this.currentFilters = {};
        this.loadIDCards(1);
    }

    // Bulk generate cards
    bulkGenerate() {
        showInfo('Bulk generation feature will be implemented');
    }

    // Print selected cards
    printCards() {
        const selected = this.getSelectedCards();
        if (selected.length === 0) {
            showWarning('Please select cards to print');
            return;
        }
        showInfo(`Print functionality will be implemented for ${selected.length} cards`);
    }

    // Export cards
    exportCards() {
        showInfo('Export functionality will be implemented');
    }

    // Get selected card IDs
    getSelectedCards() {
        const checkboxes = document.querySelectorAll('#cardsTableBody input[type="checkbox"]:checked');
        return Array.from(checkboxes).map(cb => parseInt(cb.value));
    }

    // Render pagination
    renderPagination() {
        const totalPages = Math.ceil(this.totalItems / this.itemsPerPage);
        const pagination = document.getElementById('pagination');
        pagination.innerHTML = '';

        if (totalPages <= 1) return;

        // Previous button
        const prevLi = document.createElement('li');
        prevLi.className = `page-item ${this.currentPage === 1 ? 'disabled' : ''}`;
        prevLi.innerHTML = `<a class="page-link" href="#" onclick="idCardsManager.loadIDCards(${this.currentPage - 1})">Previous</a>`;
        pagination.appendChild(prevLi);

        // Page numbers
        const startPage = Math.max(1, this.currentPage - 2);
        const endPage = Math.min(totalPages, this.currentPage + 2);

        for (let i = startPage; i <= endPage; i++) {
            const li = document.createElement('li');
            li.className = `page-item ${i === this.currentPage ? 'active' : ''}`;
            li.innerHTML = `<a class="page-link" href="#" onclick="idCardsManager.loadIDCards(${i})">${i}</a>`;
            pagination.appendChild(li);
        }

        // Next button
        const nextLi = document.createElement('li');
        nextLi.className = `page-item ${this.currentPage === totalPages ? 'disabled' : ''}`;
        nextLi.innerHTML = `<a class="page-link" href="#" onclick="idCardsManager.loadIDCards(${this.currentPage + 1})">Next</a>`;
        pagination.appendChild(nextLi);
    }

    // Utility methods
    formatDate(dateString) {
        if (!dateString) return 'N/A';
        return new Date(dateString).toLocaleDateString();
    }

    formatDateTime(dateString) {
        if (!dateString) return 'N/A';
        return new Date(dateString).toLocaleString();
    }
}

// Global instance
const idCardsManager = new IDCardsManager();

// Global functions for HTML onclick events
function loadIDCards(page = 1) {
    idCardsManager.loadIDCards(page);
}

function loadStats() {
    idCardsManager.loadStats();
}

function loadMembers() {
    idCardsManager.loadMembers();
}

function generateCard() {
    idCardsManager.generateCard();
}

function searchCards() {
    idCardsManager.searchCards();
}

function filterCards() {
    idCardsManager.filterCards();
}

function clearFilters() {
    idCardsManager.clearFilters();
}

function bulkGenerate() {
    idCardsManager.bulkGenerate();
}

function printCards() {
    idCardsManager.printCards();
}

function exportCards() {
    idCardsManager.exportCards();
}

function printPreview() {
    window.print();
}

function downloadPreview() {
    showInfo('PDF download will be implemented');
}
