// KSO Chandigarh Management System - Members Page JavaScript

// Initialize API Service
const apiService = new ApiService();

class MembersPage {
    constructor() {
        this.currentFilters = {
            search: '',
            status: '',
            approval_status: '',
            membership_type: '',
            limit: 25,
            offset: 0,
            sort_by: 'created_at',
            sort_order: 'DESC'
        };
        this.isLoading = false;
        this.stats = {};
        this.currentTab = 'pending';
        this.availableAdmins = [];
        this.myAssignments = [];
        
        // Debounced search function
        this.debouncedSearch = utils.debounce(this.performSearch.bind(this), 500);
        
        this.init();
    }

    async init() {
        try {
            this.bindEvents();
            await this.loadStats();
            await this.loadMembers();
            await this.loadAvailableAdmins();
            this.setupTabs();
        } catch (error) {
            console.error('Error initializing members page:', error);
            utils.showAlert('Failed to initialize page: ' + error.message, 'error');
        }
    }

    bindEvents() {
        // Search input
        const searchInput = document.getElementById('searchInput');
        if (searchInput) {
            searchInput.addEventListener('input', (e) => {
                this.currentFilters.search = e.target.value;
                this.currentFilters.offset = 0;
                this.debouncedSearch();
            });
        }

        // Filter dropdowns
        const statusFilter = document.getElementById('statusFilter');
        if (statusFilter) {
            statusFilter.addEventListener('change', (e) => {
                this.currentFilters.status = e.target.value;
                this.currentFilters.offset = 0;
                this.loadMembers();
            });
        }

        const approvalStatusFilter = document.getElementById('approvalStatusFilter');
        if (approvalStatusFilter) {
            approvalStatusFilter.addEventListener('change', (e) => {
                this.currentFilters.approval_status = e.target.value;
                this.currentFilters.offset = 0;
                this.loadMembers();
            });
        }

        const membershipTypeFilter = document.getElementById('membershipTypeFilter');
        if (membershipTypeFilter) {
            membershipTypeFilter.addEventListener('change', (e) => {
                this.currentFilters.membership_type = e.target.value;
                this.currentFilters.offset = 0;
                this.loadMembers();
            });
        }

        // Page size selector
        const pageSizeSelect = document.getElementById('pageSizeSelect');
        if (pageSizeSelect) {
            pageSizeSelect.addEventListener('change', (e) => {
                this.currentFilters.limit = parseInt(e.target.value);
                this.currentFilters.offset = 0;
                this.loadMembers();
            });
        }

        // Refresh button
        const refreshBtn = document.getElementById('refreshBtn');
        if (refreshBtn) {
            refreshBtn.addEventListener('click', () => {
                this.refresh();
            });
        }

        // Add member form
        const addMemberForm = document.getElementById('addMemberForm');
        if (addMemberForm) {
            addMemberForm.addEventListener('submit', (e) => {
                e.preventDefault();
                this.handleAddMember();
            });
        }

        // Clear filters button
        const clearFiltersBtn = document.getElementById('clearFiltersBtn');
        if (clearFiltersBtn) {
            clearFiltersBtn.addEventListener('click', () => {
                this.clearFilters();
            });
        }
    }

    async loadStats() {
        try {
            const response = await apiService.getMembersStats();
            this.stats = response.data;
            this.updateStatsCards();
        } catch (error) {
            console.error('Error loading stats:', error);
            utils.showAlert('Failed to load statistics: ' + error.message, 'error');
        }
    }

    updateStatsCards() {
        // Update stat cards with the loaded data
        this.updateStatCard('totalMembersCard', this.stats.total_members || 0);
        this.updateStatCard('approvedMembersCard', this.stats.approved_members || 0);
        this.updateStatCard('pendingMembersCard', this.stats.pending_members || 0);
        this.updateStatCard('rejectedMembersCard', this.stats.rejected_members || 0);
        this.updateStatCard('individualMembersCard', this.stats.individual_members || 0);
        this.updateStatCard('familyMembersCard', this.stats.family_members || 0);
        this.updateStatCard('activeMembersCard', this.stats.active_members || 0);
        this.updateStatCard('todayRegistrationsCard', this.stats.today_registrations || 0);
    }

    updateStatCard(cardId, value) {
        const card = document.getElementById(cardId);
        if (card) {
            const valueElement = card.querySelector('.stat-value, .card-text h3');
            if (valueElement) {
                valueElement.textContent = value.toLocaleString();
            }
        }
    }

    async loadMembers() {
        if (this.isLoading) return;
        
        this.isLoading = true;
        utils.showLoading('membersTableContainer');

        try {
            const response = await apiService.getMembers(this.currentFilters);
            this.renderMembersTable(response.data.members);
            this.updatePagination(response.data.pagination);
            this.updateFilterInfo(response.data.filters);
            
            // Update pending count
            const pendingCount = response.data.members.filter(m => m.approval_status === 'pending').length;
            document.getElementById('pendingCount').textContent = pendingCount;
        } catch (error) {
            console.error('Error loading members:', error);
            utils.showError('membersTableContainer', 'Failed to load members: ' + error.message);
        } finally {
            this.isLoading = false;
        }
    }

    renderMembersTable(members) {
        const container = document.getElementById('membersTableContainer');
        if (!container) return;

        if (!members || members.length === 0) {
            container.innerHTML = `
                <div class="text-center py-5">
                    <i class="bi bi-person-x display-4 text-muted"></i>
                    <p class="text-muted mt-3">No members found matching the current filters.</p>
                </div>
            `;
            return;
        }

        const tableHtml = `
            <div class="table-responsive">
                <table class="table table-hover">
                    <thead class="table-light">
                        <tr>
                            <th>KSO ID</th>
                            <th>Name</th>
                            <th>Email</th>
                            <th>College</th>
                            <th>Type</th>
                            <th>Approval Status</th>
                            <th>Status</th>
                            <th>Created</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${members.map(member => this.renderMemberRow(member)).join('')}
                    </tbody>
                </table>
            </div>
        `;

        container.innerHTML = tableHtml;
        this.bindTableEvents();
    }

    renderMemberRow(member) {
        const approvalStatusClass = utils.getStatusBadgeClass(member.approval_status);
        const statusClass = utils.getStatusBadgeClass(member.status);
        
        return `
            <tr data-member-id="${member.id}">
                <td>
                    <strong>${member.kso_id}</strong>
                    ${member.dependents_count > 0 ? `<small class="text-muted d-block">${member.dependents_count} dependents</small>` : ''}
                </td>
                <td>
                    <div>
                        <strong>${utils.escapeHtml(member.full_name)}</strong>
                        ${member.nickname ? `<small class="text-muted d-block">"${utils.escapeHtml(member.nickname)}"</small>` : ''}
                    </div>
                </td>
                <td>
                    <span class="text-break">${utils.escapeHtml(member.email)}</span>
                    ${member.phone ? `<small class="text-muted d-block">${utils.escapeHtml(member.phone)}</small>` : ''}
                </td>
                <td>
                    <span>${utils.escapeHtml(member.college)}</span>
                    <small class="text-muted d-block">${utils.escapeHtml(member.course)}</small>
                </td>
                <td>
                    <span class="badge ${member.membership_type === 'Family' ? 'text-bg-info' : 'text-bg-secondary'}">
                        ${member.membership_type}
                    </span>
                </td>
                <td>
                    <span class="badge ${approvalStatusClass}">
                        ${utils.capitalizeFirst(member.approval_status)}
                    </span>
                    ${member.assigned_to_name ? `<small class="text-muted d-block">Assigned to: ${utils.escapeHtml(member.assigned_to_name)}</small>` : ''}
                </td>
                <td>
                    <span class="badge ${statusClass}">
                        ${utils.capitalizeFirst(member.status)}
                    </span>
                </td>
                <td>
                    <span title="${member.created_at}">${utils.formatDate(member.created_at)}</span>
                    <small class="text-muted d-block">${utils.getRelativeTime(member.created_at)}</small>
                </td>
                <td>
                    <div class="btn-group btn-group-sm" role="group">
                        <button type="button" class="btn btn-outline-primary" onclick="membersPage.viewMember(${member.id})" title="View Details">
                            <i class="bi bi-eye"></i>
                        </button>
                        <button type="button" class="btn btn-outline-secondary" onclick="membersPage.editMember(${member.id})" title="Edit Member">
                            <i class="bi bi-pencil"></i>
                        </button>
                        ${this.renderApprovalButtons(member)}
                        <button type="button" class="btn btn-outline-danger" onclick="membersPage.deleteMember(${member.id})" title="Delete Member">
                            <i class="bi bi-trash"></i>
                        </button>
                    </div>
                </td>
            </tr>
        `;
    }

    renderApprovalButtons(member) {
        if (member.approval_status === 'pending' || member.approval_status === 'under_review') {
            let buttons = `
                <button type="button" class="btn btn-outline-success" onclick="membersPage.approveMember(${member.id})" title="Approve Member">
                    <i class="bi bi-check"></i>
                </button>
                <button type="button" class="btn btn-outline-warning" onclick="membersPage.rejectMember(${member.id})" title="Reject Member">
                    <i class="bi bi-x"></i>
                </button>
            `;
            
            // Add assignment buttons based on assignment status
            if (member.assigned_to) {
                // If member is assigned, show unassign button
                buttons += `
                    <button type="button" class="btn btn-outline-danger" onclick="membersPage.unassignMember(${member.id})" title="Remove Assignment">
                        <i class="bi bi-person-dash"></i>
                    </button>
                `;
            } else {
                // If member is not assigned, show assign button
                buttons += `
                    <button type="button" class="btn btn-outline-info" onclick="membersPage.assignMember(${member.id})" title="Assign to Admin">
                        <i class="bi bi-person-plus"></i>
                    </button>
                `;
            }
            
            return buttons;
        }
        return '';
    }

    // Assignment Functions
    setupTabs() {
        const tabTriggerList = document.querySelectorAll('#membersTabs button[data-bs-toggle="tab"]');
        tabTriggerList.forEach(tabTrigger => {
            tabTrigger.addEventListener('shown.bs.tab', (event) => {
                const targetId = event.target.getAttribute('data-bs-target');
                if (targetId === '#pending-members') {
                    this.currentTab = 'pending';
                    this.loadMembers();
                } else if (targetId === '#my-assignments') {
                    this.currentTab = 'assignments';
                    this.loadMyAssignments();
                }
            });
        });
    }

    async loadAvailableAdmins() {
        try {
            const response = await apiService.request('members/assign?action=available_admins');
            if (response.success) {
                this.availableAdmins = response.data;
                this.populateAdminSelect();
            }
        } catch (error) {
            console.error('Error loading available admins:', error);
        }
    }

    populateAdminSelect() {
        const select = document.getElementById('assignToAdmin');
        if (!select) return;

        // Clear existing options except the first one
        select.innerHTML = '<option value="">Select Admin...</option>';
        
        this.availableAdmins.forEach(admin => {
            const option = document.createElement('option');
            option.value = admin.id;
            option.textContent = `${admin.full_name} (${admin.role})`;
            select.appendChild(option);
        });
    }

    async loadMyAssignments() {
        try {
            const response = await apiService.request('members/assign?action=my_assignments');
            if (response.success) {
                this.myAssignments = response.data;
                this.renderAssignmentsTable();
                this.updateAssignmentStats();
            }
        } catch (error) {
            console.error('Error loading assignments:', error);
            utils.showAlert('Failed to load assignments: ' + error.message, 'error');
        }
    }

    renderAssignmentsTable() {
        const tbody = document.getElementById('assignmentsTableBody');
        if (!tbody) return;

        if (this.myAssignments.length === 0) {
            tbody.innerHTML = `
                <tr>
                    <td colspan="8" class="text-center text-muted py-4">
                        <i class="bi bi-inbox-fill fs-1"></i>
                        <div class="mt-2">No assignments found</div>
                    </td>
                </tr>
            `;
            return;
        }

        tbody.innerHTML = this.myAssignments.map(assignment => this.renderAssignmentRow(assignment)).join('');
    }

    renderAssignmentRow(assignment) {
        const priorityClass = {
            'low': 'bg-secondary',
            'normal': 'bg-primary',
            'high': 'bg-warning',
            'urgent': 'bg-danger'
        };

        const isOverdue = assignment.due_date && new Date(assignment.due_date) < new Date();
        const dueDateClass = isOverdue ? 'text-danger' : 'text-muted';

        return `
            <tr data-assignment-id="${assignment.id}">
                <td>
                    <img src="../assets/img/user-placeholder.png" class="rounded-circle" width="40" height="40" alt="Photo">
                </td>
                <td>
                    <strong>${assignment.member.kso_id}</strong>
                </td>
                <td>
                    <div>
                        <strong>${utils.escapeHtml(assignment.member.full_name)}</strong>
                        ${assignment.member.nickname ? `<small class="text-muted d-block">"${utils.escapeHtml(assignment.member.nickname)}"</small>` : ''}
                    </div>
                </td>
                <td>
                    <span>${utils.escapeHtml(assignment.assigned_by_name)}</span>
                    <small class="text-muted d-block">${utils.getRelativeTime(assignment.created_at)}</small>
                </td>
                <td>
                    <span class="badge ${priorityClass[assignment.priority] || 'bg-secondary'}">
                        ${utils.capitalizeFirst(assignment.priority)}
                    </span>
                </td>
                <td>
                    ${assignment.due_date ? 
                        `<span class="${dueDateClass}">${utils.formatDate(assignment.due_date)}</span>
                         ${isOverdue ? '<br><small class="text-danger">Overdue</small>' : ''}` : 
                        '<span class="text-muted">No deadline</span>'
                    }
                </td>
                <td>
                    ${assignment.notes ? 
                        `<span class="text-truncate" title="${utils.escapeHtml(assignment.notes)}">${utils.escapeHtml(assignment.notes.substring(0, 50))}${assignment.notes.length > 50 ? '...' : ''}</span>` : 
                        '<span class="text-muted">No notes</span>'
                    }
                </td>
                <td>
                    <div class="btn-group btn-group-sm" role="group">
                        <button type="button" class="btn btn-outline-primary" onclick="membersPage.viewMember(${assignment.member.id})" title="View Details">
                            <i class="bi bi-eye"></i>
                        </button>
                        <button type="button" class="btn btn-outline-success" onclick="membersPage.approveMember(${assignment.member.id})" title="Approve">
                            <i class="bi bi-check"></i>
                        </button>
                        <button type="button" class="btn btn-outline-warning" onclick="membersPage.rejectMember(${assignment.member.id})" title="Reject">
                            <i class="bi bi-x"></i>
                        </button>
                        <button type="button" class="btn btn-outline-danger" onclick="membersPage.unassignMember(${assignment.member.id})" title="Remove Assignment">
                            <i class="bi bi-person-dash"></i>
                        </button>
                    </div>
                </td>
            </tr>
        `;
    }

    updateAssignmentStats() {
        const total = this.myAssignments.length;
        const highPriority = this.myAssignments.filter(a => a.priority === 'high' || a.priority === 'urgent').length;
        const overdue = this.myAssignments.filter(a => a.due_date && new Date(a.due_date) < new Date()).length;
        const completed = this.myAssignments.filter(a => a.status === 'completed').length;

        document.getElementById('totalAssignments').textContent = total;
        document.getElementById('highPriorityAssignments').textContent = highPriority;
        document.getElementById('overdueAssignments').textContent = overdue;
        document.getElementById('completedAssignments').textContent = completed;
        document.getElementById('assignmentsCount').textContent = total;
    }

    async assignMember(memberId) {
        try {
            const member = await this.getMemberById(memberId);
            if (!member) {
                utils.showAlert('Member not found', 'error');
                return;
            }

            // Populate member details in modal
            document.getElementById('assignMemberId').value = memberId;
            document.getElementById('memberDetails').innerHTML = `
                <div class="d-flex align-items-center">
                    <img src="../assets/img/user-placeholder.png" class="rounded-circle me-3" width="50" height="50" alt="Photo">
                    <div>
                        <h6 class="mb-0">${utils.escapeHtml(member.full_name)}</h6>
                        <small class="text-muted">${member.kso_id} • ${utils.escapeHtml(member.email)}</small>
                    </div>
                </div>
            `;

            // Show modal
            const modal = new bootstrap.Modal(document.getElementById('assignmentModal'));
            modal.show();

            // Set up form submission
            document.getElementById('submitAssignmentBtn').onclick = () => this.submitAssignment();

        } catch (error) {
            console.error('Error preparing assignment:', error);
            utils.showAlert('Failed to prepare assignment: ' + error.message, 'error');
        }
    }

    async submitAssignment() {
        try {
            const form = document.getElementById('assignmentForm');
            const formData = new FormData(form);
            
            const assignmentData = {
                member_id: parseInt(formData.get('member_id')),
                assigned_to: parseInt(formData.get('assigned_to')),
                priority: formData.get('priority'),
                due_date: formData.get('due_date') || null,
                notes: formData.get('notes') || null
            };

            const response = await apiService.request('members/assign', {
                method: 'POST',
                body: JSON.stringify(assignmentData)
            });

            if (response.success) {
                utils.showAlert('Member assigned successfully', 'success');
                
                // Close modal
                const modal = bootstrap.Modal.getInstance(document.getElementById('assignmentModal'));
                modal.hide();
                
                // Reload current view
                if (this.currentTab === 'pending') {
                    await this.loadMembers();
                } else {
                    await this.loadMyAssignments();
                }
                
                // Reset form
                form.reset();
            } else {
                utils.showAlert(response.message || 'Failed to assign member', 'error');
            }
        } catch (error) {
            console.error('Error submitting assignment:', error);
            utils.showAlert('Failed to assign member: ' + error.message, 'error');
        }
    }

    async unassignMember(memberId) {
        if (!confirm('Are you sure you want to remove this assignment?')) {
            return;
        }

        try {
            const response = await apiService.request('members/assign', {
                method: 'POST',
                body: JSON.stringify({
                    action: 'unassign',
                    member_id: memberId
                })
            });

            if (response.success) {
                utils.showAlert('Assignment removed successfully', 'success');
                
                // Reload current view
                if (this.currentTab === 'pending') {
                    await this.loadMembers();
                } else {
                    await this.loadMyAssignments();
                }
            } else {
                utils.showAlert(response.message || 'Failed to remove assignment', 'error');
            }
        } catch (error) {
            console.error('Error removing assignment:', error);
            utils.showAlert('Failed to remove assignment: ' + error.message, 'error');
        }
    }

    async getMemberById(memberId) {
        try {
            const response = await apiService.request(`members/member?id=${memberId}`);
            return response.success ? response.data : null;
        } catch (error) {
            console.error('Error fetching member:', error);
            return null;
        }
    }

    bindTableEvents() {
        // Any additional table-specific event binding can go here
    }

    updatePagination(pagination) {
        const container = document.getElementById('paginationContainer');
        if (!container || !pagination) return;

        const { total, limit, offset, current_page, total_pages } = pagination;

        if (total_pages <= 1) {
            container.innerHTML = '';
            return;
        }

        let paginationHtml = `
            <nav aria-label="Members pagination">
                <ul class="pagination justify-content-center">
                    <li class="page-item ${current_page === 1 ? 'disabled' : ''}">
                        <button class="page-link" onclick="membersPage.goToPage(${current_page - 1})" ${current_page === 1 ? 'disabled' : ''}>
                            <i class="bi bi-chevron-left"></i> Previous
                        </button>
                    </li>
        `;

        // Generate page numbers
        const maxVisiblePages = 5;
        let startPage = Math.max(1, current_page - Math.floor(maxVisiblePages / 2));
        let endPage = Math.min(total_pages, startPage + maxVisiblePages - 1);

        if (endPage - startPage + 1 < maxVisiblePages) {
            startPage = Math.max(1, endPage - maxVisiblePages + 1);
        }

        for (let i = startPage; i <= endPage; i++) {
            paginationHtml += `
                <li class="page-item ${i === current_page ? 'active' : ''}">
                    <button class="page-link" onclick="membersPage.goToPage(${i})">${i}</button>
                </li>
            `;
        }

        paginationHtml += `
                    <li class="page-item ${current_page === total_pages ? 'disabled' : ''}">
                        <button class="page-link" onclick="membersPage.goToPage(${current_page + 1})" ${current_page === total_pages ? 'disabled' : ''}>
                            Next <i class="bi bi-chevron-right"></i>
                        </button>
                    </li>
                </ul>
            </nav>
        `;

        // Add pagination info
        const startRecord = offset + 1;
        const endRecord = Math.min(offset + limit, total);
        paginationHtml += `
            <div class="text-center text-muted mt-2">
                Showing ${startRecord} to ${endRecord} of ${total} members
            </div>
        `;

        container.innerHTML = paginationHtml;
    }

    updateFilterInfo(filters) {
        const container = document.getElementById('filterInfo');
        if (!container) return;

        const activeFilters = [];
        if (filters.search) activeFilters.push(`Search: "${filters.search}"`);
        if (filters.status) activeFilters.push(`Status: ${filters.status}`);
        if (filters.approval_status) activeFilters.push(`Approval: ${filters.approval_status}`);
        if (filters.membership_type) activeFilters.push(`Type: ${filters.membership_type}`);

        if (activeFilters.length > 0) {
            container.innerHTML = `
                <div class="alert alert-info alert-sm">
                    <i class="bi bi-filter me-2"></i>
                    Active filters: ${activeFilters.join(', ')}
                    <button type="button" class="btn btn-link btn-sm p-0 ms-2" onclick="membersPage.clearFilters()">
                        Clear all
                    </button>
                </div>
            `;
        } else {
            container.innerHTML = '';
        }
    }

    performSearch() {
        this.loadMembers();
    }

    goToPage(page) {
        if (page < 1) return;
        this.currentFilters.offset = (page - 1) * this.currentFilters.limit;
        this.loadMembers();
    }

    clearFilters() {
        // Reset filters
        this.currentFilters = {
            search: '',
            status: '',
            approval_status: '',
            membership_type: '',
            limit: this.currentFilters.limit,
            offset: 0,
            sort_by: 'created_at',
            sort_order: 'DESC'
        };

        // Reset form fields
        const searchInput = document.getElementById('searchInput');
        if (searchInput) searchInput.value = '';

        const statusFilter = document.getElementById('statusFilter');
        if (statusFilter) statusFilter.value = '';

        const approvalStatusFilter = document.getElementById('approvalStatusFilter');
        if (approvalStatusFilter) approvalStatusFilter.value = '';

        const membershipTypeFilter = document.getElementById('membershipTypeFilter');
        if (membershipTypeFilter) membershipTypeFilter.value = '';

        this.loadMembers();
    }

    async refresh() {
        await this.loadStats();
        await this.loadMembers();
        utils.showAlert('Data refreshed successfully', 'success');
    }

    async handleAddMember() {
        const form = document.getElementById('addMemberForm');
        if (!form) return;

        // Validate form
        const formErrors = utils.validateForm(form, {
            first_name: { required: true, label: 'First Name' },
            last_name: { required: true, label: 'Last Name' },
            email: { required: true, email: true, label: 'Email' },
            college: { required: true, label: 'College' },
            course: { required: true, label: 'Course' },
            membership_type: { required: true, label: 'Membership Type' }
        });

        if (formErrors.length > 0) {
            utils.showAlert(formErrors.join('<br>'), 'error');
            return;
        }

        try {
            const formData = new FormData(form);
            const memberData = Object.fromEntries(formData.entries());

            // Remove empty fields
            Object.keys(memberData).forEach(key => {
                if (memberData[key] === '') {
                    delete memberData[key];
                }
            });

            const response = await apiService.createMember(memberData);
            
            utils.showAlert(
                `Member created successfully! KSO ID: ${response.data.kso_id}. Temporary password: ${response.data.temp_password}`,
                'success'
            );

            // Reset form and close modal
            form.reset();
            const modal = bootstrap.Modal.getInstance(document.getElementById('addMemberModal'));
            if (modal) modal.hide();

            // Refresh data
            await this.refresh();

        } catch (error) {
            console.error('Error creating member:', error);
            utils.showAlert('Failed to create member: ' + error.message, 'error');
        }
    }

    async viewMember(memberId) {
        try {
            const response = await apiService.getMember(memberId);
            const member = response.data;
            
            // Create and show member details modal
            this.showMemberDetailsModal(member);

        } catch (error) {
            console.error('Error loading member:', error);
            utils.showAlert('Failed to load member details: ' + error.message, 'error');
        }
    }

    showMemberDetailsModal(member) {
        // Create modal HTML dynamically
        const modalHtml = `
            <div class="modal fade" id="memberDetailsModal" tabindex="-1" aria-labelledby="memberDetailsModalLabel" aria-hidden="true">
                <div class="modal-dialog modal-lg">
                    <div class="modal-content">
                        <div class="modal-header">
                            <h5 class="modal-title" id="memberDetailsModalLabel">Member Details - ${member.kso_id}</h5>
                            <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                        </div>
                        <div class="modal-body">
                            ${this.renderMemberDetails(member)}
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                            <button type="button" class="btn btn-primary" onclick="membersPage.editMember(${member.id})">Edit Member</button>
                        </div>
                    </div>
                </div>
            </div>
        `;

        // Remove existing modal if any
        const existingModal = document.getElementById('memberDetailsModal');
        if (existingModal) {
            existingModal.remove();
        }

        // Add new modal to page
        document.body.insertAdjacentHTML('beforeend', modalHtml);
        
        // Show modal
        const modal = new bootstrap.Modal(document.getElementById('memberDetailsModal'));
        modal.show();
    }

    renderMemberDetails(member) {
        return `
            <div class="row">
                <div class="col-md-6">
                    <h6>Personal Information</h6>
                    <table class="table table-sm">
                        <tr><td><strong>KSO ID:</strong></td><td>${member.kso_id}</td></tr>
                        <tr><td><strong>Name:</strong></td><td>${utils.escapeHtml(member.full_name)}${member.nickname ? ` (${utils.escapeHtml(member.nickname)})` : ''}</td></tr>
                        <tr><td><strong>Email:</strong></td><td>${utils.escapeHtml(member.email)}</td></tr>
                        <tr><td><strong>Phone:</strong></td><td>${member.phone || 'N/A'}</td></tr>
                        <tr><td><strong>Date of Birth:</strong></td><td>${member.date_of_birth_formatted || 'N/A'}${member.age ? ` (${member.age} years)` : ''}</td></tr>
                        <tr><td><strong>Gender:</strong></td><td>${member.gender || 'N/A'}</td></tr>
                        <tr><td><strong>Blood Group:</strong></td><td>${member.blood_group || 'N/A'}</td></tr>
                    </table>
                </div>
                <div class="col-md-6">
                    <h6>Academic Information</h6>
                    <table class="table table-sm">
                        <tr><td><strong>College:</strong></td><td>${utils.escapeHtml(member.college)}</td></tr>
                        <tr><td><strong>Course:</strong></td><td>${utils.escapeHtml(member.course)}</td></tr>
                        <tr><td><strong>Year of Study:</strong></td><td>${member.year_of_study || 'N/A'}</td></tr>
                        <tr><td><strong>Session:</strong></td><td>${member.session_from_year}-${member.session_to_year}</td></tr>
                    </table>
                    
                    <h6>Membership Information</h6>
                    <table class="table table-sm">
                        <tr><td><strong>Type:</strong></td><td>${member.membership_type}</td></tr>
                        <tr><td><strong>Status:</strong></td><td><span class="badge ${utils.getStatusBadgeClass(member.status)}">${utils.capitalizeFirst(member.status)}</span></td></tr>
                        <tr><td><strong>Approval:</strong></td><td><span class="badge ${utils.getStatusBadgeClass(member.approval_status)}">${utils.capitalizeFirst(member.approval_status)}</span></td></tr>
                        <tr><td><strong>Expiry:</strong></td><td>${member.membership_expiry_formatted || 'N/A'}</td></tr>
                    </table>
                </div>
            </div>
            
            ${member.address ? `
                <div class="row mt-3">
                    <div class="col-12">
                        <h6>Address</h6>
                        <p>${utils.escapeHtml(member.address)}</p>
                    </div>
                </div>
            ` : ''}
            
            ${member.emergency_contact_name || member.emergency_contact ? `
                <div class="row mt-3">
                    <div class="col-12">
                        <h6>Emergency Contact</h6>
                        <p><strong>${utils.escapeHtml(member.emergency_contact_name || 'N/A')}</strong><br>
                        ${member.emergency_contact || 'N/A'}</p>
                    </div>
                </div>
            ` : ''}
            
            ${member.dependents && member.dependents.length > 0 ? `
                <div class="row mt-3">
                    <div class="col-12">
                        <h6>Family Members</h6>
                        <div class="table-responsive">
                            <table class="table table-sm">
                                <thead>
                                    <tr>
                                        <th>KSO ID</th>
                                        <th>Name</th>
                                        <th>Relationship</th>
                                        <th>Age</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    ${member.dependents.map(dep => `
                                        <tr>
                                            <td>${dep.kso_id}</td>
                                            <td>${utils.escapeHtml(dep.first_name)} ${utils.escapeHtml(dep.last_name)}</td>
                                            <td>${dep.relationship}</td>
                                            <td>${dep.age || 'N/A'}</td>
                                        </tr>
                                    `).join('')}
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            ` : ''}
            
            ${member.notes ? `
                <div class="row mt-3">
                    <div class="col-12">
                        <h6>Notes</h6>
                        <p>${utils.escapeHtml(member.notes)}</p>
                    </div>
                </div>
            ` : ''}
            
            <div class="row mt-3">
                <div class="col-12">
                    <h6>System Information</h6>
                    <table class="table table-sm">
                        <tr><td><strong>Created:</strong></td><td>${member.created_at_formatted}</td></tr>
                        <tr><td><strong>Updated:</strong></td><td>${member.updated_at_formatted}</td></tr>
                        ${member.approved_at_formatted ? `<tr><td><strong>Approved:</strong></td><td>${member.approved_at_formatted}</td></tr>` : ''}
                        ${member.approver_name ? `<tr><td><strong>Approved by:</strong></td><td>${utils.escapeHtml(member.approver_name)}</td></tr>` : ''}
                        ${member.assigned_to_name ? `<tr><td><strong>Assigned to:</strong></td><td>${utils.escapeHtml(member.assigned_to_name)}</td></tr>` : ''}
                    </table>
                </div>
            </div>
        `;
    }

    editMember(memberId) {
        // TODO: Implement edit member functionality
        utils.showAlert('Edit member functionality will be implemented soon', 'info');
    }

    async approveMember(memberId) {
        const notes = prompt('Enter approval notes (optional):');
        const upiTxId = prompt('Enter UPI Transaction ID (optional):');
        
        if (notes === null) return; // User cancelled

        try {
            await apiService.approveMember(memberId, notes || '', upiTxId || '');
            utils.showAlert('Member approved successfully', 'success');
            await this.refresh();
        } catch (error) {
            console.error('Error approving member:', error);
            utils.showAlert('Failed to approve member: ' + error.message, 'error');
        }
    }

    async rejectMember(memberId) {
        const reason = prompt('Enter rejection reason:');
        if (!reason || reason.trim() === '') {
            utils.showAlert('Rejection reason is required', 'warning');
            return;
        }

        const notes = prompt('Enter additional notes (optional):');
        if (notes === null) return; // User cancelled

        try {
            await apiService.rejectMember(memberId, reason.trim(), notes || '');
            utils.showAlert('Member rejected successfully', 'success');
            await this.refresh();
        } catch (error) {
            console.error('Error rejecting member:', error);
            utils.showAlert('Failed to reject member: ' + error.message, 'error');
        }
    }

    async deleteMember(memberId) {
        const confirmed = await utils.confirmAction(
            'Are you sure you want to delete this member? This action cannot be undone.',
            'Delete Member'
        );

        if (!confirmed) return;

        try {
            await apiService.deleteMember(memberId);
            utils.showAlert('Member deleted successfully', 'success');
            await this.refresh();
        } catch (error) {
            console.error('Error deleting member:', error);
            utils.showAlert('Failed to delete member: ' + error.message, 'error');
        }
    }
}

// Initialize the members page when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
    window.membersPage = new MembersPage();
});
