// Membership Fees Management JavaScript
class MembershipFeesManager {
    constructor() {
        this.currentPage = 1;
        this.itemsPerPage = 10;
        this.payments = [];
        this.filteredPayments = [];
        this.selectedPayments = [];
        this.members = [];
        this.feeSettings = {};
    }

    // Load payments from API
    async loadPayments() {
        try {
            const response = await apiService.get('/api/membership-fees/index.php');
            if (response.success) {
                this.payments = response.data;
                this.filteredPayments = [...this.payments];
                this.renderPayments();
            } else {
                showToast('error', 'Failed to load payments');
            }
        } catch (error) {
            console.error('Error loading payments:', error);
            showToast('error', 'Error loading payments');
        }
    }

    // Load statistics
    async loadStats() {
        try {
            const response = await apiService.get('/api/membership-fees/stats.php');
            if (response.success) {
                const stats = response.data;
                document.getElementById('totalCollected').textContent = this.formatCurrency(stats.total_collected || 0);
                document.getElementById('paidMembers').textContent = stats.paid_members || 0;
                document.getElementById('pendingMembers').textContent = stats.pending_members || 0;
                document.getElementById('overdueMembers').textContent = stats.overdue_members || 0;
                
                // Update collection progress
                document.getElementById('expectedTotal').textContent = this.formatCurrency(stats.expected_total || 0);
                document.getElementById('collectedTotal').textContent = this.formatCurrency(stats.collected_total || 0);
                document.getElementById('pendingTotal').textContent = this.formatCurrency(stats.pending_total || 0);
                
                const collectionPercentage = stats.expected_total > 0 ? 
                    Math.round((stats.collected_total / stats.expected_total) * 100) : 0;
                document.getElementById('collectionProgress').style.width = collectionPercentage + '%';
                document.getElementById('collectionPercentage').textContent = collectionPercentage + '%';
            }
        } catch (error) {
            console.error('Error loading stats:', error);
        }
    }

    // Load fee settings
    async loadFeeSettings() {
        try {
            const response = await apiService.get('/api/membership-fees/settings.php');
            if (response.success) {
                this.feeSettings = response.data;
                this.updateFeeSettingsDisplay();
                this.populateFeeSettingsForm();
            }
        } catch (error) {
            console.error('Error loading fee settings:', error);
        }
    }

    // Update fee settings display
    updateFeeSettingsDisplay() {
        if (this.feeSettings) {
            document.getElementById('individualFee').textContent = this.formatCurrency(this.feeSettings.individual_fee || 500);
            document.getElementById('familyFee').textContent = this.formatCurrency(this.feeSettings.family_fee || 800);
            document.getElementById('currentAcademicYear').textContent = this.feeSettings.academic_year || '2024-25';
            document.getElementById('feesDueDate').textContent = this.formatDate(this.feeSettings.due_date) || 'March 31, 2025';
            document.getElementById('lateFee').textContent = this.feeSettings.late_fee || 100;
        }
    }

    // Populate fee settings form
    populateFeeSettingsForm() {
        if (this.feeSettings) {
            document.getElementById('settingsIndividualFee').value = this.feeSettings.individual_fee || 500;
            document.getElementById('settingsFamilyFee').value = this.feeSettings.family_fee || 800;
            document.getElementById('settingsLateFee').value = this.feeSettings.late_fee || 100;
            document.getElementById('settingsDueDate').value = this.feeSettings.due_date || '';
            document.getElementById('settingsAcademicYear').value = this.feeSettings.academic_year || '2024-25';
        }
    }

    // Initialize member search
    initializeMemberSearch() {
        const searchInput = document.getElementById('memberSearch');
        const resultsContainer = document.getElementById('memberSearchResults');
        
        searchInput.addEventListener('input', async (e) => {
            const query = e.target.value.trim();
            
            if (query.length < 2) {
                resultsContainer.style.display = 'none';
                return;
            }
            
            try {
                const response = await apiService.get(`/api/members/search.php?q=${encodeURIComponent(query)}`);
                if (response.success) {
                    this.displayMemberSearchResults(response.data);
                }
            } catch (error) {
                console.error('Error searching members:', error);
            }
        });
        
        // Hide results when clicking outside
        document.addEventListener('click', (e) => {
            if (!searchInput.contains(e.target) && !resultsContainer.contains(e.target)) {
                resultsContainer.style.display = 'none';
            }
        });
    }

    // Display member search results
    displayMemberSearchResults(members) {
        const resultsContainer = document.getElementById('memberSearchResults');
        
        if (members.length === 0) {
            resultsContainer.innerHTML = '<div class="member-search-item text-muted">No members found</div>';
            resultsContainer.style.display = 'block';
            return;
        }
        
        resultsContainer.innerHTML = '';
        members.forEach(member => {
            const item = document.createElement('div');
            item.className = 'member-search-item';
            item.innerHTML = `
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <strong>${member.first_name} ${member.last_name}</strong>
                        <small class="text-muted d-block">${member.kso_id} • ${member.membership_type}</small>
                    </div>
                    <span class="badge badge-${this.getStatusColor(member.status)}">${member.status}</span>
                </div>
            `;
            
            item.addEventListener('click', () => {
                this.selectMember(member);
                resultsContainer.style.display = 'none';
            });
            
            resultsContainer.appendChild(item);
        });
        
        resultsContainer.style.display = 'block';
    }

    // Select member for payment
    selectMember(member) {
        document.getElementById('memberSearch').value = `${member.first_name} ${member.last_name} (${member.kso_id})`;
        document.getElementById('selectedMemberId').value = member.id;
        
        // Calculate amount due
        const feeAmount = member.membership_type === 'Family' ? 
            (this.feeSettings.family_fee || 800) : 
            (this.feeSettings.individual_fee || 500);
        
        document.getElementById('paymentAmount').value = feeAmount;
        
        // Show member info
        const memberInfo = document.getElementById('selectedMemberInfo');
        const memberInfoContent = document.getElementById('memberInfoContent');
        
        memberInfoContent.innerHTML = `
            <div class="row">
                <div class="col-md-6">
                    <p><strong>Name:</strong> ${member.first_name} ${member.last_name}</p>
                    <p><strong>KSO ID:</strong> ${member.kso_id}</p>
                    <p><strong>Type:</strong> ${member.membership_type}</p>
                </div>
                <div class="col-md-6">
                    <p><strong>Email:</strong> ${member.email}</p>
                    <p><strong>Phone:</strong> ${member.phone || 'N/A'}</p>
                    <p><strong>Status:</strong> <span class="badge badge-${this.getStatusColor(member.status)}">${member.status}</span></p>
                </div>
            </div>
        `;
        
        memberInfo.style.display = 'block';
    }

    // Get status color
    getStatusColor(status) {
        const colors = {
            active: 'success',
            pending: 'warning',
            inactive: 'secondary',
            suspended: 'danger',
            expired: 'dark'
        };
        return colors[status] || 'secondary';
    }

    // Filter payments
    filterPayments() {
        const status = document.getElementById('statusFilter').value;
        const membershipType = document.getElementById('membershipTypeFilter').value;
        const paymentMethod = document.getElementById('paymentMethodFilter').value;
        const search = document.getElementById('searchInput').value.toLowerCase();
        
        this.filteredPayments = this.payments.filter(payment => {
            const matchStatus = !status || payment.status === status;
            const matchMembershipType = !membershipType || payment.membership_type === membershipType;
            const matchPaymentMethod = !paymentMethod || payment.payment_method === paymentMethod;
            const matchSearch = !search || 
                payment.member_name.toLowerCase().includes(search) ||
                payment.kso_id.toLowerCase().includes(search) ||
                payment.transaction_reference.toLowerCase().includes(search);
            
            return matchStatus && matchMembershipType && matchPaymentMethod && matchSearch;
        });
        
        this.currentPage = 1;
        this.renderPayments();
    }

    // Render payments table
    renderPayments() {
        const tbody = document.getElementById('paymentsTableBody');
        const startIndex = (this.currentPage - 1) * this.itemsPerPage;
        const endIndex = startIndex + this.itemsPerPage;
        const pagePayments = this.filteredPayments.slice(startIndex, endIndex);
        
        tbody.innerHTML = '';
        
        if (pagePayments.length === 0) {
            tbody.innerHTML = `
                <tr>
                    <td colspan="10" class="text-center py-4">
                        <i class="bi bi-credit-card" style="font-size: 3rem; color: #6c757d;"></i>
                        <h5 class="mt-3 text-muted">No payment records found</h5>
                        <p class="text-muted">Try adjusting your search criteria or record a new payment.</p>
                    </td>
                </tr>
            `;
            return;
        }
        
        pagePayments.forEach(payment => {
            const row = document.createElement('tr');
            row.className = `fee-${payment.status}`;
            
            const reminderBadge = this.isOverdue(payment) ? 
                '<span class="badge badge-danger reminder-badge">Overdue</span>' : '';
            
            row.innerHTML = `
                <td>
                    <input type="checkbox" class="payment-checkbox" value="${payment.id}" 
                           onchange="membershipFeesManager.togglePaymentSelection(${payment.id})">
                </td>
                <td>
                    <div class="position-relative">
                        <strong>${payment.member_name}</strong>
                        <br><small class="text-muted">${payment.email}</small>
                        ${reminderBadge}
                    </div>
                </td>
                <td>
                    <span class="badge badge-info">${payment.kso_id}</span>
                </td>
                <td>
                    <span class="badge badge-secondary">${payment.membership_type}</span>
                </td>
                <td>
                    <span class="text-info font-weight-bold">
                        ${this.formatCurrency(payment.amount_due)}
                    </span>
                </td>
                <td>
                    <span class="text-success font-weight-bold">
                        ${this.formatCurrency(payment.amount_paid || 0)}
                    </span>
                </td>
                <td>
                    ${this.getPaymentStatusBadge(payment.status)}
                </td>
                <td>
                    ${payment.payment_method ? 
                        `<span class="badge payment-method-badge badge-primary">${this.formatPaymentMethod(payment.payment_method)}</span>` :
                        '<span class="text-muted">-</span>'
                    }
                </td>
                <td>
                    ${payment.payment_date ? this.formatDate(payment.payment_date) : 
                      '<span class="text-muted">Not paid</span>'}
                    <br><small class="text-muted">Due: ${this.formatDate(payment.due_date)}</small>
                </td>
                <td>
                    <div class="btn-group" role="group">
                        <button class="btn btn-sm btn-outline-info" onclick="membershipFeesManager.viewPayment(${payment.id})" title="View Details">
                            <i class="bi bi-eye"></i>
                        </button>
                        ${payment.status === 'pending' || payment.status === 'partial' ? `
                            <button class="btn btn-sm btn-outline-success" onclick="membershipFeesManager.recordPaymentForMember(${payment.member_id})" title="Record Payment">
                                <i class="bi bi-credit-card"></i>
                            </button>
                        ` : ''}
                        ${payment.status === 'paid' ? `
                            <button class="btn btn-sm btn-outline-primary" onclick="membershipFeesManager.generateReceipt(${payment.id})" title="Generate Receipt">
                                <i class="bi bi-receipt"></i>
                            </button>
                        ` : ''}
                        ${payment.status === 'pending' || payment.status === 'overdue' ? `
                            <button class="btn btn-sm btn-outline-warning" onclick="membershipFeesManager.sendReminder(${payment.member_id})" title="Send Reminder">
                                <i class="bi bi-bell"></i>
                            </button>
                        ` : ''}
                        <button class="btn btn-sm btn-outline-secondary" onclick="membershipFeesManager.viewPaymentHistory(${payment.member_id})" title="Payment History">
                            <i class="bi bi-clock-history"></i>
                        </button>
                    </div>
                </td>
            `;
            
            tbody.appendChild(row);
        });
        
        this.renderPagination();
    }

    // Get payment status badge
    getPaymentStatusBadge(status) {
        const badges = {
            paid: '<span class="badge status-badge badge-success">Paid</span>',
            pending: '<span class="badge status-badge badge-warning">Pending</span>',
            overdue: '<span class="badge status-badge badge-danger">Overdue</span>',
            partial: '<span class="badge status-badge badge-info">Partial</span>'
        };
        return badges[status] || '<span class="badge status-badge badge-secondary">Unknown</span>';
    }

    // Format payment method
    formatPaymentMethod(method) {
        const methods = {
            cash: 'Cash',
            bank_transfer: 'Bank Transfer',
            upi: 'UPI',
            cheque: 'Cheque'
        };
        return methods[method] || method;
    }

    // Check if payment is overdue
    isOverdue(payment) {
        if (payment.status === 'paid') return false;
        const dueDate = new Date(payment.due_date);
        const today = new Date();
        return today > dueDate;
    }

    // Render pagination
    renderPagination() {
        const totalPages = Math.ceil(this.filteredPayments.length / this.itemsPerPage);
        const pagination = document.getElementById('pagination');
        
        // Update showing info
        const startIndex = (this.currentPage - 1) * this.itemsPerPage + 1;
        const endIndex = Math.min(this.currentPage * this.itemsPerPage, this.filteredPayments.length);
        
        document.getElementById('showingStart').textContent = this.filteredPayments.length > 0 ? startIndex : 0;
        document.getElementById('showingEnd').textContent = endIndex;
        document.getElementById('totalRecords').textContent = this.filteredPayments.length;
        
        pagination.innerHTML = '';
        
        if (totalPages <= 1) return;
        
        // Previous button
        const prevLi = document.createElement('li');
        prevLi.className = `page-item ${this.currentPage === 1 ? 'disabled' : ''}`;
        prevLi.innerHTML = `<a class="page-link" href="#" onclick="membershipFeesManager.changePage(${this.currentPage - 1})">Previous</a>`;
        pagination.appendChild(prevLi);
        
        // Page numbers
        const startPage = Math.max(1, this.currentPage - 2);
        const endPage = Math.min(totalPages, this.currentPage + 2);
        
        for (let i = startPage; i <= endPage; i++) {
            const pageLi = document.createElement('li');
            pageLi.className = `page-item ${i === this.currentPage ? 'active' : ''}`;
            pageLi.innerHTML = `<a class="page-link" href="#" onclick="membershipFeesManager.changePage(${i})">${i}</a>`;
            pagination.appendChild(pageLi);
        }
        
        // Next button
        const nextLi = document.createElement('li');
        nextLi.className = `page-item ${this.currentPage === totalPages ? 'disabled' : ''}`;
        nextLi.innerHTML = `<a class="page-link" href="#" onclick="membershipFeesManager.changePage(${this.currentPage + 1})">Next</a>`;
        pagination.appendChild(nextLi);
    }

    // Change page
    changePage(page) {
        if (page < 1 || page > Math.ceil(this.filteredPayments.length / this.itemsPerPage)) return;
        this.currentPage = page;
        this.renderPayments();
    }

    // Save payment
    async savePayment() {
        if (!this.validatePaymentForm()) return;
        
        const data = {
            member_id: document.getElementById('selectedMemberId').value,
            amount: document.getElementById('paymentAmount').value,
            payment_method: document.getElementById('paymentMethod').value,
            payment_date: document.getElementById('paymentDate').value,
            transaction_reference: document.getElementById('transactionReference').value,
            academic_year: document.getElementById('academicYear').value,
            notes: document.getElementById('paymentNotes').value
        };
        
        try {
            const response = await apiService.post('/api/membership-fees/record-payment.php', data);
            if (response.success) {
                showToast('success', 'Payment recorded successfully');
                $('#paymentModal').modal('hide');
                await this.loadPayments();
                await this.loadStats();
                this.clearPaymentForm();
            } else {
                showToast('error', response.message || 'Failed to record payment');
            }
        } catch (error) {
            console.error('Error recording payment:', error);
            showToast('error', 'Error recording payment');
        }
    }

    // Validate payment form
    validatePaymentForm() {
        const memberId = document.getElementById('selectedMemberId').value;
        const amount = document.getElementById('paymentAmount').value;
        const paymentMethod = document.getElementById('paymentMethod').value;
        const paymentDate = document.getElementById('paymentDate').value;
        
        if (!memberId) {
            showToast('error', 'Please select a member');
            return false;
        }
        
        if (!amount || amount <= 0) {
            showToast('error', 'Please enter a valid amount');
            return false;
        }
        
        if (!paymentMethod) {
            showToast('error', 'Please select a payment method');
            return false;
        }
        
        if (!paymentDate) {
            showToast('error', 'Please select payment date');
            return false;
        }
        
        return true;
    }

    // Clear payment form
    clearPaymentForm() {
        document.getElementById('paymentForm').reset();
        document.getElementById('selectedMemberId').value = '';
        document.getElementById('selectedMemberInfo').style.display = 'none';
        document.getElementById('memberSearchResults').style.display = 'none';
    }

    // Save fee settings
    async saveFeeSettings() {
        const data = {
            individual_fee: document.getElementById('settingsIndividualFee').value,
            family_fee: document.getElementById('settingsFamilyFee').value,
            late_fee: document.getElementById('settingsLateFee').value,
            due_date: document.getElementById('settingsDueDate').value,
            academic_year: document.getElementById('settingsAcademicYear').value
        };
        
        try {
            const response = await apiService.post('/api/membership-fees/settings.php', data);
            if (response.success) {
                showToast('success', 'Fee settings updated successfully');
                $('#feeSettingsModal').modal('hide');
                await this.loadFeeSettings();
            } else {
                showToast('error', response.message || 'Failed to update settings');
            }
        } catch (error) {
            console.error('Error updating settings:', error);
            showToast('error', 'Error updating settings');
        }
    }

    // View payment details
    async viewPayment(id) {
        try {
            const response = await apiService.get(`/api/membership-fees/payment.php?id=${id}`);
            if (response.success) {
                this.showPaymentDetails(response.data);
            } else {
                showToast('error', 'Failed to load payment details');
            }
        } catch (error) {
            console.error('Error loading payment:', error);
            showToast('error', 'Error loading payment details');
        }
    }

    // Show payment details in modal
    showPaymentDetails(payment) {
        const modalBody = document.getElementById('paymentDetailBody');
        const actions = document.getElementById('paymentActions');
        
        modalBody.innerHTML = `
            <div class="row">
                <div class="col-md-8">
                    <h5>${payment.member_name} (${payment.kso_id})</h5>
                    <div class="row">
                        <div class="col-md-6">
                            <p><strong>Membership Type:</strong> ${payment.membership_type}</p>
                            <p><strong>Academic Year:</strong> ${payment.academic_year}</p>
                            <p><strong>Amount Due:</strong> <span class="text-info">${this.formatCurrency(payment.amount_due)}</span></p>
                            <p><strong>Amount Paid:</strong> <span class="text-success">${this.formatCurrency(payment.amount_paid || 0)}</span></p>
                            <p><strong>Status:</strong> ${this.getPaymentStatusBadge(payment.status)}</p>
                        </div>
                        <div class="col-md-6">
                            ${payment.payment_method ? `<p><strong>Payment Method:</strong> ${this.formatPaymentMethod(payment.payment_method)}</p>` : ''}
                            ${payment.payment_date ? `<p><strong>Payment Date:</strong> ${this.formatDate(payment.payment_date)}</p>` : ''}
                            ${payment.transaction_reference ? `<p><strong>Transaction Ref:</strong> ${payment.transaction_reference}</p>` : ''}
                            <p><strong>Due Date:</strong> ${this.formatDate(payment.due_date)}</p>
                            ${payment.late_fee_applied ? `<p><strong>Late Fee:</strong> <span class="text-warning">${this.formatCurrency(payment.late_fee_applied)}</span></p>` : ''}
                        </div>
                    </div>
                    ${payment.notes ? `
                        <div class="mt-3">
                            <strong>Notes:</strong>
                            <p class="text-muted">${payment.notes}</p>
                        </div>
                    ` : ''}
                </div>
                <div class="col-md-4">
                    <div class="text-center">
                        <h6>Payment History</h6>
                        <div class="payment-history">
                            ${payment.payment_history ? payment.payment_history.map(p => `
                                <div class="border-bottom py-2">
                                    <div class="d-flex justify-content-between">
                                        <span>${this.formatCurrency(p.amount)}</span>
                                        <span class="text-muted">${this.formatDate(p.payment_date)}</span>
                                    </div>
                                    <small class="text-muted">${this.formatPaymentMethod(p.payment_method)}</small>
                                </div>
                            `).join('') : '<p class="text-muted">No payment history</p>'}
                        </div>
                    </div>
                </div>
            </div>
        `;
        
        // Setup action buttons
        actions.innerHTML = '';
        if (payment.status === 'paid') {
            actions.innerHTML = `
                <button type="button" class="btn btn-primary" onclick="membershipFeesManager.generateReceipt(${payment.id})">
                    <i class="bi bi-receipt mr-2"></i>Generate Receipt
                </button>
            `;
        } else {
            actions.innerHTML = `
                <button type="button" class="btn btn-success" onclick="membershipFeesManager.recordPaymentForMember(${payment.member_id}); $('#paymentDetailModal').modal('hide');">
                    <i class="bi bi-credit-card mr-2"></i>Record Payment
                </button>
                <button type="button" class="btn btn-warning" onclick="membershipFeesManager.sendReminder(${payment.member_id})">
                    <i class="bi bi-bell mr-2"></i>Send Reminder
                </button>
            `;
        }
        
        $('#paymentDetailModal').modal('show');
    }

    // Record payment for specific member
    async recordPaymentForMember(memberId) {
        try {
            const response = await apiService.get(`/api/members/member.php?id=${memberId}`);
            if (response.success) {
                this.selectMember(response.data);
                $('#paymentModal').modal('show');
            }
        } catch (error) {
            console.error('Error loading member:', error);
            showToast('error', 'Error loading member details');
        }
    }

    // Generate receipt
    async generateReceipt(paymentId) {
        try {
            const response = await apiService.get(`/api/membership-fees/receipt.php?id=${paymentId}`);
            if (response.success) {
                this.showReceipt(response.data);
            } else {
                showToast('error', 'Failed to generate receipt');
            }
        } catch (error) {
            console.error('Error generating receipt:', error);
            showToast('error', 'Error generating receipt');
        }
    }

    // Show receipt in modal
    showReceipt(receiptData) {
        const receiptContent = document.getElementById('receiptContent');
        
        receiptContent.innerHTML = `
            <div class="text-center mb-4">
                <h3>KSO Chandigarh</h3>
                <h4>Payment Receipt</h4>
                <hr>
            </div>
            
            <div class="row">
                <div class="col-md-6">
                    <p><strong>Receipt No:</strong> ${receiptData.receipt_number}</p>
                    <p><strong>Date:</strong> ${this.formatDate(receiptData.payment_date)}</p>
                    <p><strong>Academic Year:</strong> ${receiptData.academic_year}</p>
                </div>
                <div class="col-md-6 text-md-right">
                    <p><strong>Member:</strong> ${receiptData.member_name}</p>
                    <p><strong>KSO ID:</strong> ${receiptData.kso_id}</p>
                    <p><strong>Membership Type:</strong> ${receiptData.membership_type}</p>
                </div>
            </div>
            
            <hr>
            
            <div class="row">
                <div class="col-md-6">
                    <p><strong>Payment Method:</strong> ${this.formatPaymentMethod(receiptData.payment_method)}</p>
                    ${receiptData.transaction_reference ? `<p><strong>Transaction Ref:</strong> ${receiptData.transaction_reference}</p>` : ''}
                </div>
                <div class="col-md-6 text-md-right">
                    <h4><strong>Amount Paid: ${this.formatCurrency(receiptData.amount_paid)}</strong></h4>
                </div>
            </div>
            
            <hr>
            
            <div class="text-center">
                <p class="text-muted">Thank you for your payment!</p>
                <small class="text-muted">This is a computer generated receipt.</small>
            </div>
        `;
        
        $('#receiptModal').modal('show');
    }

    // Send reminders
    async sendReminders() {
        const selectedMembers = this.selectedPayments.filter(id => {
            const payment = this.payments.find(p => p.id === id);
            return payment && (payment.status === 'pending' || payment.status === 'overdue');
        });
        
        if (selectedMembers.length === 0) {
            showToast('warning', 'Please select members with pending payments');
            return;
        }
        
        if (!confirm(`Send payment reminders to ${selectedMembers.length} member(s)?`)) return;
        
        try {
            const response = await apiService.post('/api/membership-fees/send-reminders.php', {
                payment_ids: selectedMembers
            });
            
            if (response.success) {
                showToast('success', `Reminders sent to ${selectedMembers.length} member(s)`);
            } else {
                showToast('error', response.message || 'Failed to send reminders');
            }
        } catch (error) {
            console.error('Error sending reminders:', error);
            showToast('error', 'Error sending reminders');
        }
    }

    // Send reminder to specific member
    async sendReminder(memberId) {
        try {
            const response = await apiService.post(`/api/membership-fees/send-reminder.php?member_id=${memberId}`);
            if (response.success) {
                showToast('success', 'Reminder sent successfully');
            } else {
                showToast('error', response.message || 'Failed to send reminder');
            }
        } catch (error) {
            console.error('Error sending reminder:', error);
            showToast('error', 'Error sending reminder');
        }
    }

    // View payment history
    async viewPaymentHistory(memberId) {
        try {
            const response = await apiService.get(`/api/membership-fees/history.php?member_id=${memberId}`);
            if (response.success) {
                // This could open a modal with payment history details
                console.log('Payment history:', response.data);
                showToast('info', 'Payment history loaded');
            }
        } catch (error) {
            console.error('Error loading payment history:', error);
            showToast('error', 'Error loading payment history');
        }
    }

    // Generate receipts for selected payments
    async generateReceipts() {
        const paidPayments = this.selectedPayments.filter(id => {
            const payment = this.payments.find(p => p.id === id);
            return payment && payment.status === 'paid';
        });
        
        if (paidPayments.length === 0) {
            showToast('warning', 'Please select paid payments to generate receipts');
            return;
        }
        
        try {
            const response = await apiService.post('/api/membership-fees/generate-receipts.php', {
                payment_ids: paidPayments
            });
            
            if (response.success) {
                // Download receipts as ZIP file
                const link = document.createElement('a');
                link.href = response.data.download_url;
                link.download = `receipts_${new Date().toISOString().split('T')[0]}.zip`;
                link.click();
                
                showToast('success', 'Receipts generated successfully');
            } else {
                showToast('error', response.message || 'Failed to generate receipts');
            }
        } catch (error) {
            console.error('Error generating receipts:', error);
            showToast('error', 'Error generating receipts');
        }
    }

    // Export report
    async exportReport() {
        try {
            const filters = {
                status: document.getElementById('statusFilter').value,
                membership_type: document.getElementById('membershipTypeFilter').value,
                payment_method: document.getElementById('paymentMethodFilter').value,
                search: document.getElementById('searchInput').value
            };
            
            const queryString = new URLSearchParams(filters).toString();
            const url = `/api/membership-fees/export.php?${queryString}`;
            
            // Create download link
            const link = document.createElement('a');
            link.href = url;
            link.download = `membership_fees_${new Date().toISOString().split('T')[0]}.csv`;
            link.click();
            
            showToast('success', 'Report export started');
        } catch (error) {
            console.error('Error exporting report:', error);
            showToast('error', 'Error exporting report');
        }
    }

    // Toggle payment selection
    togglePaymentSelection(id) {
        const checkbox = document.querySelector(`input[value="${id}"]`);
        if (checkbox.checked) {
            if (!this.selectedPayments.includes(id)) {
                this.selectedPayments.push(id);
            }
        } else {
            this.selectedPayments = this.selectedPayments.filter(paymentId => paymentId !== id);
        }
        
        this.updateSelectAllCheckbox();
    }

    // Toggle select all
    toggleSelectAll() {
        const selectAllCheckbox = document.getElementById('selectAll');
        const checkboxes = document.querySelectorAll('.payment-checkbox');
        
        checkboxes.forEach(checkbox => {
            checkbox.checked = selectAllCheckbox.checked;
            const id = parseInt(checkbox.value);
            
            if (selectAllCheckbox.checked) {
                if (!this.selectedPayments.includes(id)) {
                    this.selectedPayments.push(id);
                }
            } else {
                this.selectedPayments = this.selectedPayments.filter(paymentId => paymentId !== id);
            }
        });
    }

    // Update select all checkbox state
    updateSelectAllCheckbox() {
        const selectAllCheckbox = document.getElementById('selectAll');
        const checkboxes = document.querySelectorAll('.payment-checkbox');
        const checkedBoxes = document.querySelectorAll('.payment-checkbox:checked');
        
        selectAllCheckbox.checked = checkboxes.length > 0 && checkboxes.length === checkedBoxes.length;
        selectAllCheckbox.indeterminate = checkedBoxes.length > 0 && checkboxes.length !== checkedBoxes.length;
    }

    // Utility: Format currency
    formatCurrency(amount) {
        return '₹' + parseFloat(amount).toLocaleString('en-IN', {
            minimumFractionDigits: 2,
            maximumFractionDigits: 2
        });
    }

    // Utility: Format date
    formatDate(dateString) {
        if (!dateString) return '';
        const date = new Date(dateString);
        return date.toLocaleDateString('en-IN');
    }
}

// Global functions
function loadPayments() {
    membershipFeesManager.loadPayments();
}

function loadStats() {
    membershipFeesManager.loadStats();
}

function loadFeeSettings() {
    membershipFeesManager.loadFeeSettings();
}

function initializeMemberSearch() {
    membershipFeesManager.initializeMemberSearch();
}

function filterPayments() {
    membershipFeesManager.filterPayments();
}

function savePayment() {
    membershipFeesManager.savePayment();
}

function saveFeeSettings() {
    membershipFeesManager.saveFeeSettings();
}

// Initialize
const membershipFeesManager = new MembershipFeesManager();

// Search on enter key
document.addEventListener('DOMContentLoaded', function() {
    const searchInput = document.getElementById('searchInput');
    if (searchInput) {
        searchInput.addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                filterPayments();
            }
        });
    }
});
