// Notifications Management JavaScript

// Initialize API Service
const apiService = new ApiService();

class NotificationsManager {
    constructor() {
        this.currentPage = 1;
        this.itemsPerPage = 10;
        this.notifications = [];
        this.filteredNotifications = [];
        this.recipients = [];
        this.selectedRecipients = [];
    }

    // Load notifications from API
    async loadNotifications() {
        try {
            const response = await apiService.getNotifications();
            if (response.success) {
                this.notifications = response.data;
                this.filteredNotifications = [...this.notifications];
                this.renderNotifications();
            } else {
                showToast('error', 'Failed to load notifications');
            }
        } catch (error) {
            console.error('Error loading notifications:', error);
            showToast('error', 'Error loading notifications');
        }
    }

    // Load statistics
    async loadStats() {
        try {
            const response = await apiService.getNotificationStats();
            if (response.success) {
                const stats = response.data;
                document.getElementById('totalNotifications').textContent = stats.total_notifications || 0;
                document.getElementById('deliveredNotifications').textContent = stats.delivered_notifications || 0;
                document.getElementById('scheduledNotifications').textContent = stats.scheduled_notifications || 0;
                document.getElementById('failedNotifications').textContent = stats.failed_notifications || 0;
            }
        } catch (error) {
            console.error('Error loading stats:', error);
        }
    }

    // Load recipients for selection
    async loadRecipients() {
        try {
            const response = await apiService.getNotificationRecipients();
            if (response.success) {
                this.recipients = response.data;
                this.populateRecipientOptions();
            }
        } catch (error) {
            console.error('Error loading recipients:', error);
        }
    }

    // Populate recipient options
    populateRecipientOptions() {
        const select = document.getElementById('specificRecipients');
        select.innerHTML = '';
        
        // Group by type
        const members = this.recipients.filter(r => r.type === 'member');
        const admins = this.recipients.filter(r => r.type === 'admin');
        
        if (members.length > 0) {
            const memberGroup = document.createElement('optgroup');
            memberGroup.label = 'Members';
            members.forEach(member => {
                const option = document.createElement('option');
                option.value = `member_${member.id}`;
                option.textContent = `${member.first_name} ${member.last_name} (${member.email})`;
                memberGroup.appendChild(option);
            });
            select.appendChild(memberGroup);
        }
        
        if (admins.length > 0) {
            const adminGroup = document.createElement('optgroup');
            adminGroup.label = 'Admins';
            admins.forEach(admin => {
                const option = document.createElement('option');
                option.value = `admin_${admin.id}`;
                option.textContent = `${admin.first_name} ${admin.last_name} (${admin.email})`;
                adminGroup.appendChild(option);
            });
            select.appendChild(adminGroup);
        }
    }

    // Filter notifications
    filterNotifications() {
        const status = document.getElementById('statusFilter').value;
        const type = document.getElementById('typeFilter').value;
        const priority = document.getElementById('priorityFilter').value;
        const search = document.getElementById('searchInput').value.toLowerCase();
        
        this.filteredNotifications = this.notifications.filter(notification => {
            const matchStatus = !status || notification.status === status;
            const matchType = !type || notification.type === type;
            const matchPriority = !priority || notification.priority === priority;
            const matchSearch = !search || 
                notification.title.toLowerCase().includes(search) ||
                notification.message.toLowerCase().includes(search);
            
            return matchStatus && matchType && matchPriority && matchSearch;
        });
        
        this.currentPage = 1;
        this.renderNotifications();
    }

    // Render notifications
    renderNotifications() {
        const container = document.getElementById('notificationsList');
        const startIndex = (this.currentPage - 1) * this.itemsPerPage;
        const endIndex = startIndex + this.itemsPerPage;
        const pageNotifications = this.filteredNotifications.slice(startIndex, endIndex);
        
        container.innerHTML = '';
        
        if (pageNotifications.length === 0) {
            container.innerHTML = `
                <div class="text-center py-4">
                    <i class="bi bi-bell-slash" style="font-size: 3rem; color: #6c757d;"></i>
                    <h5 class="mt-3 text-muted">No notifications found</h5>
                    <p class="text-muted">Try adjusting your search criteria or create a new notification.</p>
                </div>
            `;
            return;
        }
        
        pageNotifications.forEach(notification => {
            const card = this.createNotificationCard(notification);
            container.appendChild(card);
        });
        
        this.renderPagination();
    }

    // Create notification card
    createNotificationCard(notification) {
        const card = document.createElement('div');
        card.className = `card notification-card notification-${notification.type} mb-3`;
        
        const statusIcon = this.getStatusIcon(notification.status);
        const typeIcon = this.getTypeIcon(notification.type);
        const priorityBadge = this.getPriorityBadge(notification.priority);
        const statusBadge = this.getStatusBadge(notification.status);
        
        card.innerHTML = `
            <div class="card-body">
                <div class="row">
                    <div class="col-md-8">
                        <div class="d-flex align-items-center mb-2">
                            ${typeIcon}
                            <h5 class="card-title mb-0 ml-2">${notification.title}</h5>
                            <div class="ml-auto">
                                ${priorityBadge}
                                ${statusBadge}
                            </div>
                        </div>
                        <p class="card-text">${this.truncateText(notification.message, 150)}</p>
                        <div class="d-flex align-items-center text-muted small">
                            <i class="bi bi-person mr-1"></i>
                            <span class="recipient-count">${notification.recipient_count || 0} recipients</span>
                            <span class="mx-2">•</span>
                            <i class="bi bi-clock mr-1"></i>
                            <span>${this.getTimeInfo(notification)}</span>
                        </div>
                    </div>
                    <div class="col-md-4 text-right">
                        <div class="btn-group" role="group">
                            <button class="btn btn-sm btn-outline-info" onclick="notificationsManager.viewNotification(${notification.id})" title="View Details">
                                <i class="bi bi-eye"></i>
                            </button>
                            ${notification.status === 'draft' ? `
                                <button class="btn btn-sm btn-outline-warning" onclick="notificationsManager.editNotification(${notification.id})" title="Edit">
                                    <i class="bi bi-pencil"></i>
                                </button>
                            ` : ''}
                            ${notification.status === 'draft' || notification.status === 'scheduled' ? `
                                <button class="btn btn-sm btn-outline-primary" onclick="notificationsManager.sendNotificationNow(${notification.id})" title="Send Now">
                                    <i class="bi bi-send"></i>
                                </button>
                            ` : ''}
                            ${notification.status !== 'sent' ? `
                                <button class="btn btn-sm btn-outline-danger" onclick="notificationsManager.deleteNotificationConfirm(${notification.id})" title="Delete">
                                    <i class="bi bi-trash"></i>
                                </button>
                            ` : ''}
                        </div>
                        ${notification.status === 'sent' && notification.delivery_stats ? `
                            <div class="mt-2 small text-muted">
                                <div>Delivered: ${notification.delivery_stats.delivered || 0}</div>
                                <div>Failed: ${notification.delivery_stats.failed || 0}</div>
                                <div>Read: ${notification.delivery_stats.read || 0}</div>
                            </div>
                        ` : ''}
                    </div>
                </div>
            </div>
        `;
        
        return card;
    }

    // Get status icon
    getStatusIcon(status) {
        const icons = {
            draft: '<i class="bi bi-file-text text-secondary"></i>',
            scheduled: '<i class="bi bi-clock text-warning"></i>',
            sent: '<i class="bi bi-check-circle text-success"></i>',
            failed: '<i class="bi bi-x-circle text-danger"></i>'
        };
        return icons[status] || '<i class="bi bi-question-circle text-muted"></i>';
    }

    // Get type icon
    getTypeIcon(type) {
        const icons = {
            info: '<i class="bi bi-info-circle text-info"></i>',
            success: '<i class="bi bi-check-circle text-success"></i>',
            warning: '<i class="bi bi-exclamation-triangle text-warning"></i>',
            error: '<i class="bi bi-x-circle text-danger"></i>',
            announcement: '<i class="bi bi-megaphone text-purple"></i>'
        };
        return icons[type] || '<i class="bi bi-bell text-muted"></i>';
    }

    // Get priority badge
    getPriorityBadge(priority) {
        const badges = {
            low: '<span class="badge priority-badge priority-low">Low</span>',
            medium: '<span class="badge priority-badge priority-medium">Medium</span>',
            high: '<span class="badge priority-badge priority-high">High</span>',
            urgent: '<span class="badge priority-badge priority-urgent">Urgent</span>'
        };
        return badges[priority] || '';
    }

    // Get status badge
    getStatusBadge(status) {
        const badges = {
            draft: '<span class="badge status-badge status-draft">Draft</span>',
            scheduled: '<span class="badge status-badge status-scheduled">Scheduled</span>',
            sent: '<span class="badge status-badge status-sent">Sent</span>',
            failed: '<span class="badge status-badge status-failed">Failed</span>'
        };
        return badges[status] || '';
    }

    // Get time information
    getTimeInfo(notification) {
        if (notification.status === 'scheduled' && notification.scheduled_for) {
            return `Scheduled for ${this.formatDate(notification.scheduled_for)}`;
        } else if (notification.sent_at) {
            return `Sent ${this.formatDate(notification.sent_at)}`;
        } else {
            return `Created ${this.formatDate(notification.created_at)}`;
        }
    }

    // Truncate text
    truncateText(text, maxLength) {
        if (text.length <= maxLength) return text;
        return text.substr(0, maxLength) + '...';
    }

    // Render pagination
    renderPagination() {
        const totalPages = Math.ceil(this.filteredNotifications.length / this.itemsPerPage);
        const pagination = document.getElementById('pagination');
        
        // Update showing info
        const startIndex = (this.currentPage - 1) * this.itemsPerPage + 1;
        const endIndex = Math.min(this.currentPage * this.itemsPerPage, this.filteredNotifications.length);
        
        document.getElementById('showingStart').textContent = this.filteredNotifications.length > 0 ? startIndex : 0;
        document.getElementById('showingEnd').textContent = endIndex;
        document.getElementById('totalRecords').textContent = this.filteredNotifications.length;
        
        pagination.innerHTML = '';
        
        if (totalPages <= 1) return;
        
        // Previous button
        const prevLi = document.createElement('li');
        prevLi.className = `page-item ${this.currentPage === 1 ? 'disabled' : ''}`;
        prevLi.innerHTML = `<a class="page-link" href="#" onclick="notificationsManager.changePage(${this.currentPage - 1})">Previous</a>`;
        pagination.appendChild(prevLi);
        
        // Page numbers
        const startPage = Math.max(1, this.currentPage - 2);
        const endPage = Math.min(totalPages, this.currentPage + 2);
        
        for (let i = startPage; i <= endPage; i++) {
            const pageLi = document.createElement('li');
            pageLi.className = `page-item ${i === this.currentPage ? 'active' : ''}`;
            pageLi.innerHTML = `<a class="page-link" href="#" onclick="notificationsManager.changePage(${i})">${i}</a>`;
            pagination.appendChild(pageLi);
        }
        
        // Next button
        const nextLi = document.createElement('li');
        nextLi.className = `page-item ${this.currentPage === totalPages ? 'disabled' : ''}`;
        nextLi.innerHTML = `<a class="page-link" href="#" onclick="notificationsManager.changePage(${this.currentPage + 1})">Next</a>`;
        pagination.appendChild(nextLi);
    }

    // Change page
    changePage(page) {
        if (page < 1 || page > Math.ceil(this.filteredNotifications.length / this.itemsPerPage)) return;
        this.currentPage = page;
        this.renderNotifications();
    }

    // Toggle recipients selection
    toggleRecipients() {
        const allMembers = document.getElementById('allMembers').checked;
        const allAdmins = document.getElementById('allAdmins').checked;
        const activeMembers = document.getElementById('activeMembers').checked;
        
        this.selectedRecipients = [];
        
        if (allMembers) {
            this.selectedRecipients = this.selectedRecipients.concat(
                this.recipients.filter(r => r.type === 'member').map(r => ({...r, selected_type: 'member'}))
            );
        }
        
        if (allAdmins) {
            this.selectedRecipients = this.selectedRecipients.concat(
                this.recipients.filter(r => r.type === 'admin').map(r => ({...r, selected_type: 'admin'}))
            );
        }
        
        if (activeMembers) {
            this.selectedRecipients = this.selectedRecipients.concat(
                this.recipients.filter(r => r.type === 'member' && r.status === 'active').map(r => ({...r, selected_type: 'member'}))
            );
        }
        
        this.updateSelectedRecipientsDisplay();
    }

    // Update selected recipients display
    updateSelectedRecipientsDisplay() {
        const container = document.getElementById('selectedRecipients');
        container.innerHTML = '';
        
        if (this.selectedRecipients.length === 0) {
            container.innerHTML = '<small class="text-muted">No recipients selected</small>';
            return;
        }
        
        const uniqueRecipients = this.selectedRecipients.filter((recipient, index, self) => 
            index === self.findIndex(r => r.id === recipient.id && r.type === recipient.type)
        );
        
        uniqueRecipients.forEach(recipient => {
            const chip = document.createElement('span');
            chip.className = 'recipient-chip';
            chip.innerHTML = `
                ${recipient.first_name} ${recipient.last_name}
                <span class="remove" onclick="notificationsManager.removeRecipient('${recipient.type}_${recipient.id}')">&times;</span>
            `;
            container.appendChild(chip);
        });
        
        const countInfo = document.createElement('div');
        countInfo.className = 'mt-2 small text-muted';
        countInfo.textContent = `${uniqueRecipients.length} recipient(s) selected`;
        container.appendChild(countInfo);
    }

    // Remove recipient
    removeRecipient(recipientId) {
        const [type, id] = recipientId.split('_');
        this.selectedRecipients = this.selectedRecipients.filter(r => 
            !(r.type === type && r.id === parseInt(id))
        );
        this.updateSelectedRecipientsDisplay();
    }

    // Initialize preview functionality
    initializePreview() {
        document.getElementById('notificationTitle').addEventListener('input', this.updatePreview.bind(this));
        document.getElementById('notificationMessage').addEventListener('input', this.updatePreview.bind(this));
        document.getElementById('notificationType').addEventListener('change', this.updatePreview.bind(this));
    }

    // Update preview
    updatePreview() {
        const title = document.getElementById('notificationTitle').value || 'Notification Title';
        const message = document.getElementById('notificationMessage').value || 'Your notification message will appear here...';
        const type = document.getElementById('notificationType').value || 'info';
        
        const previewTitle = document.getElementById('previewTitle');
        const previewMessage = document.getElementById('previewMessage');
        const previewIcon = document.querySelector('#notificationPreview i');
        
        previewTitle.textContent = title;
        previewMessage.textContent = message;
        
        // Update icon based on type
        previewIcon.className = this.getTypeIcon(type).match(/class="([^"]*)">/)[1];
    }

    // Send notification
    async sendNotification() {
        if (!this.validateNotificationForm()) return;
        
        const data = this.getNotificationFormData();
        data.status = 'sent';
        data.sent_at = new Date().toISOString();
        
        try {
            const response = await apiService.post('/api/notifications/send.php', data);
            if (response.success) {
                showToast('success', 'Notification sent successfully');
                $('#composeModal').modal('hide');
                await this.loadNotifications();
                await this.loadStats();
                this.clearForm();
            } else {
                showToast('error', response.message || 'Failed to send notification');
            }
        } catch (error) {
            console.error('Error sending notification:', error);
            showToast('error', 'Error sending notification');
        }
    }

    // Schedule notification
    async scheduleNotification() {
        if (!this.validateNotificationForm()) return;
        
        const scheduledFor = document.getElementById('scheduledFor').value;
        if (!scheduledFor) {
            showToast('error', 'Please select a schedule time');
            return;
        }
        
        const data = this.getNotificationFormData();
        data.status = 'scheduled';
        data.scheduled_for = scheduledFor;
        
        try {
            const response = await apiService.post('/api/notifications/schedule.php', data);
            if (response.success) {
                showToast('success', 'Notification scheduled successfully');
                $('#composeModal').modal('hide');
                await this.loadNotifications();
                await this.loadStats();
                this.clearForm();
            } else {
                showToast('error', response.message || 'Failed to schedule notification');
            }
        } catch (error) {
            console.error('Error scheduling notification:', error);
            showToast('error', 'Error scheduling notification');
        }
    }

    // Save as draft
    async saveDraft() {
        if (!this.validateNotificationForm(false)) return;
        
        const data = this.getNotificationFormData();
        data.status = 'draft';
        
        try {
            const response = await apiService.post('/api/notifications/draft.php', data);
            if (response.success) {
                showToast('success', 'Draft saved successfully');
                $('#composeModal').modal('hide');
                await this.loadNotifications();
                this.clearForm();
            } else {
                showToast('error', response.message || 'Failed to save draft');
            }
        } catch (error) {
            console.error('Error saving draft:', error);
            showToast('error', 'Error saving draft');
        }
    }

    // Validate notification form
    validateNotificationForm(requireRecipients = true) {
        const title = document.getElementById('notificationTitle').value.trim();
        const message = document.getElementById('notificationMessage').value.trim();
        
        if (!title) {
            showToast('error', 'Please enter a notification title');
            return false;
        }
        
        if (!message) {
            showToast('error', 'Please enter a notification message');
            return false;
        }
        
        if (requireRecipients && this.selectedRecipients.length === 0) {
            showToast('error', 'Please select at least one recipient');
            return false;
        }
        
        return true;
    }

    // Get notification form data
    getNotificationFormData() {
        return {
            title: document.getElementById('notificationTitle').value.trim(),
            message: document.getElementById('notificationMessage').value.trim(),
            type: document.getElementById('notificationType').value,
            priority: document.getElementById('notificationPriority').value,
            scheduled_for: document.getElementById('scheduledFor').value || null,
            expires_at: document.getElementById('expiresAt').value || null,
            recipients: this.selectedRecipients.map(r => ({
                type: r.type,
                id: r.id
            }))
        };
    }

    // Clear form
    clearForm() {
        document.getElementById('notificationForm').reset();
        document.getElementById('notificationId').value = '';
        this.selectedRecipients = [];
        document.getElementById('allMembers').checked = false;
        document.getElementById('allAdmins').checked = false;
        document.getElementById('activeMembers').checked = false;
        this.updateSelectedRecipientsDisplay();
        this.updatePreview();
    }

    // View notification details
    async viewNotification(id) {
        try {
            const response = await apiService.get(`/api/notifications/notification.php?id=${id}`);
            if (response.success) {
                this.showNotificationDetails(response.data);
            } else {
                showToast('error', 'Failed to load notification details');
            }
        } catch (error) {
            console.error('Error loading notification:', error);
            showToast('error', 'Error loading notification details');
        }
    }

    // Show notification details in modal
    showNotificationDetails(notification) {
        const modalBody = document.getElementById('viewModalBody');
        modalBody.innerHTML = `
            <div class="row">
                <div class="col-md-8">
                    <h5>${notification.title}</h5>
                    <p class="text-muted">${notification.message}</p>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <strong>Type:</strong> ${this.getTypeIcon(notification.type)} ${notification.type}<br>
                            <strong>Priority:</strong> ${this.getPriorityBadge(notification.priority)}<br>
                            <strong>Status:</strong> ${this.getStatusBadge(notification.status)}<br>
                        </div>
                        <div class="col-md-6">
                            <strong>Created:</strong> ${this.formatDate(notification.created_at)}<br>
                            ${notification.scheduled_for ? `<strong>Scheduled For:</strong> ${this.formatDate(notification.scheduled_for)}<br>` : ''}
                            ${notification.sent_at ? `<strong>Sent:</strong> ${this.formatDate(notification.sent_at)}<br>` : ''}
                        </div>
                    </div>
                </div>
                <div class="col-md-4">
                    <h6>Recipients (${notification.recipients ? notification.recipients.length : 0})</h6>
                    <div style="max-height: 200px; overflow-y: auto;">
                        ${notification.recipients ? notification.recipients.map(r => 
                            `<div class="small text-muted">${r.name} (${r.email})</div>`
                        ).join('') : 'No recipients'}
                    </div>
                </div>
            </div>
            
            ${notification.delivery_stats ? `
                <hr>
                <div class="row">
                    <div class="col-md-3">
                        <div class="text-center">
                            <h4 class="text-info">${notification.delivery_stats.total || 0}</h4>
                            <small>Total</small>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="text-center">
                            <h4 class="text-success">${notification.delivery_stats.delivered || 0}</h4>
                            <small>Delivered</small>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="text-center">
                            <h4 class="text-warning">${notification.delivery_stats.read || 0}</h4>
                            <small>Read</small>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="text-center">
                            <h4 class="text-danger">${notification.delivery_stats.failed || 0}</h4>
                            <small>Failed</small>
                        </div>
                    </div>
                </div>
            ` : ''}
        `;
        
        $('#viewModal').modal('show');
    }

    // Edit notification
    async editNotification(id) {
        const notification = this.notifications.find(n => n.id === id);
        if (!notification) return;
        
        // Populate form with notification data
        document.getElementById('notificationId').value = notification.id;
        document.getElementById('notificationTitle').value = notification.title;
        document.getElementById('notificationMessage').value = notification.message;
        document.getElementById('notificationType').value = notification.type;
        document.getElementById('notificationPriority').value = notification.priority;
        document.getElementById('scheduledFor').value = notification.scheduled_for || '';
        document.getElementById('expiresAt').value = notification.expires_at || '';
        
        // Load recipients
        if (notification.recipients) {
            this.selectedRecipients = notification.recipients;
            this.updateSelectedRecipientsDisplay();
        }
        
        this.updatePreview();
        $('#composeModal').modal('show');
    }

    // Send notification now
    async sendNotificationNow(id) {
        if (!confirm('Are you sure you want to send this notification now?')) return;
        
        try {
            const response = await apiService.post(`/api/notifications/send-now.php?id=${id}`);
            if (response.success) {
                showToast('success', 'Notification sent successfully');
                await this.loadNotifications();
                await this.loadStats();
            } else {
                showToast('error', response.message || 'Failed to send notification');
            }
        } catch (error) {
            console.error('Error sending notification:', error);
            showToast('error', 'Error sending notification');
        }
    }

    // Delete notification confirmation
    deleteNotificationConfirm(id) {
        if (confirm('Are you sure you want to delete this notification?')) {
            this.deleteNotification(id);
        }
    }

    // Delete notification
    async deleteNotification(id) {
        try {
            const response = await apiService.delete(`/api/notifications/notification.php?id=${id}`);
            if (response.success) {
                showToast('success', 'Notification deleted successfully');
                await this.loadNotifications();
                await this.loadStats();
            } else {
                showToast('error', response.message || 'Failed to delete notification');
            }
        } catch (error) {
            console.error('Error deleting notification:', error);
            showToast('error', 'Error deleting notification');
        }
    }

    // Utility: Format date
    formatDate(dateString) {
        const date = new Date(dateString);
        return date.toLocaleDateString() + ' ' + date.toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'});
    }
}

// Global functions
function loadNotifications() {
    notificationsManager.loadNotifications();
}

function loadStats() {
    notificationsManager.loadStats();
}

function loadRecipients() {
    notificationsManager.loadRecipients();
}

function filterNotifications() {
    notificationsManager.filterNotifications();
}

function toggleRecipients() {
    notificationsManager.toggleRecipients();
}

function initializePreview() {
    notificationsManager.initializePreview();
}

function updatePreview() {
    notificationsManager.updatePreview();
}

function sendNotification() {
    notificationsManager.sendNotification();
}

function scheduleNotification() {
    notificationsManager.scheduleNotification();
}

function saveDraft() {
    notificationsManager.saveDraft();
}

// Initialize
const notificationsManager = new NotificationsManager();

// Search on enter key
document.addEventListener('DOMContentLoaded', function() {
    const searchInput = document.getElementById('searchInput');
    if (searchInput) {
        searchInput.addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                filterNotifications();
            }
        });
    }
});
