// Receipt Management JavaScript for KSO Chandigarh

class ReceiptManager {
    constructor() {
        this.currentPage = 1;
        this.itemsPerPage = 20;
        this.receipts = [];
        this.currentReceiptId = null;
        this.filters = {
            search: '',
            transaction_type: '',
            year: '',
            month: ''
        };
    }

    // Load receipt statistics
    async loadReceiptStats() {
        try {
            const response = await fetch('../../api/receipts/stats');
            const data = await response.json();
            
            if (data.success) {
                const stats = data.data;
                document.getElementById('totalReceipts').textContent = stats.total_receipts || 0;
                document.getElementById('todayReceipts').textContent = stats.today_receipts || 0;
                document.getElementById('thisMonthReceipts').textContent = stats.this_month_receipts || 0;
                
                // Format total amount
                const totalAmount = stats.total_amount || 0;
                document.getElementById('totalAmount').textContent = '₹' + this.formatNumber(totalAmount);
            } else {
                console.error('Failed to load receipt stats:', data.message);
            }
        } catch (error) {
            console.error('Error loading receipt stats:', error);
        }
    }

    // Load receipts with pagination and filters
    async loadReceipts() {
        try {
            const params = new URLSearchParams({
                limit: this.itemsPerPage,
                offset: (this.currentPage - 1) * this.itemsPerPage,
                ...this.filters
            });

            const response = await fetch(`../../api/receipts/receipts?${params}`);
            const data = await response.json();
            
            if (data.success) {
                this.receipts = data.data;
                this.renderReceipts();
                this.updatePaginationInfo();
            } else {
                showError('Failed to load receipts: ' + data.message);
            }
        } catch (error) {
            console.error('Error loading receipts:', error);
            showError('Error loading receipts');
        }
    }

    // Render receipts table
    renderReceipts() {
        const tbody = document.getElementById('receiptsTableBody');
        tbody.innerHTML = '';

        if (this.receipts.length === 0) {
            tbody.innerHTML = `
                <tr>
                    <td colspan="7" class="text-center text-muted">
                        <i class="bi bi-inbox me-2"></i>No receipts found
                    </td>
                </tr>
            `;
            return;
        }

        this.receipts.forEach(receipt => {
            const row = this.createReceiptRow(receipt);
            tbody.appendChild(row);
        });
    }

    // Create receipt table row
    createReceiptRow(receipt) {
        const row = document.createElement('tr');
        row.className = 'receipt-row';
        
        const statusClass = receipt.file_path && receipt.file_path !== 'pending_generation_' + receipt.receipt_number ? 'generated' : 'pending';
        const statusText = statusClass === 'generated' ? 'Generated' : 'Pending';
        
        row.innerHTML = `
            <td>
                <strong>${receipt.receipt_number}</strong>
                <br><small class="text-muted">#${receipt.id}</small>
            </td>
            <td>
                <strong>${receipt.first_name} ${receipt.last_name}</strong>
                <br><small class="text-muted">${receipt.kso_id}</small>
                <br><small class="text-muted">${receipt.email}</small>
            </td>
            <td>
                <span class="badge bg-primary">${this.formatTransactionType(receipt.transaction_type)}</span>
                <br><small class="text-muted">${receipt.payment_method || 'N/A'}</small>
            </td>
            <td>
                <div class="receipt-amount">₹${this.formatNumber(receipt.amount)}</div>
            </td>
            <td>
                <strong>${this.formatDate(receipt.created_at)}</strong>
                <br><small class="text-muted">${this.formatTime(receipt.created_at)}</small>
            </td>
            <td>
                <span class="receipt-status ${statusClass}">${statusText}</span>
                <br><small class="text-muted">By: ${receipt.generated_by_name || 'System'}</small>
            </td>
            <td>
                <div class="btn-group-vertical btn-group-sm">
                    <button class="btn btn-info btn-action" onclick="receiptManager.viewReceipt(${receipt.id})" title="View Receipt">
                        <i class="bi bi-eye"></i>
                    </button>
                    <button class="btn btn-primary btn-action" onclick="receiptManager.downloadReceipt(${receipt.id})" title="Download">
                        <i class="bi bi-download"></i>
                    </button>
                    <button class="btn btn-warning btn-action" onclick="receiptManager.regenerateReceipt(${receipt.id})" title="Regenerate">
                        <i class="bi bi-arrow-clockwise"></i>
                    </button>
                    <button class="btn btn-danger btn-action" onclick="receiptManager.deleteReceipt(${receipt.id})" title="Delete">
                        <i class="bi bi-trash"></i>
                    </button>
                </div>
            </td>
        `;
        
        return row;
    }

    // Search receipts
    searchReceipts() {
        this.filters.search = document.getElementById('searchReceipts').value;
        this.filters.transaction_type = document.getElementById('filterTransactionType').value;
        this.filters.year = document.getElementById('filterYear').value;
        this.currentPage = 1;
        this.loadReceipts();
    }

    // Clear filters
    clearFilters() {
        document.getElementById('searchReceipts').value = '';
        document.getElementById('filterTransactionType').value = '';
        document.getElementById('filterYear').value = '';
        this.filters = { search: '', transaction_type: '', year: '', month: '' };
        this.currentPage = 1;
        this.loadReceipts();
    }

    // View receipt
    async viewReceipt(receiptId) {
        try {
            const response = await fetch(`../../api/receipts/view?id=${receiptId}`);
            
            if (response.ok) {
                const htmlContent = await response.text();
                document.getElementById('receiptPreviewContent').innerHTML = htmlContent;
                this.currentReceiptId = receiptId;
                
                // Show modal
                new bootstrap.Modal(document.getElementById('receiptPreviewModal')).show();
            } else {
                showError('Failed to load receipt');
            }
        } catch (error) {
            console.error('Error viewing receipt:', error);
            showError('Error loading receipt');
        }
    }

    // Download receipt
    async downloadReceipt(receiptId) {
        try {
            const response = await fetch(`../../api/receipts/download?id=${receiptId}`);
            
            if (response.ok) {
                const blob = await response.blob();
                const url = window.URL.createObjectURL(blob);
                const a = document.createElement('a');
                a.href = url;
                a.download = `receipt_${receiptId}.html`;
                document.body.appendChild(a);
                a.click();
                window.URL.revokeObjectURL(url);
                document.body.removeChild(a);
                
                showSuccess('Receipt downloaded successfully');
            } else {
                showError('Failed to download receipt');
            }
        } catch (error) {
            console.error('Error downloading receipt:', error);
            showError('Error downloading receipt');
        }
    }

    // Generate receipt
    async generateReceipt() {
        const transactionId = document.getElementById('transactionSelect').value;
        const template = document.getElementById('templateSelect').value;
        
        if (!transactionId) {
            showError('Please select a transaction');
            return;
        }
        
        try {
            const response = await fetch('../../api/receipts/generate-receipt', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    transaction_id: parseInt(transactionId),
                    template: template
                })
            });
            
            const data = await response.json();
            
            if (data.success) {
                showSuccess(`Receipt generated successfully: ${data.receipt_number}`);
                bootstrap.Modal.getInstance(document.getElementById('generateReceiptModal')).hide();
                this.loadReceipts();
                this.loadReceiptStats();
            } else {
                showError('Failed to generate receipt: ' + data.error);
            }
        } catch (error) {
            console.error('Error generating receipt:', error);
            showError('Error generating receipt');
        }
    }

    // Regenerate receipt
    async regenerateReceipt(receiptId) {
        if (!confirm('Are you sure you want to regenerate this receipt? The old file will be replaced.')) {
            return;
        }
        
        try {
            const response = await fetch('../../api/receipts/regenerate-receipt', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    receipt_id: receiptId,
                    template: 'standard'
                })
            });
            
            const data = await response.json();
            
            if (data.success) {
                showSuccess('Receipt regenerated successfully');
                this.loadReceipts();
            } else {
                showError('Failed to regenerate receipt: ' + data.error);
            }
        } catch (error) {
            console.error('Error regenerating receipt:', error);
            showError('Error regenerating receipt');
        }
    }

    // Delete receipt
    async deleteReceipt(receiptId) {
        if (!confirm('Are you sure you want to delete this receipt? This action cannot be undone.')) {
            return;
        }
        
        try {
            const response = await fetch(`../../api/receipts/receipt?id=${receiptId}`, {
                method: 'DELETE'
            });
            
            const data = await response.json();
            
            if (data.success) {
                showSuccess('Receipt deleted successfully');
                this.loadReceipts();
                this.loadReceiptStats();
            } else {
                showError('Failed to delete receipt: ' + data.error);
            }
        } catch (error) {
            console.error('Error deleting receipt:', error);
            showError('Error deleting receipt');
        }
    }

    // Load transactions for receipt generation
    async loadTransactionsForReceipt() {
        try {
            // Load completed transactions without receipts
            const response = await fetch('../../api/finance/index.php?completed_without_receipts=1');
            const data = await response.json();
            
            const select = document.getElementById('transactionSelect');
            select.innerHTML = '<option value="">Select a transaction...</option>';
            
            if (data.success && data.data) {
                data.data.forEach(transaction => {
                    const option = document.createElement('option');
                    option.value = transaction.id;
                    option.textContent = `${transaction.receipt_number || 'No Receipt'} - ${transaction.first_name} ${transaction.last_name} - ₹${this.formatNumber(transaction.amount)}`;
                    option.setAttribute('data-transaction', JSON.stringify(transaction));
                    select.appendChild(option);
                });
            }
        } catch (error) {
            console.error('Error loading transactions:', error);
        }
    }

    // Bulk generate receipts
    async bulkGenerateReceipts() {
        try {
            // Get all transaction IDs from the bulk list
            const checkboxes = document.querySelectorAll('#bulkTransactionsList input[type="checkbox"]:checked');
            const transactionIds = Array.from(checkboxes).map(cb => parseInt(cb.value));
            
            if (transactionIds.length === 0) {
                showError('Please select at least one transaction');
                return;
            }
            
            const response = await fetch('../../api/receipts/bulk-generate', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    transaction_ids: transactionIds
                })
            });
            
            const data = await response.json();
            
            if (data.success) {
                const summary = data.data.summary;
                showSuccess(`Bulk generation completed: ${summary.success} successful, ${summary.failed} failed`);
                bootstrap.Modal.getInstance(document.getElementById('bulkGenerateModal')).hide();
                this.loadReceipts();
                this.loadReceiptStats();
            } else {
                showError('Failed to bulk generate receipts: ' + data.message);
            }
        } catch (error) {
            console.error('Error bulk generating receipts:', error);
            showError('Error bulk generating receipts');
        }
    }

    // Pagination methods
    previousPage() {
        if (this.currentPage > 1) {
            this.currentPage--;
            this.loadReceipts();
        }
    }

    nextPage() {
        this.currentPage++;
        this.loadReceipts();
    }

    updatePaginationInfo() {
        const startItem = (this.currentPage - 1) * this.itemsPerPage + 1;
        const endItem = Math.min(this.currentPage * this.itemsPerPage, startItem + this.receipts.length - 1);
        
        document.getElementById('paginationInfo').textContent = 
            `Showing ${startItem}-${endItem} of ${this.receipts.length} receipts`;
        
        document.getElementById('prevPage').disabled = this.currentPage === 1;
        document.getElementById('nextPage').disabled = this.receipts.length < this.itemsPerPage;
    }

    // Utility methods
    formatNumber(number) {
        return new Intl.NumberFormat('en-IN').format(number);
    }

    formatDate(dateString) {
        return new Date(dateString).toLocaleDateString('en-IN');
    }

    formatTime(dateString) {
        return new Date(dateString).toLocaleTimeString('en-IN');
    }

    formatTransactionType(type) {
        return type.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
    }

    // Populate year filter
    populateYearFilter() {
        const yearSelect = document.getElementById('filterYear');
        const currentYear = new Date().getFullYear();
        
        for (let year = currentYear; year >= currentYear - 5; year--) {
            const option = document.createElement('option');
            option.value = year;
            option.textContent = year;
            yearSelect.appendChild(option);
        }
    }

    // Refresh all data
    refreshData() {
        this.loadReceiptStats();
        this.loadReceipts();
        showInfo('Data refreshed');
    }
}

// Initialize receipt manager
const receiptManager = new ReceiptManager();

// Global functions
function loadReceiptStats() {
    receiptManager.loadReceiptStats();
}

function loadReceipts() {
    receiptManager.loadReceipts();
}

function searchReceipts() {
    receiptManager.searchReceipts();
}

function clearFilters() {
    receiptManager.clearFilters();
}

function refreshData() {
    receiptManager.refreshData();
}

function previousPage() {
    receiptManager.previousPage();
}

function nextPage() {
    receiptManager.nextPage();
}

function showGenerateReceiptModal() {
    receiptManager.loadTransactionsForReceipt();
    new bootstrap.Modal(document.getElementById('generateReceiptModal')).show();
}

function generateReceipt() {
    receiptManager.generateReceipt();
}

function showBulkGenerateModal() {
    // Load transactions for bulk generation
    receiptManager.loadTransactionsForReceipt().then(() => {
        // Populate bulk modal with transactions
        const select = document.getElementById('transactionSelect');
        const bulkList = document.getElementById('bulkTransactionsList');
        
        let html = '<div class="form-check-all mb-3"><input type="checkbox" class="form-check-input" id="selectAll" onchange="toggleSelectAll()"> <label class="form-check-label" for="selectAll"><strong>Select All</strong></label></div>';
        
        for (let i = 1; i < select.options.length; i++) {
            const option = select.options[i];
            const transaction = JSON.parse(option.getAttribute('data-transaction'));
            
            html += `
                <div class="form-check mb-2">
                    <input class="form-check-input" type="checkbox" value="${transaction.id}" id="tx_${transaction.id}">
                    <label class="form-check-label" for="tx_${transaction.id}">
                        <strong>${option.textContent}</strong><br>
                        <small class="text-muted">${transaction.description || transaction.transaction_type}</small>
                    </label>
                </div>
            `;
        }
        
        bulkList.innerHTML = html;
    });
    
    new bootstrap.Modal(document.getElementById('bulkGenerateModal')).show();
}

function toggleSelectAll() {
    const selectAll = document.getElementById('selectAll');
    const checkboxes = document.querySelectorAll('#bulkTransactionsList input[type="checkbox"]:not(#selectAll)');
    
    checkboxes.forEach(cb => {
        cb.checked = selectAll.checked;
    });
}

function bulkGenerateReceipts() {
    receiptManager.bulkGenerateReceipts();
}

function printReceipt() {
    const content = document.getElementById('receiptPreviewContent').innerHTML;
    const printWindow = window.open('', '_blank');
    printWindow.document.write(content);
    printWindow.document.close();
    printWindow.print();
}

function downloadReceipt() {
    if (receiptManager.currentReceiptId) {
        receiptManager.downloadReceipt(receiptManager.currentReceiptId);
    }
}

function populateYearFilter() {
    receiptManager.populateYearFilter();
}

// Transaction preview functionality
document.addEventListener('change', function(e) {
    if (e.target.id === 'transactionSelect') {
        const selectedOption = e.target.selectedOptions[0];
        if (selectedOption && selectedOption.getAttribute('data-transaction')) {
            const transaction = JSON.parse(selectedOption.getAttribute('data-transaction'));
            const preview = document.getElementById('transactionPreview');
            const content = document.getElementById('previewContent');
            
            content.innerHTML = `
                <div class="row">
                    <div class="col-md-6">
                        <strong>Member:</strong> ${transaction.first_name} ${transaction.last_name}<br>
                        <strong>KSO ID:</strong> ${transaction.kso_id}<br>
                        <strong>Email:</strong> ${transaction.email}
                    </div>
                    <div class="col-md-6">
                        <strong>Amount:</strong> ₹${receiptManager.formatNumber(transaction.amount)}<br>
                        <strong>Type:</strong> ${receiptManager.formatTransactionType(transaction.transaction_type)}<br>
                        <strong>Date:</strong> ${receiptManager.formatDate(transaction.payment_date)}
                    </div>
                </div>
            `;
            
            preview.style.display = 'block';
        } else {
            document.getElementById('transactionPreview').style.display = 'none';
        }
    }
});
