// Settings Management JavaScript
class SettingsManager {
    constructor() {
        this.currentSection = 'general';
        this.settings = {};
        this.originalSettings = {};
    }

    // Load all settings
    async loadSettings() {
        try {
            const response = await fetch('../../api/settings/index.php');
            const data = await response.json();
            
            if (data.success) {
                this.settings = data.data;
                this.originalSettings = JSON.parse(JSON.stringify(data.data));
                this.populateSettingsForm();
            } else {
                console.error('Failed to load settings:', data.message);
                this.loadDefaultSettings();
            }
        } catch (error) {
            console.error('Error loading settings:', error);
            this.loadDefaultSettings();
        }
    }

    // Load default settings if API fails
    loadDefaultSettings() {
        this.settings = {
            system_name: 'KSO Chandigarh Management System',
            system_version: '1.0.0',
            timezone: 'Asia/Kolkata',
            date_format: 'd/m/Y',
            currency: 'INR',
            language: 'en',
            maintenance_mode: false,
            org_name: 'Kashmiri Samaj Organization',
            org_short_name: 'KSO',
            org_description: 'Kashmiri Samaj Organization Chandigarh - Preserving Kashmiri culture and tradition',
            org_address: 'Chandigarh, India',
            org_phone: '+91-XXXXXXXXXX',
            org_email: 'info@ksochandigarh.org',
            org_website: 'https://ksochandigarh.org',
            org_established: '1980'
        };
        this.populateSettingsForm();
    }

    // Populate settings form with data
    populateSettingsForm() {
        // General Settings
        document.getElementById('systemName').value = this.settings.system_name || '';
        document.getElementById('systemVersion').value = this.settings.system_version || '1.0.0';
        document.getElementById('timezone').value = this.settings.timezone || 'Asia/Kolkata';
        document.getElementById('dateFormat').value = this.settings.date_format || 'd/m/Y';
        document.getElementById('currency').value = this.settings.currency || 'INR';
        document.getElementById('language').value = this.settings.language || 'en';
        document.getElementById('maintenanceMode').checked = this.settings.maintenance_mode || false;

        // Organization Settings
        document.getElementById('orgName').value = this.settings.org_name || '';
        document.getElementById('orgShortName').value = this.settings.org_short_name || '';
        document.getElementById('orgDescription').value = this.settings.org_description || '';
        document.getElementById('orgAddress').value = this.settings.org_address || '';
        document.getElementById('orgPhone').value = this.settings.org_phone || '';
        document.getElementById('orgEmail').value = this.settings.org_email || '';
        document.getElementById('orgWebsite').value = this.settings.org_website || '';
        document.getElementById('orgEstablished').value = this.settings.org_established || '';
    }

    // Save general settings
    async saveGeneralSettings() {
        try {
            showLoading();
            
            const data = {
                system_name: document.getElementById('systemName').value,
                timezone: document.getElementById('timezone').value,
                date_format: document.getElementById('dateFormat').value,
                currency: document.getElementById('currency').value,
                language: document.getElementById('language').value,
                maintenance_mode: document.getElementById('maintenanceMode').checked
            };

            const response = await fetch('../../api/settings/index.php', {
                method: 'PUT',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({ category: 'general', settings: data })
            });

            const result = await response.json();
            
            if (result.success) {
                showSuccess('General settings saved successfully');
                Object.assign(this.settings, data);
            } else {
                showError('Error saving settings: ' + result.message);
            }
        } catch (error) {
            console.error('Error saving general settings:', error);
            showError('Failed to save settings');
        } finally {
            hideLoading();
        }
    }

    // Save organization settings
    async saveOrganizationSettings() {
        try {
            showLoading();
            
            const data = {
                org_name: document.getElementById('orgName').value,
                org_short_name: document.getElementById('orgShortName').value,
                org_description: document.getElementById('orgDescription').value,
                org_address: document.getElementById('orgAddress').value,
                org_phone: document.getElementById('orgPhone').value,
                org_email: document.getElementById('orgEmail').value,
                org_website: document.getElementById('orgWebsite').value,
                org_established: document.getElementById('orgEstablished').value
            };

            const response = await fetch('../../api/settings/index.php', {
                method: 'PUT',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({ category: 'organization', settings: data })
            });

            const result = await response.json();
            
            if (result.success) {
                showSuccess('Organization settings saved successfully');
                Object.assign(this.settings, data);
            } else {
                showError('Error saving settings: ' + result.message);
            }
        } catch (error) {
            console.error('Error saving organization settings:', error);
            showError('Failed to save settings');
        } finally {
            hideLoading();
        }
    }

    // Load system information
    async loadSystemInfo() {
        try {
            const response = await fetch('../../api/settings/system-info.php');
            const data = await response.json();
            
            if (data.success) {
                this.displaySystemInfo(data.data);
            } else {
                this.displayDemoSystemInfo();
            }
        } catch (error) {
            console.error('Error loading system info:', error);
            this.displayDemoSystemInfo();
        }
    }

    // Display system information
    displaySystemInfo(info) {
        document.getElementById('serverOS').textContent = info.server_os || 'Unknown';
        document.getElementById('phpVersion').textContent = info.php_version || 'Unknown';
        document.getElementById('mysqlVersion').textContent = info.mysql_version || 'Unknown';
        document.getElementById('webServer').textContent = info.web_server || 'Unknown';
        document.getElementById('dbSize').textContent = info.db_size || 'Unknown';
        document.getElementById('totalTables').textContent = info.total_tables || 'Unknown';
        document.getElementById('totalRecords').textContent = info.total_records || 'Unknown';
        document.getElementById('lastBackup').textContent = info.last_backup || 'Never';

        // Update progress bars
        const diskUsage = parseInt(info.disk_usage || 0);
        const memoryUsage = parseInt(info.memory_usage || 0);
        
        document.getElementById('diskUsage').style.width = diskUsage + '%';
        document.getElementById('diskUsage').textContent = diskUsage + '%';
        
        document.getElementById('memoryUsage').style.width = memoryUsage + '%';
        document.getElementById('memoryUsage').textContent = memoryUsage + '%';
    }

    // Display demo system information
    displayDemoSystemInfo() {
        document.getElementById('serverOS').textContent = 'Linux Ubuntu 20.04';
        document.getElementById('phpVersion').textContent = '8.1.0';
        document.getElementById('mysqlVersion').textContent = '8.0.28';
        document.getElementById('webServer').textContent = 'Apache/2.4.41';
        document.getElementById('dbSize').textContent = '15.2 MB';
        document.getElementById('totalTables').textContent = '25';
        document.getElementById('totalRecords').textContent = '1,247';
        document.getElementById('lastBackup').textContent = '2 days ago';

        // Demo progress bars
        document.getElementById('diskUsage').style.width = '45%';
        document.getElementById('diskUsage').textContent = '45%';
        
        document.getElementById('memoryUsage').style.width = '62%';
        document.getElementById('memoryUsage').textContent = '62%';
    }

    // Refresh system information
    refreshSystemInfo() {
        showLoading();
        setTimeout(() => {
            this.loadSystemInfo();
            hideLoading();
            showSuccess('System information refreshed');
        }, 1000);
    }

    // Download system report
    downloadSystemReport() {
        showInfo('System report download functionality will be implemented');
    }

    // Load backup history
    async loadBackupHistory() {
        try {
            const response = await fetch('../../api/settings/backup-history.php');
            const data = await response.json();
            
            if (data.success && data.data.length > 0) {
                this.displayBackupHistory(data.data);
            } else {
                this.displayEmptyBackupHistory();
            }
        } catch (error) {
            console.error('Error loading backup history:', error);
            this.displayEmptyBackupHistory();
        }
    }

    // Display backup history
    displayBackupHistory(backups) {
        const tbody = document.getElementById('backupHistory');
        tbody.innerHTML = '';

        backups.forEach(backup => {
            const row = document.createElement('tr');
            row.innerHTML = `
                <td>${this.formatDateTime(backup.created_at)}</td>
                <td><span class="badge bg-primary">${backup.type}</span></td>
                <td>${backup.file_size}</td>
                <td>
                    <span class="badge bg-${backup.status === 'completed' ? 'success' : 'warning'}">
                        ${backup.status}
                    </span>
                </td>
                <td>
                    <button class="btn btn-sm btn-outline-primary" onclick="downloadBackup('${backup.filename}')">
                        <i class="bi bi-download"></i>
                    </button>
                    <button class="btn btn-sm btn-outline-danger" onclick="deleteBackup('${backup.id}')">
                        <i class="bi bi-trash"></i>
                    </button>
                </td>
            `;
            tbody.appendChild(row);
        });
    }

    // Display empty backup history
    displayEmptyBackupHistory() {
        document.getElementById('backupHistory').innerHTML = `
            <tr>
                <td colspan="5" class="text-center text-muted">No backups available</td>
            </tr>
        `;
    }

    // Create backup
    async createBackup() {
        try {
            showLoading();
            
            const backupType = document.querySelector('input[name="backupType"]:checked').value;
            
            const response = await fetch('../../api/settings/backup.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({ type: backupType })
            });

            const result = await response.json();
            
            if (result.success) {
                showSuccess('Backup created successfully');
                this.loadBackupHistory();
                
                // Show backup status
                document.getElementById('backupStatus').style.display = 'block';
                document.getElementById('lastBackupDate').textContent = new Date().toLocaleString();
            } else {
                showError('Error creating backup: ' + result.message);
            }
        } catch (error) {
            console.error('Error creating backup:', error);
            showError('Failed to create backup');
        } finally {
            hideLoading();
        }
    }

    // Restore backup
    async restoreBackup() {
        const fileInput = document.getElementById('backupFile');
        if (!fileInput.files[0]) {
            showWarning('Please select a backup file');
            return;
        }

        if (!confirm('Are you sure you want to restore from backup? This will overwrite current data.')) {
            return;
        }

        try {
            showLoading();
            
            const formData = new FormData();
            formData.append('backup_file', fileInput.files[0]);

            const response = await fetch('../../api/settings/restore.php', {
                method: 'POST',
                body: formData
            });

            const result = await response.json();
            
            if (result.success) {
                showSuccess('Backup restored successfully');
                setTimeout(() => {
                    window.location.reload();
                }, 2000);
            } else {
                showError('Error restoring backup: ' + result.message);
            }
        } catch (error) {
            console.error('Error restoring backup:', error);
            showError('Failed to restore backup');
        } finally {
            hideLoading();
        }
    }

    // Load system logs
    async loadLogs() {
        try {
            const response = await fetch('../../api/settings/logs.php');
            const data = await response.json();
            
            if (data.success) {
                this.displayLogs(data.data);
            } else {
                this.displayDemoLogs();
            }
        } catch (error) {
            console.error('Error loading logs:', error);
            this.displayDemoLogs();
        }
    }

    // Display logs
    displayLogs(logs) {
        const container = document.getElementById('logsContainer');
        
        if (logs.length === 0) {
            container.innerHTML = '<div class="text-center text-muted">No logs available</div>';
            return;
        }

        let logsHtml = '';
        logs.forEach(log => {
            const levelClass = this.getLogLevelClass(log.level);
            logsHtml += `
                <div class="log-entry mb-2">
                    <span class="badge ${levelClass}">${log.level}</span>
                    <span class="text-muted">[${log.timestamp}]</span>
                    <span class="text-primary">${log.category}:</span>
                    ${log.message}
                </div>
            `;
        });
        
        container.innerHTML = logsHtml;
    }

    // Display demo logs
    displayDemoLogs() {
        const demoLogs = [
            { level: 'INFO', timestamp: '2023-12-01 10:30:15', category: 'AUTH', message: 'User admin logged in successfully' },
            { level: 'INFO', timestamp: '2023-12-01 10:25:42', category: 'MEMBER', message: 'New member registration: John Doe' },
            { level: 'WARNING', timestamp: '2023-12-01 09:15:23', category: 'SYSTEM', message: 'High memory usage detected: 85%' },
            { level: 'INFO', timestamp: '2023-12-01 08:45:12', category: 'EVENT', message: 'Event "Annual Meeting" created successfully' },
            { level: 'ERROR', timestamp: '2023-11-30 18:22:05', category: 'EMAIL', message: 'Failed to send notification email to member@example.com' },
            { level: 'INFO', timestamp: '2023-11-30 17:30:18', category: 'BACKUP', message: 'Automatic backup completed successfully' },
            { level: 'INFO', timestamp: '2023-11-30 16:45:33', category: 'FINANCE', message: 'Payment received: ₹5,000 from KSO-001' }
        ];
        
        this.displayLogs(demoLogs);
    }

    // Get log level CSS class
    getLogLevelClass(level) {
        const classes = {
            'INFO': 'bg-primary',
            'WARNING': 'bg-warning',
            'ERROR': 'bg-danger',
            'CRITICAL': 'bg-dark'
        };
        return classes[level] || 'bg-secondary';
    }

    // Filter logs
    filterLogs() {
        const logType = document.getElementById('logType').value;
        const logLevel = document.getElementById('logLevel').value;
        const logDate = document.getElementById('logDate').value;

        showInfo('Log filtering functionality will be implemented');
    }

    // Refresh logs
    refreshLogs() {
        showLoading();
        setTimeout(() => {
            this.loadLogs();
            hideLoading();
            showSuccess('Logs refreshed');
        }, 1000);
    }

    // Clear logs
    clearLogs() {
        if (!confirm('Are you sure you want to clear all logs? This action cannot be undone.')) {
            return;
        }

        showInfo('Log clearing functionality will be implemented');
    }

    // Export logs
    exportLogs() {
        showInfo('Log export functionality will be implemented');
    }

    // Show settings section
    showSettingsSection(section) {
        // Hide all sections
        const sections = ['general', 'organization', 'membership', 'fees', 'notifications', 'email', 'backup', 'security', 'system', 'logs'];
        sections.forEach(s => {
            const element = document.getElementById(s + 'Settings');
            if (element) {
                element.style.display = 'none';
            }
        });

        // Show selected section
        const targetElement = document.getElementById(section + 'Settings');
        if (targetElement) {
            targetElement.style.display = 'block';
        }

        // Update navigation
        document.querySelectorAll('.list-group-item-action').forEach(item => {
            item.classList.remove('active');
        });
        document.querySelector(`a[href="#${section}"]`).classList.add('active');

        this.currentSection = section;

        // Load section-specific data
        if (section === 'system') {
            this.loadSystemInfo();
        } else if (section === 'backup') {
            this.loadBackupHistory();
        } else if (section === 'logs') {
            this.loadLogs();
        } else if (section === 'email') {
            this.loadEmailSettings();
            this.loadEmailTemplates();
            this.refreshEmailStats();
        }
    }

    // Utility methods
    formatDateTime(dateString) {
        if (!dateString) return 'N/A';
        return new Date(dateString).toLocaleString();
    }

    // Email Settings Functions
    async loadEmailSettings() {
        try {
            const response = await fetch('../../api/emails/settings');
            const data = await response.json();
            
            if (data.success) {
                const settings = {};
                data.data.forEach(setting => {
                    settings[setting.setting_key] = setting.setting_value;
                });
                
                // Populate email form
                document.getElementById('smtpHost').value = settings.smtp_host || '';
                document.getElementById('smtpPort').value = settings.smtp_port || '587';
                document.getElementById('smtpUsername').value = settings.smtp_username || '';
                document.getElementById('smtpPassword').value = settings.smtp_password || '';
                document.getElementById('smtpEncryption').value = settings.smtp_encryption || 'tls';
                document.getElementById('emailFromAddress').value = settings.email_from_address || '';
                document.getElementById('emailFromName').value = settings.email_from_name || '';
                document.getElementById('emailEnabled').checked = settings.email_enabled === '1';
                
                // Automation settings
                document.getElementById('emailAutoWelcome').checked = settings.email_auto_welcome === '1';
                document.getElementById('emailAutoApproval').checked = settings.email_auto_approval === '1';
                document.getElementById('emailAutoEvents').checked = settings.email_auto_events === '1';
                document.getElementById('emailAutoFees').checked = settings.email_auto_fees === '1';
                
                // Update status
                this.updateEmailServiceStatus(settings.email_enabled === '1');
            } else {
                showError('Failed to load email settings: ' + data.message);
            }
        } catch (error) {
            console.error('Error loading email settings:', error);
            showError('Error loading email settings');
        }
    }

    async saveEmailSettings() {
        const settings = {
            smtp_host: document.getElementById('smtpHost').value,
            smtp_port: document.getElementById('smtpPort').value,
            smtp_username: document.getElementById('smtpUsername').value,
            smtp_password: document.getElementById('smtpPassword').value,
            smtp_encryption: document.getElementById('smtpEncryption').value,
            email_from_address: document.getElementById('emailFromAddress').value,
            email_from_name: document.getElementById('emailFromName').value,
            email_enabled: document.getElementById('emailEnabled').checked ? '1' : '0',
            email_auto_welcome: document.getElementById('emailAutoWelcome').checked ? '1' : '0',
            email_auto_approval: document.getElementById('emailAutoApproval').checked ? '1' : '0',
            email_auto_events: document.getElementById('emailAutoEvents').checked ? '1' : '0',
            email_auto_fees: document.getElementById('emailAutoFees').checked ? '1' : '0'
        };
        
        try {
            const response = await fetch('../../api/emails/save-settings', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(settings)
            });
            
            const data = await response.json();
            
            if (data.success) {
                showSuccess('Email settings saved successfully');
                this.updateEmailServiceStatus(settings.email_enabled === '1');
            } else {
                showError('Failed to save email settings: ' + data.message);
            }
        } catch (error) {
            console.error('Error saving email settings:', error);
            showError('Error saving email settings');
        }
    }

    async sendTestEmail() {
        const emailAddress = document.getElementById('testEmailAddress').value;
        
        if (!emailAddress) {
            showError('Please enter a test email address');
            return;
        }
        
        if (!emailAddress.includes('@')) {
            showError('Please enter a valid email address');
            return;
        }
        
        try {
            const response = await fetch('../../api/emails/test-email', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({ email: emailAddress })
            });
            
            const data = await response.json();
            
            if (data.success) {
                showSuccess('Test email sent successfully! Check your inbox.');
            } else {
                showError('Failed to send test email: ' + data.message);
            }
        } catch (error) {
            console.error('Error sending test email:', error);
            showError('Error sending test email');
        }
    }

    async loadEmailTemplates() {
        try {
            const response = await fetch('../../api/emails/templates');
            const data = await response.json();
            
            if (data.success) {
                const tbody = document.getElementById('emailTemplatesTable');
                tbody.innerHTML = '';
                
                const templateDescriptions = {
                    'welcome': 'Welcome email for new members',
                    'approval': 'Membership approval notification',
                    'rejection': 'Membership rejection notification',
                    'event-reminder': 'Event reminder for members',
                    'fee-reminder': 'Fee payment reminder'
                };
                
                data.data.forEach(template => {
                    const row = document.createElement('tr');
                    row.innerHTML = `
                        <td>
                            <strong>${template.name}</strong>
                        </td>
                        <td>
                            <small class="text-muted">${templateDescriptions[template.name] || 'Email template'}</small>
                        </td>
                        <td>
                            <span class="badge bg-success">Available</span>
                        </td>
                        <td>
                            <button class="btn btn-sm btn-outline-primary" onclick="previewTemplate('${template.name}')">
                                <i class="bi bi-eye"></i> Preview
                           