/**
 * KSO Chandigarh Management System - Utility Functions
 * Common utility functions used across the application
 */

/**
 * Authentication check - redirects to login if not authenticated
 * This function is required by the dashboard to protect pages from unauthorized access
 */
function requireAuth() {
    // Prevent redirect loops with a cookie check
    if (document.cookie.includes('KSO_REDIRECT_PREVENTION')) {
        return false;
    }
    
    // Try to restore from localStorage first
    try {
        const storedAuth = localStorage.getItem('kso_auth_state');
        if (storedAuth) {
            const authData = JSON.parse(storedAuth);
            const timestamp = authData.timestamp || 0;
            const now = Date.now();
            const fourHoursInMs = 4 * 60 * 60 * 1000;
            
            // If we have recent auth data, consider the user authenticated
            if (now - timestamp < fourHoursInMs && authData.admin) {
                return true;
            }
        }
    } catch (e) {
        // Ignore parse errors
    }
    
    // Check if authService exists and is initialized
    if (!window.authService) {
        // Direct redirection to main login page
        const isInPagesDirectory = window.location.pathname.includes('/frontend/pages/');
        const loginPath = isInPagesDirectory ? '../../login.html' : 'login.html';
        
        // Set prevention cookie before redirecting
        document.cookie = "KSO_REDIRECT_PREVENTION=true; path=/; max-age=5";
        window.location.href = loginPath;
        return false;
    }
    
    // Check if user is authenticated
    if (!window.authService.isAuthenticated) {
        // Use the service's built-in redirect method
        window.authService.redirectToLogin();
        return false;
    }
    
    return true;
}

const utils = {
    /**
     * Debounce function to limit function calls
     */
    debounce: function(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func.apply(this, args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    },

    /**
     * Show alert messages
     */
    showAlert: function(message, type = 'info') {
        const alertContainer = document.getElementById('alertContainer') || document.body;
        const alertId = 'alert-' + Date.now();
        
        const alertTypes = {
            success: { class: 'alert-success', icon: 'bi-check-circle' },
            error: { class: 'alert-danger', icon: 'bi-exclamation-triangle' },
            warning: { class: 'alert-warning', icon: 'bi-exclamation-triangle' },
            info: { class: 'alert-info', icon: 'bi-info-circle' }
        };
        
        const alertType = alertTypes[type] || alertTypes.info;
        
        const alertDiv = document.createElement('div');
        alertDiv.id = alertId;
        alertDiv.className = `alert ${alertType.class} alert-dismissible fade show`;
        alertDiv.innerHTML = `
            <i class="${alertType.icon}"></i>
            <span>${message}</span>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        `;
        
        alertContainer.appendChild(alertDiv);
        
        // Auto-dismiss after 5 seconds
        setTimeout(() => {
            const alert = document.getElementById(alertId);
            if (alert) {
                alert.remove();
            }
        }, 5000);
    },

    /**
     * Show loading state in container
     */
    showLoading: function(containerId) {
        const container = document.getElementById(containerId);
        if (container) {
            container.innerHTML = `
                <div class="text-center py-5">
                    <div class="spinner-border text-primary" role="status">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                    <p class="mt-2 text-muted">Loading...</p>
                </div>
            `;
        }
    },

    /**
     * Show error state in container
     */
    showError: function(containerId, message) {
        const container = document.getElementById(containerId);
        if (container) {
            container.innerHTML = `
                <div class="text-center py-5">
                    <i class="bi bi-exclamation-triangle text-danger fs-1"></i>
                    <p class="mt-2 text-danger">${message}</p>
                    <button class="btn btn-outline-primary" onclick="window.location.reload()">
                        <i class="bi bi-arrow-clockwise"></i> Retry
                    </button>
                </div>
            `;
        }
    },

    /**
     * Get CSS class for status badges
     */
    getStatusBadgeClass: function(status) {
        const statusClasses = {
            'approved': 'badge-success',
            'pending': 'badge-warning',
            'rejected': 'badge-danger',
            'active': 'badge-success',
            'inactive': 'badge-secondary',
            'suspended': 'badge-danger'
        };
        return statusClasses[status?.toLowerCase()] || 'badge-secondary';
    },

    /**
     * Escape HTML to prevent XSS
     */
    escapeHtml: function(text) {
        if (!text) return '';
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    },

    /**
     * Capitalize first letter
     */
    capitalizeFirst: function(str) {
        if (!str) return '';
        return str.charAt(0).toUpperCase() + str.slice(1).toLowerCase();
    },

    /**
     * Format date
     */
    formatDate: function(dateString) {
        if (!dateString) return '';
        const date = new Date(dateString);
        return date.toLocaleDateString('en-IN', {
            year: 'numeric',
            month: 'short',
            day: 'numeric'
        });
    },

    /**
     * Get relative time
     */
    getRelativeTime: function(dateString) {
        if (!dateString) return '';
        const date = new Date(dateString);
        const now = new Date();
        const diff = now - date;
        
        const minutes = Math.floor(diff / 60000);
        const hours = Math.floor(minutes / 60);
        const days = Math.floor(hours / 24);
        
        if (days > 0) {
            return `${days} day${days > 1 ? 's' : ''} ago`;
        } else if (hours > 0) {
            return `${hours} hour${hours > 1 ? 's' : ''} ago`;
        } else if (minutes > 0) {
            return `${minutes} minute${minutes > 1 ? 's' : ''} ago`;
        } else {
            return 'Just now';
        }
    },

    /**
     * Validate form fields
     */
    validateForm: function(form, rules = {}) {
        const errors = [];
        
        // Basic validation for required fields
        const requiredFields = form.querySelectorAll('[required]');
        requiredFields.forEach(field => {
            if (!field.value.trim()) {
                const label = field.previousElementSibling?.textContent || field.name || field.id;
                errors.push(`${label} is required`);
            }
        });
        
        // Email validation
        const emailFields = form.querySelectorAll('input[type="email"]');
        emailFields.forEach(field => {
            if (field.value && !this.isValidEmail(field.value)) {
                errors.push('Please enter a valid email address');
            }
        });
        
        // Phone validation
        const phoneFields = form.querySelectorAll('input[type="tel"]');
        phoneFields.forEach(field => {
            if (field.value && !this.isValidPhone(field.value)) {
                errors.push('Please enter a valid phone number');
            }
        });
        
        return errors;
    },

    /**
     * Validate email format
     */
    isValidEmail: function(email) {
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return emailRegex.test(email);
    },

    /**
     * Validate phone format
     */
    isValidPhone: function(phone) {
        const phoneRegex = /^[\+]?[\d\s\-\(\)]{10,15}$/;
        return phoneRegex.test(phone);
    },

    /**
     * Generate unique ID
     */
    generateId: function() {
        return 'id-' + Math.random().toString(36).substr(2, 9);
    },

    /**
     * Convert form data to object
     */
    formToObject: function(form) {
        const formData = new FormData(form);
        const obj = {};
        
        for (let [key, value] of formData.entries()) {
            if (obj[key]) {
                // Handle multiple values (checkboxes, multi-select)
                if (Array.isArray(obj[key])) {
                    obj[key].push(value);
                } else {
                    obj[key] = [obj[key], value];
                }
            } else {
                obj[key] = value;
            }
        }
        
        return obj;
    },

    /**
     * Copy text to clipboard
     */
    copyToClipboard: function(text) {
        if (navigator.clipboard) {
            navigator.clipboard.writeText(text).then(() => {
                this.showAlert('Copied to clipboard', 'success');
            }).catch(() => {
                this.showAlert('Failed to copy to clipboard', 'error');
            });
        } else {
            // Fallback for older browsers
            const textArea = document.createElement('textarea');
            textArea.value = text;
            document.body.appendChild(textArea);
            textArea.select();
            try {
                document.execCommand('copy');
                this.showAlert('Copied to clipboard', 'success');
            } catch (err) {
                this.showAlert('Failed to copy to clipboard', 'error');
            }
            document.body.removeChild(textArea);
        }
    },

    /**
     * Format currency
     */
    formatCurrency: function(amount, currency = '₹') {
        if (isNaN(amount)) return currency + '0';
        return currency + parseFloat(amount).toLocaleString('en-IN');
    },

    /**
     * Truncate text
     */
    truncateText: function(text, maxLength = 100) {
        if (!text || text.length <= maxLength) return text;
        return text.substring(0, maxLength) + '...';
    }
};

// Export for use in other modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = utils;
}
