<?php
/**
 * Password Hash Analyzer - Comprehensive Password Testing
 * This script analyzes how passwords might be hashed on the hosting site
 */

echo "=== KSO Chandigarh Password Hash Analyzer ===\n\n";

$stored_hash = 'f925916e2754e5e03f75dd58a5733251';
echo "Target hash from database: " . $stored_hash . "\n\n";

// Test different hashing algorithms
function testHashingMethods($password) {
    global $stored_hash;
    
    $methods = [
        'MD5' => md5($password),
        'SHA1' => sha1($password),
        'SHA256' => hash('sha256', $password),
        'MD5 + Salt (kso)' => md5($password . 'kso'),
        'MD5 + Salt (chandigarh)' => md5($password . 'chandigarh'),
        'MD5 + Salt (kso_chandigarh)' => md5($password . 'kso_chandigarh'),
        'MD5 + Username Salt' => md5($password . 'haominthang'),
        'MD5 Reversed' => md5(strrev($password)),
        'MD5 Uppercase' => md5(strtoupper($password)),
        'MD5 Lowercase' => md5(strtolower($password)),
        'Double MD5' => md5(md5($password)),
        'MD5 + 123' => md5($password . '123'),
        'MD5 + 2024' => md5($password . '2024'),
        'MD5 + 2025' => md5($password . '2025'),
    ];
    
    echo "Testing password: '$password'\n";
    foreach ($methods as $method => $hash) {
        echo "  $method: $hash";
        if ($hash === $stored_hash) {
            echo " ✅ MATCH FOUND!";
            return true;
        }
        echo "\n";
    }
    echo "\n";
    return false;
}

// Comprehensive password list
$test_passwords = [
    // Basic passwords
    'password', 'admin', 'admin123', '123456', '12345', '1234',
    
    // KSO specific
    'kso', 'kso123', 'kso2024', 'kso2025', 'ksochandigarh',
    'chandigarh', 'chandigarh123',
    
    // User names from database
    'haominthang', 'haolam', 'kamgoulen', 'alex', 'lamneihoi',
    'lungouhen', 'lhingboineng', 'lhinghoithem', 'thangminlun',
    
    // Nicknames
    'hao', 'lam', 'len', 'alex', 'neihoi', 'gouhen', 'neng', 'them', 'lun',
    
    // Common variations
    'test', 'demo', 'guest', 'user', 'default', 'temp',
    'welcome', 'letmein', 'changeme', 'system',
    
    // With numbers
    'test123', 'demo123', 'user123', 'default123', 'temp123',
    'welcome123', 'system123',
    
    // Empty or simple
    '', '0', '1', 'a', 'abc', 'abc123',
    
    // Variations of admin
    'administrator', 'root', 'superuser',
    
    // Date variations
    '2024', '2025', '15122024', '15/12/2024',
    
    // Organization related
    'student', 'students', 'organization', 'org', 'club',
    'university', 'college', 'panjab', 'chandigarhuni'
];

// Test all passwords
$found = false;
foreach ($test_passwords as $password) {
    if (testHashingMethods($password)) {
        $found = true;
        echo "🎉 PASSWORD FOUND: '$password'\n\n";
        break;
    }
}

if (!$found) {
    echo "❌ No password found with standard methods.\n\n";
}

// Show current authentication options
echo "=== CURRENT AUTHENTICATION OPTIONS ===\n\n";

echo "🔧 DEVELOPMENT MODE (No Database Required):\n";
echo "   Username: admin\n";
echo "   Password: admin123\n";
echo "   OR\n";
echo "   Username: haominthang\n";
echo "   Password: password\n\n";

echo "🗄️ DATABASE MODE:\n";
echo "   All users in database have the same hash: $stored_hash\n";
echo "   Users available: haominthang, haolam, kamgoulen, alex.mangboi, etc.\n";
echo "   Password: [UNKNOWN - needs to be cracked or reset]\n\n";

// Generate new password hashes for reference
echo "=== GENERATING NEW SECURE PASSWORDS ===\n\n";

$new_passwords = ['password', 'admin123', 'kso123', 'chandigarh123'];
foreach ($new_passwords as $pwd) {
    echo "Password: '$pwd'\n";
    echo "  MD5: " . md5($pwd) . "\n";
    echo "  password_hash: " . password_hash($pwd, PASSWORD_DEFAULT) . "\n";
    echo "  Bcrypt cost 10: " . password_hash($pwd, PASSWORD_BCRYPT, ['cost' => 10]) . "\n\n";
}

// Create SQL update statements
echo "=== SQL COMMANDS TO UPDATE PASSWORDS ===\n\n";

$new_hash_password = password_hash('password', PASSWORD_DEFAULT);
$new_hash_admin123 = password_hash('admin123', PASSWORD_DEFAULT);
$new_hash_kso123 = password_hash('kso123', PASSWORD_DEFAULT);

echo "-- Update all admin passwords to 'password'\n";
echo "UPDATE admins SET password_hash = '$new_hash_password';\n\n";

echo "-- Update all admin passwords to 'admin123'\n";
echo "UPDATE admins SET password_hash = '$new_hash_admin123';\n\n";

echo "-- Update all admin passwords to 'kso123'\n";
echo "UPDATE admins SET password_hash = '$new_hash_kso123';\n\n";

echo "-- Update specific user (haominthang) to 'password'\n";
echo "UPDATE admins SET password_hash = '$new_hash_password' WHERE username = 'haominthang';\n\n";

// Test hosting-specific patterns
echo "=== TESTING HOSTING-SPECIFIC PATTERNS ===\n\n";

// Some hosting providers use specific salting methods
$hosting_patterns = [
    'razorhost' => 'razorhost',
    'cpanel' => 'cpanel',
    'mysql' => 'mysql',
    'localhost' => 'localhost',
    'server' => 'server',
    'hosting' => 'hosting',
    'web' => 'web',
    'site' => 'site'
];

foreach ($hosting_patterns as $name => $salt) {
    $test_password = 'password';
    $hash_with_salt = md5($test_password . $salt);
    echo "MD5('password' + '$salt'): $hash_with_salt";
    if ($hash_with_salt === $stored_hash) {
        echo " ✅ HOSTING PATTERN MATCH!";
    }
    echo "\n";
}

echo "\n=== RECOMMENDATIONS ===\n\n";
echo "1. Use development mode credentials for testing:\n";
echo "   - admin / admin123\n";
echo "   - haominthang / password\n\n";

echo "2. If you need database access, update the password hashes using the SQL above\n\n";

echo "3. For production, always use password_hash() instead of MD5\n\n";

echo "4. The current authentication system supports both legacy MD5 and modern password_hash\n";
echo "   and will automatically upgrade MD5 to secure hashing on successful login.\n\n";

?>
