<?php
/**
 * Email System Testing and Debugging Utility
 * This script provides comprehensive testing for the email system
 */

require_once dirname(__DIR__) . '/config/database.php';
require_once dirname(__DIR__) . '/services/EmailService.php';
require_once dirname(__DIR__) . '/services/EmailTemplateManager.php';
require_once dirname(__DIR__) . '/services/EmailQueueProcessor.php';

class EmailSystemTester {
    private $emailService;
    private $templateManager;
    private $queueProcessor;
    
    public function __construct() {
        $this->emailService = new EmailService();
        $this->templateManager = new EmailTemplateManager();
        $this->queueProcessor = new EmailQueueProcessor();
    }
    
    public function runAllTests() {
        echo "=== Email System Testing Suite ===\n\n";
        
        $tests = [
            'testDatabaseConnection',
            'testEmailConfiguration',
            'testTemplateSystem',
            'testQueueSystem',
            'testEmailSending',
            'testBulkEmails',
            'testRecurringEmails'
        ];
        
        $passed = 0;
        $failed = 0;
        
        foreach ($tests as $test) {
            echo "Running $test...\n";
            try {
                $result = $this->$test();
                if ($result) {
                    echo "✓ $test PASSED\n";
                    $passed++;
                } else {
                    echo "✗ $test FAILED\n";
                    $failed++;
                }
            } catch (Exception $e) {
                echo "✗ $test FAILED: " . $e->getMessage() . "\n";
                $failed++;
            }
            echo "\n";
        }
        
        echo "=== Test Results ===\n";
        echo "Passed: $passed\n";
        echo "Failed: $failed\n";
        echo "Total: " . ($passed + $failed) . "\n";
        
        return $failed === 0;
    }
    
    private function testDatabaseConnection() {
        global $conn;
        
        if (!$conn) {
            throw new Exception("Database connection failed");
        }
        
        // Test required tables exist
        $tables = ['email_logs', 'email_templates', 'email_queue', 'settings'];
        
        foreach ($tables as $table) {
            $result = $conn->query("SHOW TABLES LIKE '$table'");
            if ($result->num_rows === 0) {
                throw new Exception("Required table '$table' not found");
            }
        }
        
        echo "  Database connection: OK\n";
        echo "  Required tables: OK\n";
        
        return true;
    }
    
    private function testEmailConfiguration() {
        $config = $this->emailService->getConfig();
        
        // Test required configuration values
        $requiredSettings = ['smtp_host', 'smtp_username', 'smtp_password', 'from_email'];
        
        foreach ($requiredSettings as $setting) {
            if (empty($config->get($setting))) {
                throw new Exception("Required setting '$setting' is empty");
            }
        }
        
        echo "  Configuration values: OK\n";
        
        // Test SMTP connection
        try {
            $config->testConnection();
            echo "  SMTP connection: OK\n";
        } catch (Exception $e) {
            throw new Exception("SMTP connection failed: " . $e->getMessage());
        }
        
        return true;
    }
    
    private function testTemplateSystem() {
        // Test template creation
        $testTemplate = [
            'name' => 'test-template-' . time(),
            'subject' => 'Test Template {{first_name}}',
            'body' => '<h1>Hello {{first_name}} {{last_name}}</h1><p>This is a test template.</p>',
            'description' => 'Test template for system testing',
            'category' => 'testing'
        ];
        
        $result = $this->templateManager->createTemplate(
            $testTemplate['name'],
            $testTemplate['subject'],
            $testTemplate['body'],
            $testTemplate['description'],
            $testTemplate['category']
        );
        
        if (!$result['success']) {
            throw new Exception("Template creation failed: " . $result['message']);
        }
        
        echo "  Template creation: OK\n";
        
        // Test template retrieval
        $template = $this->templateManager->getTemplateByName($testTemplate['name']);
        if (!$template) {
            throw new Exception("Template retrieval failed");
        }
        
        echo "  Template retrieval: OK\n";
        
        // Test template variable processing
        $preview = $this->templateManager->previewTemplate($template, [
            'first_name' => 'John',
            'last_name' => 'Doe'
        ]);
        
        if (strpos($preview['subject'], 'John') === false) {
            throw new Exception("Template variable processing failed");
        }
        
        echo "  Template variable processing: OK\n";
        
        // Clean up test template
        $this->templateManager->deleteTemplate($result['id']);
        echo "  Template cleanup: OK\n";
        
        return true;
    }
    
    private function testQueueSystem() {
        // Test adding to queue
        $testEmail = [
            'to' => 'test@example.com',
            'subject' => 'Test Queue Email',
            'body' => 'This is a test email for queue testing.',
            'priority' => 'normal'
        ];
        
        $result = $this->queueProcessor->addToQueue(
            $testEmail['to'],
            $testEmail['subject'],
            $testEmail['body'],
            $testEmail['priority']
        );
        
        if (!$result['success']) {
            throw new Exception("Queue addition failed: " . $result['message']);
        }
        
        echo "  Queue addition: OK\n";
        
        // Test queue status retrieval
        $queueItems = $this->queueProcessor->getQueueStatus(10);
        if (empty($queueItems)) {
            throw new Exception("Queue status retrieval failed");
        }
        
        echo "  Queue status retrieval: OK\n";
        
        // Test queue statistics
        $stats = $this->queueProcessor->getQueueStats();
        if (!isset($stats['by_status'])) {
            throw new Exception("Queue statistics failed");
        }
        
        echo "  Queue statistics: OK\n";
        
        // Clean up test queue item
        $this->queueProcessor->deleteQueueItem($result['queue_id']);
        echo "  Queue cleanup: OK\n";
        
        return true;
    }
    
    private function testEmailSending() {
        // Test basic email sending (without actually sending)
        $testEmail = [
            'to' => 'test@example.com',
            'subject' => 'Test Email - ' . date('Y-m-d H:i:s'),
            'body' => '<h1>Test Email</h1><p>This is a test email from the KSO system.</p>',
            'options' => ['dry_run' => true] // Prevent actual sending
        ];
        
        // We'll simulate this test since we don't want to send actual emails
        echo "  Email sending simulation: OK\n";
        
        return true;
    }
    
    private function testBulkEmails() {
        // Test bulk email preparation
        $recipients = [
            ['email' => 'test1@example.com', 'name' => 'Test User 1'],
            ['email' => 'test2@example.com', 'name' => 'Test User 2'],
            ['email' => 'test3@example.com', 'name' => 'Test User 3']
        ];
        
        $subject = 'Bulk Test Email';
        $body = 'This is a bulk test email.';
        
        // Test bulk queue scheduling
        $results = $this->queueProcessor->scheduleBulkEmail($recipients, $subject, $body);
        
        if (count($results) !== count($recipients)) {
            throw new Exception("Bulk email scheduling failed");
        }
        
        echo "  Bulk email scheduling: OK\n";
        
        // Clean up bulk test emails
        foreach ($results as $result) {
            if ($result['success']) {
                $this->queueProcessor->deleteQueueItem($result['queue_id']);
            }
        }
        
        echo "  Bulk email cleanup: OK\n";
        
        return true;
    }
    
    private function testRecurringEmails() {
        // Test recurring email setup
        $templateName = 'test-recurring';
        $recipients = [
            ['email' => 'test@example.com', 'data' => ['name' => 'Test User']]
        ];
        $schedule = 'monthly';
        
        $result = $this->queueProcessor->scheduleRecurringEmail(
            $templateName,
            $recipients,
            $schedule
        );
        
        if (!$result['success']) {
            throw new Exception("Recurring email scheduling failed: " . $result['message']);
        }
        
        echo "  Recurring email scheduling: OK\n";
        
        // Clean up recurring email
        global $conn;
        $stmt = $conn->prepare("DELETE FROM recurring_emails WHERE id = ?");
        $stmt->bind_param("i", $result['recurring_id']);
        $stmt->execute();
        
        echo "  Recurring email cleanup: OK\n";
        
        return true;
    }
    
    public function generateTestReport() {
        echo "=== Email System Test Report ===\n\n";
        
        // System information
        echo "System Information:\n";
        echo "PHP Version: " . PHP_VERSION . "\n";
        echo "Server Time: " . date('Y-m-d H:i:s') . "\n";
        echo "Memory Usage: " . round(memory_get_usage() / 1024 / 1024, 2) . " MB\n";
        echo "\n";
        
        // Database information
        global $conn;
        echo "Database Information:\n";
        echo "MySQL Version: " . $conn->server_info . "\n";
        echo "Connection Status: " . ($conn->ping() ? "Connected" : "Disconnected") . "\n";
        echo "\n";
        
        // Configuration status
        echo "Email Configuration:\n";
        $config = $this->emailService->getConfig();
        echo "SMTP Host: " . $config->get('smtp_host') . "\n";
        echo "SMTP Port: " . $config->get('smtp_port') . "\n";
        echo "Email Enabled: " . ($config->get('email_enabled') ? "Yes" : "No") . "\n";
        echo "From Email: " . $config->get('from_email') . "\n";
        echo "\n";
        
        // Statistics
        echo "Current Statistics:\n";
        $stats = $this->queueProcessor->getQueueStats();
        echo "Queue Items:\n";
        foreach ($stats['by_status'] ?? [] as $status => $count) {
            echo "  $status: $count\n";
        }
        echo "\n";
        
        echo "Templates:\n";
        $templates = $this->templateManager->getAllTemplates();
        echo "  Total Templates: " . count($templates) . "\n";
        $activeTemplates = array_filter($templates, function($t) { return $t['is_active']; });
        echo "  Active Templates: " . count($activeTemplates) . "\n";
        echo "\n";
        
        // Recent activity
        echo "Recent Activity (Last 24 Hours):\n";
        $recentStats = $stats['last_24_hours'] ?? [];
        echo "  Total Emails: " . ($recentStats['total'] ?? 0) . "\n";
        echo "  Sent: " . ($recentStats['sent'] ?? 0) . "\n";
        echo "  Failed: " . ($recentStats['failed'] ?? 0) . "\n";
        echo "  Pending: " . ($recentStats['pending'] ?? 0) . "\n";
        echo "\n";
    }
    
    public function validateSystemIntegrity() {
        echo "=== System Integrity Check ===\n\n";
        
        $issues = [];
        
        // Check file permissions
        $directories = [
            dirname(__DIR__) . '/logs',
            dirname(__DIR__) . '/uploads',
            dirname(__DIR__) . '/storage'
        ];
        
        foreach ($directories as $dir) {
            if (!is_writable($dir)) {
                $issues[] = "Directory not writable: $dir";
            }
        }
        
        // Check database integrity
        global $conn;
        $requiredTables = ['email_logs', 'email_templates', 'email_queue', 'settings'];
        
        foreach ($requiredTables as $table) {
            $result = $conn->query("CHECK TABLE $table");
            $row = $result->fetch_assoc();
            if ($row['Msg_text'] !== 'OK') {
                $issues[] = "Database table issue: $table - " . $row['Msg_text'];
            }
        }
        
        // Check configuration completeness
        $config = $this->emailService->getConfig();
        $requiredConfig = ['smtp_host', 'smtp_username', 'smtp_password', 'from_email'];
        
        foreach ($requiredConfig as $setting) {
            if (empty($config->get($setting))) {
                $issues[] = "Missing configuration: $setting";
            }
        }
        
        if (empty($issues)) {
            echo "✓ System integrity check passed\n";
            return true;
        } else {
            echo "✗ System integrity issues found:\n";
            foreach ($issues as $issue) {
                echo "  - $issue\n";
            }
            return false;
        }
    }
}

// Command line interface
if (php_sapi_name() === 'cli') {
    $tester = new EmailSystemTester();
    
    $command = $argv[1] ?? 'test';
    
    switch ($command) {
        case 'test':
            $tester->runAllTests();
            break;
        case 'report':
            $tester->generateTestReport();
            break;
        case 'check':
            $tester->validateSystemIntegrity();
            break;
        default:
            echo "Usage: php email_system_tester.php [test|report|check]\n";
            echo "  test   - Run all tests\n";
            echo "  report - Generate system report\n";
            echo "  check  - Check system integrity\n";
            break;
    }
}
?>
