<?php
/**
 * Email Queue Processor Cron Job
 * This script should be run periodically to process queued emails
 * 
 * Add to crontab:
 * * /5 * * * * /usr/bin/php /path/to/your/project/access/scripts/process_email_queue.php
 * 
 * This runs every 5 minutes
 */

// Set execution time limit
set_time_limit(300); // 5 minutes

// Include required files
require_once dirname(__DIR__) . '/config/database.php';
require_once dirname(__DIR__) . '/services/EmailService.php';
require_once dirname(__DIR__) . '/services/EmailQueueProcessor.php';

// Log file
$logFile = dirname(__DIR__) . '/logs/email_queue_processing.log';

function writeLog($message) {
    global $logFile;
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[$timestamp] $message\n";
    file_put_contents($logFile, $logMessage, FILE_APPEND | LOCK_EX);
}

function main() {
    writeLog("Starting email queue processing");
    
    try {
        $queueProcessor = new EmailQueueProcessor();
        
        // Process pending emails (limit to 50 per run to avoid timeout)
        $result = $queueProcessor->processQueue(50);
        
        if ($result['processed'] > 0) {
            writeLog("Successfully processed {$result['processed']} emails");
            
            // Log details of failed sends
            foreach ($result['results'] as $emailResult) {
                if (!$emailResult['result']['success']) {
                    writeLog("Failed to send email to {$emailResult['email']}: {$emailResult['result']['message']}");
                }
            }
        } else {
            writeLog("No emails to process");
        }
        
        // Process recurring emails
        $recurringResult = $queueProcessor->processRecurringEmails();
        if ($recurringResult['processed'] > 0) {
            writeLog("Processed {$recurringResult['processed']} recurring emails");
        }
        
        // Clean up old logs (keep last 30 days)
        cleanupOldLogs();
        
        writeLog("Email queue processing completed successfully");
        
    } catch (Exception $e) {
        writeLog("Error during email queue processing: " . $e->getMessage());
        writeLog("Stack trace: " . $e->getTraceAsString());
    }
}

function cleanupOldLogs() {
    global $conn;
    
    try {
        // Delete email logs older than 30 days
        $stmt = $conn->prepare("DELETE FROM email_logs WHERE sent_at < DATE_SUB(NOW(), INTERVAL 30 DAY)");
        $stmt->execute();
        $deletedLogs = $stmt->affected_rows;
        
        // Delete processed queue items older than 7 days
        $stmt = $conn->prepare("DELETE FROM email_queue WHERE status IN ('sent', 'failed') AND updated_at < DATE_SUB(NOW(), INTERVAL 7 DAY)");
        $stmt->execute();
        $deletedQueue = $stmt->affected_rows;
        
        if ($deletedLogs > 0 || $deletedQueue > 0) {
            writeLog("Cleanup: Deleted $deletedLogs old email logs and $deletedQueue old queue items");
        }
        
    } catch (Exception $e) {
        writeLog("Error during cleanup: " . $e->getMessage());
    }
}

// Run the main function
main();
?>
