<?php
/**
 * Enhanced DataTable Service with Filtering, Search, Export, and AdminLTE Integration
 * Provides comprehensive data management functionality for all KSO tables
 */

class DataTableService {
    private $conn;
    private $tableName;
    private $columns = [];
    private $searchableColumns = [];
    private $filterableColumns = [];
    private $exportableColumns = [];
    private $joinTables = [];
    private $customWhere = '';
    private $orderBy = '';
    private $groupBy = '';
    
    public function __construct($tableName) {
        global $conn;
        $this->conn = $conn;
        $this->tableName = $tableName;
        $this->initializeTableConfig();
    }
    
    private function initializeTableConfig() {
        // Configure table-specific settings
        switch ($this->tableName) {
            case 'members':
                $this->columns = [
                    'id' => ['title' => 'ID', 'type' => 'number'],
                    'kso_id' => ['title' => 'KSO ID', 'type' => 'text'],
                    'first_name' => ['title' => 'First Name', 'type' => 'text'],
                    'last_name' => ['title' => 'Last Name', 'type' => 'text'],
                    'email' => ['title' => 'Email', 'type' => 'email'],
                    'phone' => ['title' => 'Phone', 'type' => 'text'],
                    'college' => ['title' => 'College', 'type' => 'text'],
                    'course' => ['title' => 'Course', 'type' => 'text'],
                    'membership_type' => ['title' => 'Membership Type', 'type' => 'select', 'options' => ['Individual', 'Family']],
                    'status' => ['title' => 'Status', 'type' => 'select', 'options' => ['active', 'inactive', 'pending', 'suspended', 'expired', 'approved', 'rejected']],
                    'created_at' => ['title' => 'Created', 'type' => 'datetime'],
                    'updated_at' => ['title' => 'Updated', 'type' => 'datetime']
                ];
                $this->searchableColumns = ['kso_id', 'first_name', 'last_name', 'email', 'phone', 'college', 'course'];
                $this->filterableColumns = ['membership_type', 'status', 'college', 'course'];
                break;
                
            case 'events':
                $this->columns = [
                    'id' => ['title' => 'ID', 'type' => 'number'],
                    'event_title' => ['title' => 'Event Title', 'type' => 'text'],
                    'event_description' => ['title' => 'Description', 'type' => 'text'],
                    'event_date' => ['title' => 'Date', 'type' => 'date'],
                    'event_time' => ['title' => 'Time', 'type' => 'time'],
                    'event_location' => ['title' => 'Location', 'type' => 'text'],
                    'event_type' => ['title' => 'Type', 'type' => 'select', 'options' => ['meeting', 'cultural', 'sports', 'educational', 'social', 'other']],
                    'max_participants' => ['title' => 'Max Participants', 'type' => 'number'],
                    'registration_fee' => ['title' => 'Fee', 'type' => 'currency'],
                    'status' => ['title' => 'Status', 'type' => 'select', 'options' => ['upcoming', 'ongoing', 'completed', 'cancelled']],
                    'created_at' => ['title' => 'Created', 'type' => 'datetime']
                ];
                $this->searchableColumns = ['event_title', 'event_description', 'event_location'];
                $this->filterableColumns = ['event_type', 'status'];
                break;
                
            case 'finance':
                $this->columns = [
                    'id' => ['title' => 'ID', 'type' => 'number'],
                    'member_id' => ['title' => 'Member ID', 'type' => 'number'],
                    'member_name' => ['title' => 'Member Name', 'type' => 'text'],
                    'transaction_type' => ['title' => 'Type', 'type' => 'select', 'options' => ['membership_fee', 'donation', 'event_fee', 'other']],
                    'amount' => ['title' => 'Amount', 'type' => 'currency'],
                    'payment_method' => ['title' => 'Payment Method', 'type' => 'select', 'options' => ['cash', 'bank_transfer', 'mobile_money', 'card', 'cheque']],
                    'receipt_number' => ['title' => 'Receipt #', 'type' => 'text'],
                    'session_year' => ['title' => 'Session Year', 'type' => 'number'],
                    'status' => ['title' => 'Status', 'type' => 'select', 'options' => ['pending', 'completed', 'failed', 'cancelled']],
                    'payment_date' => ['title' => 'Payment Date', 'type' => 'date'],
                    'created_at' => ['title' => 'Created', 'type' => 'datetime']
                ];
                $this->searchableColumns = ['member_name', 'receipt_number', 'description'];
                $this->filterableColumns = ['transaction_type', 'payment_method', 'status', 'session_year'];
                $this->joinTables = [
                    'members' => ['table' => 'members', 'on' => 'members.id = finance.member_id', 'select' => 'CONCAT(members.first_name, " ", members.last_name) as member_name']
                ];
                break;
                
            case 'expenses':
                $this->columns = [
                    'id' => ['title' => 'ID', 'type' => 'number'],
                    'expense_title' => ['title' => 'Title', 'type' => 'text'],
                    'expense_category' => ['title' => 'Category', 'type' => 'select', 'options' => ['office_supplies', 'events', 'travel', 'utilities', 'maintenance', 'other']],
                    'amount' => ['title' => 'Amount', 'type' => 'currency'],
                    'expense_date' => ['title' => 'Date', 'type' => 'date'],
                    'receipt_number' => ['title' => 'Receipt #', 'type' => 'text'],
                    'status' => ['title' => 'Status', 'type' => 'select', 'options' => ['pending', 'approved', 'rejected', 'paid']],
                    'payment_method' => ['title' => 'Payment Method', 'type' => 'select', 'options' => ['cash', 'bank_transfer', 'cheque']],
                    'created_at' => ['title' => 'Created', 'type' => 'datetime']
                ];
                $this->searchableColumns = ['expense_title', 'receipt_number', 'description'];
                $this->filterableColumns = ['expense_category', 'status', 'payment_method'];
                break;
                
            case 'admins':
                $this->columns = [
                    'id' => ['title' => 'ID', 'type' => 'number'],
                    'kso_id' => ['title' => 'KSO ID', 'type' => 'text'],
                    'first_name' => ['title' => 'First Name', 'type' => 'text'],
                    'last_name' => ['title' => 'Last Name', 'type' => 'text'],
                    'email' => ['title' => 'Email', 'type' => 'email'],
                    'role' => ['title' => 'Role', 'type' => 'select', 'options' => ['President', 'Vice President', 'General Secretary', 'Finance Secretary']],
                    'status' => ['title' => 'Status', 'type' => 'select', 'options' => ['active', 'inactive', 'pending', 'expired']],
                    'term_from_year' => ['title' => 'Term From', 'type' => 'number'],
                    'term_to_year' => ['title' => 'Term To', 'type' => 'number'],
                    'last_login' => ['title' => 'Last Login', 'type' => 'datetime'],
                    'created_at' => ['title' => 'Created', 'type' => 'datetime']
                ];
                $this->searchableColumns = ['kso_id', 'first_name', 'last_name', 'email'];
                $this->filterableColumns = ['role', 'status'];
                break;
                
            case 'documents':
                $this->columns = [
                    'id' => ['title' => 'ID', 'type' => 'number'],
                    'title' => ['title' => 'Title', 'type' => 'text'],
                    'file_name' => ['title' => 'File Name', 'type' => 'text'],
                    'file_type' => ['title' => 'Type', 'type' => 'text'],
                    'file_size' => ['title' => 'Size', 'type' => 'filesize'],
                    'category' => ['title' => 'Category', 'type' => 'select', 'options' => ['general', 'financial', 'membership', 'events', 'policies', 'other']],
                    'is_public' => ['title' => 'Public', 'type' => 'boolean'],
                    'download_count' => ['title' => 'Downloads', 'type' => 'number'],
                    'created_at' => ['title' => 'Created', 'type' => 'datetime']
                ];
                $this->searchableColumns = ['title', 'file_name', 'description'];
                $this->filterableColumns = ['category', 'file_type', 'is_public'];
                break;
                
            case 'notifications':
                $this->columns = [
                    'id' => ['title' => 'ID', 'type' => 'number'],
                    'title' => ['title' => 'Title', 'type' => 'text'],
                    'message' => ['title' => 'Message', 'type' => 'text'],
                    'type' => ['title' => 'Type', 'type' => 'select', 'options' => ['info', 'warning', 'success', 'error', 'announcement']],
                    'priority' => ['title' => 'Priority', 'type' => 'select', 'options' => ['low', 'medium', 'high', 'urgent']],
                    'target_type' => ['title' => 'Target', 'type' => 'select', 'options' => ['all', 'members', 'admins', 'specific']],
                    'is_read' => ['title' => 'Read', 'type' => 'boolean'],
                    'created_at' => ['title' => 'Created', 'type' => 'datetime']
                ];
                $this->searchableColumns = ['title', 'message'];
                $this->filterableColumns = ['type', 'priority', 'target_type', 'is_read'];
                break;
        }
        
        // Set default exportable columns (all columns except sensitive ones)
        $this->exportableColumns = array_keys($this->columns);
    }
    
    public function getData($params = []) {
        // Extract parameters
        $draw = $params['draw'] ?? 1;
        $start = $params['start'] ?? 0;
        $length = $params['length'] ?? 10;
        $search = $params['search']['value'] ?? '';
        $order = $params['order'] ?? [];
        $filters = $params['filters'] ?? [];
        $dateRange = $params['dateRange'] ?? [];
        
        // Build base query
        $baseQuery = $this->buildBaseQuery();
        
        // Add search conditions
        $whereConditions = $this->buildWhereConditions($search, $filters, $dateRange);
        
        // Build complete query
        $query = $baseQuery . $whereConditions;
        
        // Add ordering
        if (!empty($order)) {
            $query .= $this->buildOrderBy($order);
        } else {
            $query .= " ORDER BY {$this->tableName}.created_at DESC";
        }
        
        // Get total count without pagination
        $countQuery = "SELECT COUNT(*) as total FROM ({$query}) as count_table";
        $totalResult = $this->conn->query($countQuery);
        $totalRecords = $totalResult->fetch_assoc()['total'];
        
        // Add pagination
        $query .= " LIMIT $start, $length";
        
        // Execute query
        $result = $this->conn->query($query);
        $data = [];
        
        if ($result) {
            while ($row = $result->fetch_assoc()) {
                $data[] = $this->formatRow($row);
            }
        }
        
        return [
            'draw' => intval($draw),
            'recordsTotal' => $totalRecords,
            'recordsFiltered' => $totalRecords,
            'data' => $data
        ];
    }
    
    private function buildBaseQuery() {
        $selectFields = [];
        
        // Add main table fields
        foreach ($this->columns as $column => $config) {
            if ($column !== 'member_name') { // Skip virtual columns
                $selectFields[] = "{$this->tableName}.{$column}";
            }
        }
        
        // Add joined table fields
        foreach ($this->joinTables as $joinConfig) {
            if (isset($joinConfig['select'])) {
                $selectFields[] = $joinConfig['select'];
            }
        }
        
        $query = "SELECT " . implode(', ', $selectFields) . " FROM {$this->tableName}";
        
        // Add joins
        foreach ($this->joinTables as $joinConfig) {
            $query .= " LEFT JOIN {$joinConfig['table']} ON {$joinConfig['on']}";
        }
        
        return $query;
    }
    
    private function buildWhereConditions($search, $filters, $dateRange) {
        $conditions = [];
        
        // Add custom where conditions
        if (!empty($this->customWhere)) {
            $conditions[] = $this->customWhere;
        }
        
        // Add search conditions
        if (!empty($search)) {
            $searchConditions = [];
            foreach ($this->searchableColumns as $column) {
                $searchConditions[] = "{$this->tableName}.{$column} LIKE '%" . $this->conn->real_escape_string($search) . "%'";
            }
            if (!empty($searchConditions)) {
                $conditions[] = '(' . implode(' OR ', $searchConditions) . ')';
            }
        }
        
        // Add filter conditions
        foreach ($filters as $column => $value) {
            if (!empty($value) && in_array($column, $this->filterableColumns)) {
                if (is_array($value)) {
                    $escapedValues = array_map([$this->conn, 'real_escape_string'], $value);
                    $conditions[] = "{$this->tableName}.{$column} IN ('" . implode("', '", $escapedValues) . "')";
                } else {
                    $conditions[] = "{$this->tableName}.{$column} = '" . $this->conn->real_escape_string($value) . "'";
                }
            }
        }
        
        // Add date range conditions
        if (!empty($dateRange['start']) && !empty($dateRange['end'])) {
            $startDate = $this->conn->real_escape_string($dateRange['start']);
            $endDate = $this->conn->real_escape_string($dateRange['end']);
            $dateColumn = $dateRange['column'] ?? 'created_at';
            $conditions[] = "{$this->tableName}.{$dateColumn} BETWEEN '{$startDate}' AND '{$endDate}'";
        }
        
        return empty($conditions) ? '' : ' WHERE ' . implode(' AND ', $conditions);
    }
    
    private function buildOrderBy($order) {
        $orderBy = [];
        
        foreach ($order as $orderItem) {
            $columnIndex = $orderItem['column'];
            $direction = $orderItem['dir'] === 'desc' ? 'DESC' : 'ASC';
            
            $columnKeys = array_keys($this->columns);
            if (isset($columnKeys[$columnIndex])) {
                $columnName = $columnKeys[$columnIndex];
                $orderBy[] = "{$this->tableName}.{$columnName} {$direction}";
            }
        }
        
        return empty($orderBy) ? '' : ' ORDER BY ' . implode(', ', $orderBy);
    }
    
    private function formatRow($row) {
        $formatted = [];
        
        foreach ($this->columns as $column => $config) {
            $value = $row[$column] ?? '';
            
            switch ($config['type']) {
                case 'currency':
                    $formatted[$column] = '₹' . number_format($value, 2);
                    break;
                case 'filesize':
                    $formatted[$column] = $this->formatFileSize($value);
                    break;
                case 'boolean':
                    $formatted[$column] = $value ? 'Yes' : 'No';
                    break;
                case 'datetime':
                    $formatted[$column] = $value ? date('Y-m-d H:i', strtotime($value)) : '';
                    break;
                case 'date':
                    $formatted[$column] = $value ? date('Y-m-d', strtotime($value)) : '';
                    break;
                case 'email':
                    $formatted[$column] = "<a href='mailto:{$value}'>{$value}</a>";
                    break;
                default:
                    $formatted[$column] = htmlspecialchars($value);
            }
        }
        
        // Add action buttons
        $formatted['actions'] = $this->generateActionButtons($row['id']);
        
        return $formatted;
    }
    
    private function formatFileSize($bytes) {
        if ($bytes >= 1073741824) {
            $bytes = number_format($bytes / 1073741824, 2) . ' GB';
        } elseif ($bytes >= 1048576) {
            $bytes = number_format($bytes / 1048576, 2) . ' MB';
        } elseif ($bytes >= 1024) {
            $bytes = number_format($bytes / 1024, 2) . ' KB';
        } elseif ($bytes > 1) {
            $bytes = $bytes . ' bytes';
        } elseif ($bytes == 1) {
            $bytes = $bytes . ' byte';
        } else {
            $bytes = '0 bytes';
        }
        
        return $bytes;
    }
    
    private function generateActionButtons($id) {
        $buttons = [];
        
        // View button
        $buttons[] = "<button class='btn btn-sm btn-info' onclick='viewRecord({$id})' title='View'><i class='fas fa-eye'></i></button>";
        
        // Edit button
        $buttons[] = "<button class='btn btn-sm btn-warning' onclick='editRecord({$id})' title='Edit'><i class='fas fa-edit'></i></button>";
        
        // Delete button
        $buttons[] = "<button class='btn btn-sm btn-danger' onclick='deleteRecord({$id})' title='Delete'><i class='fas fa-trash'></i></button>";
        
        // Table-specific buttons
        switch ($this->tableName) {
            case 'members':
                $buttons[] = "<button class='btn btn-sm btn-success' onclick='generateIdCard({$id})' title='ID Card'><i class='fas fa-id-card'></i></button>";
                break;
            case 'finance':
                $buttons[] = "<button class='btn btn-sm btn-primary' onclick='generateReceipt({$id})' title='Receipt'><i class='fas fa-receipt'></i></button>";
                break;
            case 'documents':
                $buttons[] = "<button class='btn btn-sm btn-secondary' onclick='downloadDocument({$id})' title='Download'><i class='fas fa-download'></i></button>";
                break;
        }
        
        return "<div class='btn-group'>" . implode('', $buttons) . "</div>";
    }
    
    public function exportData($format = 'csv', $filters = [], $columns = null) {
        // Use specified columns or all exportable columns
        $exportColumns = $columns ?? $this->exportableColumns;
        
        // Build query without pagination
        $baseQuery = $this->buildBaseQuery();
        $whereConditions = $this->buildWhereConditions('', $filters, []);
        $query = $baseQuery . $whereConditions . " ORDER BY {$this->tableName}.created_at DESC";
        
        $result = $this->conn->query($query);
        
        if (!$result) {
            throw new Exception("Export query failed: " . $this->conn->error);
        }
        
        $data = [];
        while ($row = $result->fetch_assoc()) {
            $exportRow = [];
            foreach ($exportColumns as $column) {
                $exportRow[$column] = $row[$column] ?? '';
            }
            $data[] = $exportRow;
        }
        
        switch ($format) {
            case 'csv':
                return $this->exportToCsv($data, $exportColumns);
            case 'excel':
                return $this->exportToExcel($data, $exportColumns);
            case 'pdf':
                return $this->exportToPdf($data, $exportColumns);
            default:
                throw new Exception("Unsupported export format: {$format}");
        }
    }
    
    private function exportToCsv($data, $columns) {
        $filename = $this->tableName . '_export_' . date('Y-m-d_H-i-s') . '.csv';
        
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        
        $output = fopen('php://output', 'w');
        
        // Write headers
        $headers = [];
        foreach ($columns as $column) {
            $headers[] = $this->columns[$column]['title'] ?? ucfirst($column);
        }
        fputcsv($output, $headers);
        
        // Write data
        foreach ($data as $row) {
            $csvRow = [];
            foreach ($columns as $column) {
                $csvRow[] = $row[$column] ?? '';
            }
            fputcsv($output, $csvRow);
        }
        
        fclose($output);
        exit;
    }
    
    private function exportToExcel($data, $columns) {
        // This would require a library like PhpSpreadsheet
        throw new Exception("Excel export not implemented yet");
    }
    
    private function exportToPdf($data, $columns) {
        // This would require a library like TCPDF
        throw new Exception("PDF export not implemented yet");
    }
    
    public function getFilterOptions($column) {
        if (!in_array($column, $this->filterableColumns)) {
            return [];
        }
        
        // Check if column has predefined options
        if (isset($this->columns[$column]['options'])) {
            $options = [];
            foreach ($this->columns[$column]['options'] as $option) {
                $options[] = ['value' => $option, 'label' => ucfirst($option)];
            }
            return $options;
        }
        
        // Get distinct values from database
        $query = "SELECT DISTINCT {$column} FROM {$this->tableName} WHERE {$column} IS NOT NULL AND {$column} != '' ORDER BY {$column}";
        $result = $this->conn->query($query);
        
        $options = [];
        while ($row = $result->fetch_assoc()) {
            $value = $row[$column];
            $options[] = ['value' => $value, 'label' => ucfirst($value)];
        }
        
        return $options;
    }
    
    public function getTableConfig() {
        return [
            'tableName' => $this->tableName,
            'columns' => $this->columns,
            'searchableColumns' => $this->searchableColumns,
            'filterableColumns' => $this->filterableColumns,
            'exportableColumns' => $this->exportableColumns
        ];
    }
    
    public function setCustomWhere($where) {
        $this->customWhere = $where;
        return $this;
    }
    
    public function setOrderBy($orderBy) {
        $this->orderBy = $orderBy;
        return $this;
    }
}
