<?php
require_once 'config/database.php';
require_once 'services/EmailService.php';

class EmailQueueProcessor {
    private $conn;
    private $emailService;
    
    public function __construct() {
        global $conn;
        $this->conn = $conn;
        $this->emailService = new EmailService();
    }
    
    public function addToQueue($to, $subject, $body, $priority = 'normal', $options = []) {
        $priorityLevel = $this->getPriorityLevel($priority);
        $data = json_encode($options);
        
        $stmt = $this->conn->prepare("INSERT INTO email_queue (to_email, subject, body, priority, data, status, created_at) VALUES (?, ?, ?, ?, ?, 'pending', NOW())");
        $stmt->bind_param("sssis", $to, $subject, $body, $priorityLevel, $data);
        
        if ($stmt->execute()) {
            return [
                'success' => true,
                'message' => 'Email added to queue successfully',
                'queue_id' => $this->conn->insert_id
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Failed to add email to queue: ' . $stmt->error
            ];
        }
    }
    
    public function addTemplateToQueue($templateName, $templateData, $to, $priority = 'normal', $options = []) {
        $template = $this->getTemplate($templateName);
        
        if (!$template) {
            return [
                'success' => false,
                'message' => "Template '{$templateName}' not found"
            ];
        }
        
        $subject = $this->processTemplateVariables($template['subject'], $templateData);
        $body = $this->processTemplateVariables($template['body'], $templateData);
        
        $options['template'] = $templateName;
        $options['template_data'] = $templateData;
        
        return $this->addToQueue($to, $subject, $body, $priority, $options);
    }
    
    public function processQueue($limit = 10) {
        $stmt = $this->conn->prepare("SELECT * FROM email_queue WHERE status = 'pending' ORDER BY priority DESC, created_at ASC LIMIT ?");
        $stmt->bind_param("i", $limit);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $processedCount = 0;
        $results = [];
        
        while ($queueItem = $result->fetch_assoc()) {
            $this->updateQueueStatus($queueItem['id'], 'processing');
            
            try {
                $options = json_decode($queueItem['data'], true) ?? [];
                $sendResult = $this->emailService->sendEmail($queueItem['to_email'], $queueItem['subject'], $queueItem['body'], $options);
                
                if ($sendResult['success']) {
                    $this->updateQueueStatus($queueItem['id'], 'sent', $sendResult['log_id']);
                    $processedCount++;
                } else {
                    $this->updateQueueStatus($queueItem['id'], 'failed', null, $sendResult['message']);
                }
                
                $results[] = [
                    'queue_id' => $queueItem['id'],
                    'email' => $queueItem['to_email'],
                    'result' => $sendResult
                ];
                
            } catch (Exception $e) {
                $this->updateQueueStatus($queueItem['id'], 'failed', null, $e->getMessage());
                $results[] = [
                    'queue_id' => $queueItem['id'],
                    'email' => $queueItem['to_email'],
                    'result' => [
                        'success' => false,
                        'message' => $e->getMessage()
                    ]
                ];
            }
            
            // Small delay between emails
            usleep(100000); // 0.1 seconds
        }
        
        return [
            'processed' => $processedCount,
            'results' => $results
        ];
    }
    
    public function scheduleBulkEmail($recipients, $subject, $body, $scheduleTime = null, $priority = 'normal', $options = []) {
        $results = [];
        
        foreach ($recipients as $recipient) {
            $email = is_array($recipient) ? $recipient['email'] : $recipient;
            $personalizedSubject = $subject;
            $personalizedBody = $body;
            
            // Personalize if data provided
            if (is_array($recipient) && isset($recipient['data'])) {
                $personalizedSubject = $this->processTemplateVariables($subject, $recipient['data']);
                $personalizedBody = $this->processTemplateVariables($body, $recipient['data']);
            }
            
            $queueOptions = $options;
            if ($scheduleTime) {
                $queueOptions['schedule_time'] = $scheduleTime;
            }
            
            $result = $this->addToQueue($email, $personalizedSubject, $personalizedBody, $priority, $queueOptions);
            $results[] = array_merge($result, ['email' => $email]);
        }
        
        return $results;
    }
    
    public function retryFailedQueue($limit = 5) {
        $stmt = $this->conn->prepare("SELECT * FROM email_queue WHERE status = 'failed' AND retry_count < 3 ORDER BY updated_at DESC LIMIT ?");
        $stmt->bind_param("i", $limit);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $retryResults = [];
        
        while ($queueItem = $result->fetch_assoc()) {
            $this->updateQueueStatus($queueItem['id'], 'pending');
            $this->incrementRetryCount($queueItem['id']);
            
            $retryResults[] = [
                'queue_id' => $queueItem['id'],
                'email' => $queueItem['to_email'],
                'retry_count' => $queueItem['retry_count'] + 1
            ];
        }
        
        return $retryResults;
    }
    
    public function getQueueStatus($limit = 50, $offset = 0, $filters = []) {
        $where = [];
        $params = [];
        $types = '';
        
        if (!empty($filters['status'])) {
            $where[] = "status = ?";
            $params[] = $filters['status'];
            $types .= 's';
        }
        
        if (!empty($filters['priority'])) {
            $where[] = "priority = ?";
            $params[] = $this->getPriorityLevel($filters['priority']);
            $types .= 'i';
        }
        
        if (!empty($filters['from_date'])) {
            $where[] = "created_at >= ?";
            $params[] = $filters['from_date'];
            $types .= 's';
        }
        
        if (!empty($filters['to_date'])) {
            $where[] = "created_at <= ?";
            $params[] = $filters['to_date'];
            $types .= 's';
        }
        
        if (!empty($filters['email'])) {
            $where[] = "to_email LIKE ?";
            $params[] = '%' . $filters['email'] . '%';
            $types .= 's';
        }
        
        $whereClause = empty($where) ? '' : 'WHERE ' . implode(' AND ', $where);
        
        $sql = "SELECT *, 
                CASE 
                    WHEN priority = 3 THEN 'high'
                    WHEN priority = 2 THEN 'normal'
                    WHEN priority = 1 THEN 'low'
                END as priority_text
                FROM email_queue $whereClause ORDER BY priority DESC, created_at DESC LIMIT $limit OFFSET $offset";
        
        $stmt = $this->conn->prepare($sql);
        if (!empty($params)) {
            $stmt->bind_param($types, ...$params);
        }
        $stmt->execute();
        
        return $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    }
    
    public function getQueueStats() {
        $stats = [];
        
        // Total counts by status
        $statusQuery = "SELECT status, COUNT(*) as count FROM email_queue GROUP BY status";
        $result = $this->conn->query($statusQuery);
        while ($row = $result->fetch_assoc()) {
            $stats['by_status'][$row['status']] = $row['count'];
        }
        
        // Total counts by priority
        $priorityQuery = "SELECT 
            CASE 
                WHEN priority = 3 THEN 'high'
                WHEN priority = 2 THEN 'normal'
                WHEN priority = 1 THEN 'low'
            END as priority_text,
            COUNT(*) as count 
            FROM email_queue 
            GROUP BY priority";
        $result = $this->conn->query($priorityQuery);
        while ($row = $result->fetch_assoc()) {
            $stats['by_priority'][$row['priority_text']] = $row['count'];
        }
        
        // Recent activity (last 24 hours)
        $recentQuery = "SELECT 
            COUNT(*) as total,
            SUM(CASE WHEN status = 'sent' THEN 1 ELSE 0 END) as sent,
            SUM(CASE WHEN status = 'failed' THEN 1 ELSE 0 END) as failed,
            SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending
            FROM email_queue 
            WHERE created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)";
        $result = $this->conn->query($recentQuery);
        $stats['last_24_hours'] = $result->fetch_assoc();
        
        return $stats;
    }
    
    public function clearQueue($status = null) {
        if ($status) {
            $stmt = $this->conn->prepare("DELETE FROM email_queue WHERE status = ?");
            $stmt->bind_param("s", $status);
        } else {
            $stmt = $this->conn->prepare("DELETE FROM email_queue");
        }
        
        if ($stmt->execute()) {
            return [
                'success' => true,
                'message' => 'Queue cleared successfully',
                'deleted_count' => $stmt->affected_rows
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Failed to clear queue: ' . $stmt->error
            ];
        }
    }
    
    public function pauseQueue() {
        $stmt = $this->conn->prepare("UPDATE email_queue SET status = 'paused' WHERE status = 'pending'");
        
        if ($stmt->execute()) {
            return [
                'success' => true,
                'message' => 'Queue paused successfully',
                'affected_count' => $stmt->affected_rows
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Failed to pause queue: ' . $stmt->error
            ];
        }
    }
    
    public function resumeQueue() {
        $stmt = $this->conn->prepare("UPDATE email_queue SET status = 'pending' WHERE status = 'paused'");
        
        if ($stmt->execute()) {
            return [
                'success' => true,
                'message' => 'Queue resumed successfully',
                'affected_count' => $stmt->affected_rows
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Failed to resume queue: ' . $stmt->error
            ];
        }
    }
    
    public function deleteQueueItem($id) {
        $stmt = $this->conn->prepare("DELETE FROM email_queue WHERE id = ?");
        $stmt->bind_param("i", $id);
        
        if ($stmt->execute()) {
            return [
                'success' => true,
                'message' => 'Queue item deleted successfully'
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Failed to delete queue item: ' . $stmt->error
            ];
        }
    }
    
    public function scheduleRecurringEmail($templateName, $recipients, $schedule, $templateData = []) {
        // This would be used for recurring emails like monthly newsletters
        // Store in a separate recurring_emails table
        $stmt = $this->conn->prepare("INSERT INTO recurring_emails (template_name, recipients, schedule_pattern, template_data, is_active, created_at) VALUES (?, ?, ?, ?, 1, NOW())");
        $recipientsJson = json_encode($recipients);
        $templateDataJson = json_encode($templateData);
        $stmt->bind_param("ssss", $templateName, $recipientsJson, $schedule, $templateDataJson);
        
        if ($stmt->execute()) {
            return [
                'success' => true,
                'message' => 'Recurring email scheduled successfully',
                'recurring_id' => $this->conn->insert_id
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Failed to schedule recurring email: ' . $stmt->error
            ];
        }
    }
    
    public function processRecurringEmails() {
        // This would check for recurring emails that need to be sent
        // Based on their schedule pattern (daily, weekly, monthly, etc.)
        $stmt = $this->conn->prepare("SELECT * FROM recurring_emails WHERE is_active = 1");
        $stmt->execute();
        $result = $stmt->get_result();
        
        $processedCount = 0;
        
        while ($recurringEmail = $result->fetch_assoc()) {
            if ($this->shouldSendRecurring($recurringEmail)) {
                $recipients = json_decode($recurringEmail['recipients'], true);
                $templateData = json_decode($recurringEmail['template_data'], true) ?? [];
                
                foreach ($recipients as $recipient) {
                    $this->addTemplateToQueue(
                        $recurringEmail['template_name'],
                        array_merge($templateData, $recipient['data'] ?? []),
                        $recipient['email'],
                        'normal'
                    );
                }
                
                $this->updateRecurringLastSent($recurringEmail['id']);
                $processedCount++;
            }
        }
        
        return [
            'processed' => $processedCount,
            'message' => "Processed {$processedCount} recurring emails"
        ];
    }
    
    private function shouldSendRecurring($recurringEmail) {
        // Check if enough time has passed since last sent based on schedule
        $lastSent = $recurringEmail['last_sent'] ?? '1970-01-01';
        $schedule = $recurringEmail['schedule_pattern'];
        
        $nextSendTime = strtotime($lastSent);
        
        switch ($schedule) {
            case 'daily':
                $nextSendTime = strtotime($lastSent . ' +1 day');
                break;
            case 'weekly':
                $nextSendTime = strtotime($lastSent . ' +1 week');
                break;
            case 'monthly':
                $nextSendTime = strtotime($lastSent . ' +1 month');
                break;
            default:
                return false;
        }
        
        return time() >= $nextSendTime;
    }
    
    private function updateRecurringLastSent($id) {
        $stmt = $this->conn->prepare("UPDATE recurring_emails SET last_sent = NOW() WHERE id = ?");
        $stmt->bind_param("i", $id);
        $stmt->execute();
    }
    
    private function getPriorityLevel($priority) {
        switch (strtolower($priority)) {
            case 'high':
                return 3;
            case 'normal':
                return 2;
            case 'low':
                return 1;
            default:
                return 2;
        }
    }
    
    private function updateQueueStatus($id, $status, $logId = null, $errorMessage = null) {
        if ($logId) {
            $stmt = $this->conn->prepare("UPDATE email_queue SET status = ?, email_log_id = ?, updated_at = NOW() WHERE id = ?");
            $stmt->bind_param("sii", $status, $logId, $id);
        } elseif ($errorMessage) {
            $stmt = $this->conn->prepare("UPDATE email_queue SET status = ?, error_message = ?, updated_at = NOW() WHERE id = ?");
            $stmt->bind_param("ssi", $status, $errorMessage, $id);
        } else {
            $stmt = $this->conn->prepare("UPDATE email_queue SET status = ?, updated_at = NOW() WHERE id = ?");
            $stmt->bind_param("si", $status, $id);
        }
        
        $stmt->execute();
    }
    
    private function incrementRetryCount($id) {
        $stmt = $this->conn->prepare("UPDATE email_queue SET retry_count = retry_count + 1 WHERE id = ?");
        $stmt->bind_param("i", $id);
        $stmt->execute();
    }
    
    private function getTemplate($name) {
        $stmt = $this->conn->prepare("SELECT * FROM email_templates WHERE name = ? AND is_active = 1");
        $stmt->bind_param("s", $name);
        $stmt->execute();
        $result = $stmt->get_result();
        
        return $result->fetch_assoc();
    }
    
    private function processTemplateVariables($content, $data) {
        foreach ($data as $key => $value) {
            $content = str_replace('{{' . $key . '}}', $value, $content);
        }
        
        // Add common variables
        $commonVars = [
            'current_date' => date('Y-m-d'),
            'current_time' => date('H:i:s'),
            'current_year' => date('Y'),
            'organization_name' => 'KSO Chandigarh',
            'website_url' => 'https://ksochandigarh.com',
            'support_email' => 'support@ksochandigarh.com',
            'login_url' => 'https://ksochandigarh.com/login'
        ];
        
        foreach ($commonVars as $key => $value) {
            $content = str_replace('{{' . $key . '}}', $value, $content);
        }
        
        return $content;
    }
}
