<?php
require_once 'config/email.php';
require_once 'config/database.php';

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\SMTP;
use PHPMailer\PHPMailer\Exception;

class EmailService {
    private $config;
    private $mailer;
    private $conn;
    
    public function __construct() {
        global $conn;
        $this->conn = $conn;
        $this->config = new EmailConfig();
        $this->initializeMailer();
    }
    
    private function initializeMailer() {
        $this->mailer = new PHPMailer(true);
        
        try {
            // Server settings
            $this->mailer->isSMTP();
            $this->mailer->Host = $this->config->get('smtp_host');
            $this->mailer->SMTPAuth = $this->config->get('smtp_auth');
            $this->mailer->Username = $this->config->get('smtp_username');
            $this->mailer->Password = $this->config->get('smtp_password');
            $this->mailer->SMTPSecure = $this->config->get('smtp_encryption');
            $this->mailer->Port = $this->config->get('smtp_port');
            $this->mailer->Timeout = $this->config->get('timeout');
            $this->mailer->CharSet = $this->config->get('charset');
            
            // Default from
            $this->mailer->setFrom($this->config->get('from_email'), $this->config->get('from_name'));
            
            // Reply-to if configured
            if (!empty($this->config->get('email_reply_to'))) {
                $this->mailer->addReplyTo($this->config->get('email_reply_to'), $this->config->get('from_name'));
            }
            
            // Debug level
            $this->mailer->SMTPDebug = $this->config->get('debug_level');
            
            // Priority
            $this->mailer->Priority = $this->config->get('email_priority');
            
        } catch (Exception $e) {
            error_log("Email service initialization failed: " . $e->getMessage());
            throw new Exception("Email service initialization failed");
        }
    }
    
    public function sendEmail($to, $subject, $body, $options = []) {
        if (!$this->config->isEnabled()) {
            throw new Exception("Email service is disabled");
        }
        
        if ($this->config->isRateLimited()) {
            throw new Exception("Rate limit exceeded. Please try again later.");
        }
        
        $logId = $this->logEmail($to, $subject, $options['template'] ?? null, 'pending');
        
        try {
            $this->mailer->clearAddresses();
            $this->mailer->clearAttachments();
            $this->mailer->clearCustomHeaders();
            
            // Recipients
            if (is_array($to)) {
                foreach ($to as $email => $name) {
                    if (is_numeric($email)) {
                        $this->mailer->addAddress($name);
                    } else {
                        $this->mailer->addAddress($email, $name);
                    }
                }
            } else {
                $this->mailer->addAddress($to, $options['to_name'] ?? '');
            }
            
            // CC and BCC
            if (!empty($options['cc'])) {
                if (is_array($options['cc'])) {
                    foreach ($options['cc'] as $email => $name) {
                        $this->mailer->addCC(is_numeric($email) ? $name : $email, is_numeric($email) ? '' : $name);
                    }
                } else {
                    $this->mailer->addCC($options['cc']);
                }
            }
            
            if (!empty($options['bcc'])) {
                if (is_array($options['bcc'])) {
                    foreach ($options['bcc'] as $email => $name) {
                        $this->mailer->addBCC(is_numeric($email) ? $name : $email, is_numeric($email) ? '' : $name);
                    }
                } else {
                    $this->mailer->addBCC($options['bcc']);
                }
            }
            
            // Attachments
            if (!empty($options['attachments'])) {
                foreach ($options['attachments'] as $attachment) {
                    if (is_array($attachment)) {
                        $this->mailer->addAttachment($attachment['path'], $attachment['name'] ?? '');
                    } else {
                        $this->mailer->addAttachment($attachment);
                    }
                }
            }
            
            // Content
            $this->mailer->isHTML($this->config->get('email_html'));
            $this->mailer->Subject = $subject;
            $this->mailer->Body = $body;
            
            // Alt body for HTML emails
            if ($this->config->get('email_html')) {
                $this->mailer->AltBody = strip_tags($body);
            }
            
            // Custom headers
            if (!empty($options['headers'])) {
                foreach ($options['headers'] as $name => $value) {
                    $this->mailer->addCustomHeader($name, $value);
                }
            }
            
            // Tracking
            if ($this->config->get('email_tracking')) {
                $trackingId = $this->generateTrackingId();
                $this->mailer->addCustomHeader('X-Tracking-ID', $trackingId);
                $this->updateEmailLog($logId, ['tracking_id' => $trackingId]);
            }
            
            $this->mailer->send();
            
            $this->updateEmailLog($logId, ['status' => 'sent', 'sent_at' => date('Y-m-d H:i:s')]);
            
            return [
                'success' => true,
                'message' => 'Email sent successfully',
                'log_id' => $logId
            ];
            
        } catch (Exception $e) {
            $this->updateEmailLog($logId, [
                'status' => 'failed',
                'error_message' => $e->getMessage(),
                'retry_count' => 0
            ]);
            
            error_log("Email sending failed: " . $e->getMessage());
            
            return [
                'success' => false,
                'message' => 'Email sending failed: ' . $e->getMessage(),
                'log_id' => $logId
            ];
        }
    }
    
    public function sendTemplate($templateName, $data, $to, $subject = null, $options = []) {
        $template = $this->getTemplate($templateName);
        
        if (!$template) {
            throw new Exception("Email template '{$templateName}' not found");
        }
        
        if (!$template['is_active']) {
            throw new Exception("Email template '{$templateName}' is not active");
        }
        
        $subject = $subject ?? $template['subject'];
        $body = $template['body'];
        
        // Process template variables
        $body = $this->processTemplateVariables($body, $data);
        $subject = $this->processTemplateVariables($subject, $data);
        
        $options['template'] = $templateName;
        
        return $this->sendEmail($to, $subject, $body, $options);
    }
    
    public function sendBulkEmail($recipients, $subject, $body, $options = []) {
        if (!$this->config->isEnabled()) {
            throw new Exception("Email service is disabled");
        }
        
        $results = [];
        $batchSize = $this->config->get('batch_size');
        $batches = array_chunk($recipients, $batchSize);
        
        foreach ($batches as $batch) {
            foreach ($batch as $recipient) {
                if ($this->config->isRateLimited()) {
                    sleep(3600); // Wait 1 hour if rate limited
                }
                
                $email = is_array($recipient) ? $recipient['email'] : $recipient;
                $name = is_array($recipient) ? $recipient['name'] : '';
                
                $personalizedBody = $body;
                $personalizedSubject = $subject;
                
                // Personalize if data provided
                if (is_array($recipient) && isset($recipient['data'])) {
                    $personalizedBody = $this->processTemplateVariables($body, $recipient['data']);
                    $personalizedSubject = $this->processTemplateVariables($subject, $recipient['data']);
                }
                
                $result = $this->sendEmail($email, $personalizedSubject, $personalizedBody, array_merge($options, ['to_name' => $name]));
                $results[] = array_merge($result, ['email' => $email]);
                
                // Small delay between emails
                usleep(100000); // 0.1 seconds
            }
        }
        
        return $results;
    }
    
    public function sendWelcomeEmail($member) {
        if (!$this->config->get('email_auto_welcome')) {
            return ['success' => false, 'message' => 'Welcome emails are disabled'];
        }
        
        $data = [
            'first_name' => $member['first_name'],
            'last_name' => $member['last_name'],
            'kso_id' => $member['kso_id'],
            'membership_type' => $member['membership_type'],
            'current_date' => date('Y-m-d'),
            'current_year' => date('Y'),
            'organization_name' => 'KSO Chandigarh',
            'website_url' => 'https://ksochandigarh.com'
        ];
        
        return $this->sendTemplate('welcome', $data, $member['email']);
    }
    
    public function sendApprovalEmail($member) {
        if (!$this->config->get('email_auto_approval')) {
            return ['success' => false, 'message' => 'Approval emails are disabled'];
        }
        
        $data = [
            'first_name' => $member['first_name'],
            'last_name' => $member['last_name'],
            'kso_id' => $member['kso_id'],
            'current_year' => date('Y'),
            'organization_name' => 'KSO Chandigarh',
            'website_url' => 'https://ksochandigarh.com'
        ];
        
        return $this->sendTemplate('approval', $data, $member['email']);
    }
    
    public function sendRejectionEmail($member, $reason = '') {
        $data = [
            'first_name' => $member['first_name'],
            'last_name' => $member['last_name'],
            'reason' => $reason,
            'current_year' => date('Y'),
            'organization_name' => 'KSO Chandigarh',
            'website_url' => 'https://ksochandigarh.com'
        ];
        
        return $this->sendTemplate('rejection', $data, $member['email']);
    }
    
    public function sendEventNotification($event, $members) {
        if (!$this->config->get('email_auto_events')) {
            return ['success' => false, 'message' => 'Event emails are disabled'];
        }
        
        $recipients = [];
        foreach ($members as $member) {
            $recipients[] = [
                'email' => $member['email'],
                'name' => $member['first_name'] . ' ' . $member['last_name'],
                'data' => [
                    'first_name' => $member['first_name'],
                    'last_name' => $member['last_name'],
                    'event_name' => $event['title'],
                    'event_date' => date('F d, Y', strtotime($event['event_date'])),
                    'event_time' => date('g:i A', strtotime($event['event_time'])),
                    'event_location' => $event['location'],
                    'event_description' => $event['description'],
                    'registration_url' => $event['registration_url'] ?? '',
                    'current_year' => date('Y'),
                    'organization_name' => 'KSO Chandigarh'
                ]
            ];
        }
        
        return $this->sendBulkTemplate('event-notification', $recipients);
    }
    
    public function sendFeeReminder($member, $feeDetails) {
        if (!$this->config->get('email_auto_fees')) {
            return ['success' => false, 'message' => 'Fee reminder emails are disabled'];
        }
        
        $data = [
            'first_name' => $member['first_name'],
            'last_name' => $member['last_name'],
            'amount' => $feeDetails['amount'],
            'due_date' => date('F d, Y', strtotime($feeDetails['due_date'])),
            'payment_url' => $feeDetails['payment_url'] ?? '',
            'description' => $feeDetails['description'] ?? 'Membership Fee',
            'current_year' => date('Y'),
            'organization_name' => 'KSO Chandigarh'
        ];
        
        return $this->sendTemplate('fee-reminder', $data, $member['email']);
    }
    
    public function sendBulkTemplate($templateName, $recipients) {
        $template = $this->getTemplate($templateName);
        
        if (!$template) {
            throw new Exception("Email template '{$templateName}' not found");
        }
        
        $results = [];
        foreach ($recipients as $recipient) {
            $result = $this->sendTemplate($templateName, $recipient['data'], $recipient['email'], null, ['to_name' => $recipient['name']]);
            $results[] = array_merge($result, ['email' => $recipient['email']]);
        }
        
        return $results;
    }
    
    public function testEmailConfiguration($testEmail = null) {
        if (!$testEmail) {
            $testEmail = $this->config->get('smtp_username');
        }
        
        try {
            $this->config->testConnection();
            
            $testSubject = 'KSO Email Configuration Test';
            $testBody = '
                <h2>Email Configuration Test</h2>
                <p>This is a test email to verify your email configuration is working correctly.</p>
                <p><strong>Test Time:</strong> ' . date('Y-m-d H:i:s') . '</p>
                <p><strong>Server:</strong> ' . $this->config->get('smtp_host') . ':' . $this->config->get('smtp_port') . '</p>
                <p>If you receive this email, your configuration is working properly!</p>
            ';
            
            $result = $this->sendEmail($testEmail, $testSubject, $testBody);
            
            return [
                'success' => true,
                'message' => 'Test email sent successfully',
                'details' => $result
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Email test failed: ' . $e->getMessage()
            ];
        }
    }
    
    private function getTemplate($name) {
        $stmt = $this->conn->prepare("SELECT * FROM email_templates WHERE name = ?");
        $stmt->bind_param("s", $name);
        $stmt->execute();
        $result = $stmt->get_result();
        
        return $result->fetch_assoc();
    }
    
    private function processTemplateVariables($content, $data) {
        foreach ($data as $key => $value) {
            $content = str_replace('{{' . $key . '}}', $value, $content);
        }
        
        // Add common variables
        $commonVars = [
            'current_date' => date('Y-m-d'),
            'current_time' => date('H:i:s'),
            'current_year' => date('Y'),
            'organization_name' => 'KSO Chandigarh',
            'website_url' => 'https://ksochandigarh.com',
            'support_email' => $this->config->get('from_email'),
            'login_url' => 'https://ksochandigarh.com/login'
        ];
        
        foreach ($commonVars as $key => $value) {
            $content = str_replace('{{' . $key . '}}', $value, $content);
        }
        
        return $content;
    }
    
    private function logEmail($to, $subject, $template, $status) {
        $stmt = $this->conn->prepare("INSERT INTO email_logs (to_email, subject, template_used, status, sent_at) VALUES (?, ?, ?, ?, NOW())");
        $stmt->bind_param("ssss", $to, $subject, $template, $status);
        $stmt->execute();
        
        return $this->conn->insert_id;
    }
    
    private function updateEmailLog($logId, $updates) {
        $setParts = [];
        $values = [];
        $types = '';
        
        foreach ($updates as $key => $value) {
            $setParts[] = "$key = ?";
            $values[] = $value;
            $types .= 's';
        }
        
        $values[] = $logId;
        $types .= 'i';
        
        $sql = "UPDATE email_logs SET " . implode(', ', $setParts) . " WHERE id = ?";
        $stmt = $this->conn->prepare($sql);
        $stmt->bind_param($types, ...$values);
        $stmt->execute();
    }
    
    private function generateTrackingId() {
        return 'track_' . uniqid() . '_' . time();
    }
    
    public function getEmailLogs($limit = 50, $offset = 0, $filters = []) {
        $where = [];
        $params = [];
        $types = '';
        
        if (!empty($filters['status'])) {
            $where[] = "status = ?";
            $params[] = $filters['status'];
            $types .= 's';
        }
        
        if (!empty($filters['template'])) {
            $where[] = "template_used = ?";
            $params[] = $filters['template'];
            $types .= 's';
        }
        
        if (!empty($filters['from_date'])) {
            $where[] = "sent_at >= ?";
            $params[] = $filters['from_date'];
            $types .= 's';
        }
        
        if (!empty($filters['to_date'])) {
            $where[] = "sent_at <= ?";
            $params[] = $filters['to_date'];
            $types .= 's';
        }
        
        if (!empty($filters['email'])) {
            $where[] = "to_email LIKE ?";
            $params[] = '%' . $filters['email'] . '%';
            $types .= 's';
        }
        
        $whereClause = empty($where) ? '' : 'WHERE ' . implode(' AND ', $where);
        
        $sql = "SELECT * FROM email_logs $whereClause ORDER BY sent_at DESC LIMIT $limit OFFSET $offset";
        
        $stmt = $this->conn->prepare($sql);
        if (!empty($params)) {
            $stmt->bind_param($types, ...$params);
        }
        $stmt->execute();
        
        return $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    }
    
    public function retryFailedEmails($limit = 10) {
        $stmt = $this->conn->prepare("SELECT * FROM email_logs WHERE status = 'failed' AND retry_count < ? ORDER BY sent_at DESC LIMIT ?");
        $maxRetry = $this->config->get('max_retry');
        $stmt->bind_param("ii", $maxRetry, $limit);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $retryResults = [];
        
        while ($log = $result->fetch_assoc()) {
            // Get original template and data (you might need to store this)
            $retryResult = $this->sendEmail($log['to_email'], $log['subject'], '', ['template' => $log['template_used']]);
            
            if ($retryResult['success']) {
                $this->updateEmailLog($log['id'], ['status' => 'sent', 'sent_at' => date('Y-m-d H:i:s')]);
            } else {
                $this->updateEmailLog($log['id'], ['retry_count' => $log['retry_count'] + 1]);
            }
            
            $retryResults[] = [
                'email' => $log['to_email'],
                'result' => $retryResult
            ];
        }
        
        return $retryResults;
    }
    
    public function getConfig() {
        return $this->config;
    }
    
    public function updateConfig($settings) {
        foreach ($settings as $key => $value) {
            $this->config->set($key, $value);
        }
        
        // Reinitialize mailer with new settings
        $this->initializeMailer();
        
        return true;
    }
}
