<?php
require_once 'config/database.php';

class EmailTemplateManager {
    private $conn;
    
    public function __construct() {
        global $conn;
        $this->conn = $conn;
    }
    
    public function createTemplate($name, $subject, $body, $description = '', $category = 'general') {
        $stmt = $this->conn->prepare("INSERT INTO email_templates (name, subject, body, description, category, is_active, created_at) VALUES (?, ?, ?, ?, ?, 1, NOW())");
        $stmt->bind_param("sssss", $name, $subject, $body, $description, $category);
        
        if ($stmt->execute()) {
            return [
                'success' => true,
                'message' => 'Template created successfully',
                'id' => $this->conn->insert_id
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Failed to create template: ' . $stmt->error
            ];
        }
    }
    
    public function updateTemplate($id, $name, $subject, $body, $description = '', $category = 'general') {
        $stmt = $this->conn->prepare("UPDATE email_templates SET name = ?, subject = ?, body = ?, description = ?, category = ?, updated_at = NOW() WHERE id = ?");
        $stmt->bind_param("sssssi", $name, $subject, $body, $description, $category, $id);
        
        if ($stmt->execute()) {
            return [
                'success' => true,
                'message' => 'Template updated successfully'
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Failed to update template: ' . $stmt->error
            ];
        }
    }
    
    public function deleteTemplate($id) {
        $stmt = $this->conn->prepare("DELETE FROM email_templates WHERE id = ?");
        $stmt->bind_param("i", $id);
        
        if ($stmt->execute()) {
            return [
                'success' => true,
                'message' => 'Template deleted successfully'
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Failed to delete template: ' . $stmt->error
            ];
        }
    }
    
    public function getTemplate($id) {
        $stmt = $this->conn->prepare("SELECT * FROM email_templates WHERE id = ?");
        $stmt->bind_param("i", $id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        return $result->fetch_assoc();
    }
    
    public function getTemplateByName($name) {
        $stmt = $this->conn->prepare("SELECT * FROM email_templates WHERE name = ?");
        $stmt->bind_param("s", $name);
        $stmt->execute();
        $result = $stmt->get_result();
        
        return $result->fetch_assoc();
    }
    
    public function getAllTemplates($category = null, $activeOnly = false) {
        $where = [];
        $params = [];
        $types = '';
        
        if ($category) {
            $where[] = "category = ?";
            $params[] = $category;
            $types .= 's';
        }
        
        if ($activeOnly) {
            $where[] = "is_active = 1";
        }
        
        $whereClause = empty($where) ? '' : 'WHERE ' . implode(' AND ', $where);
        
        $sql = "SELECT * FROM email_templates $whereClause ORDER BY category, name";
        
        $stmt = $this->conn->prepare($sql);
        if (!empty($params)) {
            $stmt->bind_param($types, ...$params);
        }
        $stmt->execute();
        
        return $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    }
    
    public function toggleTemplate($id) {
        $stmt = $this->conn->prepare("UPDATE email_templates SET is_active = NOT is_active WHERE id = ?");
        $stmt->bind_param("i", $id);
        
        if ($stmt->execute()) {
            return [
                'success' => true,
                'message' => 'Template status updated successfully'
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Failed to update template status: ' . $stmt->error
            ];
        }
    }
    
    public function createDefaultTemplates() {
        $templates = [
            [
                'name' => 'welcome',
                'subject' => 'Welcome to KSO Chandigarh - {{first_name}} {{last_name}}',
                'body' => $this->getWelcomeTemplate(),
                'description' => 'Welcome email sent to new members',
                'category' => 'membership'
            ],
            [
                'name' => 'approval',
                'subject' => 'Membership Approved - Welcome to KSO Chandigarh',
                'body' => $this->getApprovalTemplate(),
                'description' => 'Approval email sent when membership is approved',
                'category' => 'membership'
            ],
            [
                'name' => 'rejection',
                'subject' => 'Membership Application Status - KSO Chandigarh',
                'body' => $this->getRejectionTemplate(),
                'description' => 'Rejection email sent when membership is rejected',
                'category' => 'membership'
            ],
            [
                'name' => 'event-notification',
                'subject' => 'Event Invitation: {{event_name}}',
                'body' => $this->getEventNotificationTemplate(),
                'description' => 'Event notification email',
                'category' => 'events'
            ],
            [
                'name' => 'fee-reminder',
                'subject' => 'Fee Reminder - {{description}}',
                'body' => $this->getFeeReminderTemplate(),
                'description' => 'Fee reminder email',
                'category' => 'finance'
            ],
            [
                'name' => 'password-reset',
                'subject' => 'Password Reset - KSO Chandigarh',
                'body' => $this->getPasswordResetTemplate(),
                'description' => 'Password reset email',
                'category' => 'security'
            ]
        ];
        
        $results = [];
        foreach ($templates as $template) {
            $existing = $this->getTemplateByName($template['name']);
            if (!$existing) {
                $result = $this->createTemplate(
                    $template['name'],
                    $template['subject'],
                    $template['body'],
                    $template['description'],
                    $template['category']
                );
                $results[] = array_merge($result, ['template' => $template['name']]);
            } else {
                $results[] = [
                    'success' => true,
                    'message' => 'Template already exists',
                    'template' => $template['name']
                ];
            }
        }
        
        return $results;
    }
    
    private function getWelcomeTemplate() {
        return '
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Welcome to KSO Chandigarh</title>
    <style>
        body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
        .container { max-width: 600px; margin: 0 auto; padding: 20px; }
        .header { background: #28a745; color: white; padding: 20px; text-align: center; }
        .content { padding: 20px; background: #f8f9fa; }
        .footer { text-align: center; padding: 20px; font-size: 12px; color: #666; }
        .button { background: #007bff; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px; display: inline-block; margin: 10px 0; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>Welcome to {{organization_name}}</h1>
        </div>
        <div class="content">
            <h2>Dear {{first_name}} {{last_name}},</h2>
            <p>Welcome to the Kashmiri Samaj Organization (KSO) Chandigarh! We are delighted to have you as a member of our community.</p>
            
            <p><strong>Your Membership Details:</strong></p>
            <ul>
                <li>Member ID: {{kso_id}}</li>
                <li>Membership Type: {{membership_type}}</li>
                <li>Registration Date: {{current_date}}</li>
            </ul>
            
            <p>As a member, you now have access to:</p>
            <ul>
                <li>Exclusive member events and cultural programs</li>
                <li>Community networking opportunities</li>
                <li>Member directory and communication platform</li>
                <li>Special discounts on KSO events</li>
            </ul>
            
            <p>Please keep your membership ID safe as you will need it for all future transactions and events.</p>
            
            <a href="{{login_url}}" class="button">Login to Your Account</a>
            
            <p>If you have any questions or need assistance, please don\'t hesitate to contact us at {{support_email}}.</p>
            
            <p>Welcome aboard!</p>
            <p>The KSO Chandigarh Team</p>
        </div>
        <div class="footer">
            <p>&copy; {{current_year}} {{organization_name}}. All rights reserved.</p>
            <p>Visit us at: {{website_url}}</p>
        </div>
    </div>
</body>
</html>';
    }
    
    private function getApprovalTemplate() {
        return '
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Membership Approved</title>
    <style>
        body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
        .container { max-width: 600px; margin: 0 auto; padding: 20px; }
        .header { background: #28a745; color: white; padding: 20px; text-align: center; }
        .content { padding: 20px; background: #f8f9fa; }
        .footer { text-align: center; padding: 20px; font-size: 12px; color: #666; }
        .button { background: #007bff; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px; display: inline-block; margin: 10px 0; }
        .success { background: #d4edda; border: 1px solid #c3e6cb; color: #155724; padding: 15px; border-radius: 5px; margin: 15px 0; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>Membership Approved!</h1>
        </div>
        <div class="content">
            <div class="success">
                <strong>Congratulations!</strong> Your membership application has been approved.
            </div>
            
            <h2>Dear {{first_name}} {{last_name}},</h2>
            <p>We are pleased to inform you that your membership application to {{organization_name}} has been approved!</p>
            
            <p><strong>Your Membership Details:</strong></p>
            <ul>
                <li>Member ID: {{kso_id}}</li>
                <li>Approval Date: {{current_date}}</li>
                <li>Status: Active</li>
            </ul>
            
            <p>You can now access all member benefits including:</p>
            <ul>
                <li>Participation in all KSO events and activities</li>
                <li>Access to member directory</li>
                <li>Voting rights in organizational matters</li>
                <li>Priority booking for events</li>
            </ul>
            
            <a href="{{login_url}}" class="button">Access Your Account</a>
            
            <p>We look forward to your active participation in our community activities.</p>
            
            <p>Best regards,<br>
            The KSO Chandigarh Team</p>
        </div>
        <div class="footer">
            <p>&copy; {{current_year}} {{organization_name}}. All rights reserved.</p>
            <p>Visit us at: {{website_url}}</p>
        </div>
    </div>
</body>
</html>';
    }
    
    private function getRejectionTemplate() {
        return '
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Membership Application Status</title>
    <style>
        body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
        .container { max-width: 600px; margin: 0 auto; padding: 20px; }
        .header { background: #dc3545; color: white; padding: 20px; text-align: center; }
        .content { padding: 20px; background: #f8f9fa; }
        .footer { text-align: center; padding: 20px; font-size: 12px; color: #666; }
        .info { background: #fff3cd; border: 1px solid #ffeaa7; color: #856404; padding: 15px; border-radius: 5px; margin: 15px 0; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>Membership Application Status</h1>
        </div>
        <div class="content">
            <h2>Dear {{first_name}} {{last_name}},</h2>
            <p>Thank you for your interest in joining {{organization_name}}.</p>
            
            <div class="info">
                We regret to inform you that your membership application could not be approved at this time.
            </div>
            
            <p><strong>Reason:</strong> {{reason}}</p>
            
            <p>Please note that this decision does not prevent you from reapplying in the future. We encourage you to:</p>
            <ul>
                <li>Review our membership requirements</li>
                <li>Address any issues mentioned above</li>
                <li>Reapply when you meet all criteria</li>
            </ul>
            
            <p>For any questions regarding this decision or the reapplication process, please contact us at {{support_email}}.</p>
            
            <p>Thank you for your understanding.</p>
            
            <p>Best regards,<br>
            The KSO Chandigarh Team</p>
        </div>
        <div class="footer">
            <p>&copy; {{current_year}} {{organization_name}}. All rights reserved.</p>
            <p>Visit us at: {{website_url}}</p>
        </div>
    </div>
</body>
</html>';
    }
    
    private function getEventNotificationTemplate() {
        return '
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Event Invitation</title>
    <style>
        body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
        .container { max-width: 600px; margin: 0 auto; padding: 20px; }
        .header { background: #17a2b8; color: white; padding: 20px; text-align: center; }
        .content { padding: 20px; background: #f8f9fa; }
        .footer { text-align: center; padding: 20px; font-size: 12px; color: #666; }
        .button { background: #007bff; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px; display: inline-block; margin: 10px 0; }
        .event-details { background: white; padding: 20px; border-radius: 5px; margin: 20px 0; border-left: 4px solid #17a2b8; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>Event Invitation</h1>
        </div>
        <div class="content">
            <h2>Dear {{first_name}} {{last_name}},</h2>
            <p>You are cordially invited to attend our upcoming event!</p>
            
            <div class="event-details">
                <h3>{{event_name}}</h3>
                <p><strong>Date:</strong> {{event_date}}</p>
                <p><strong>Time:</strong> {{event_time}}</p>
                <p><strong>Location:</strong> {{event_location}}</p>
                <p><strong>Description:</strong></p>
                <p>{{event_description}}</p>
            </div>
            
            <p>This is an exclusive event for KSO Chandigarh members and their families. Don\'t miss this opportunity to connect with fellow community members.</p>
            
            <a href="{{registration_url}}" class="button">Register Now</a>
            
            <p>Please RSVP by the registration deadline to secure your spot.</p>
            
            <p>Looking forward to seeing you there!</p>
            
            <p>Best regards,<br>
            The KSO Chandigarh Events Team</p>
        </div>
        <div class="footer">
            <p>&copy; {{current_year}} {{organization_name}}. All rights reserved.</p>
            <p>Visit us at: {{website_url}}</p>
        </div>
    </div>
</body>
</html>';
    }
    
    private function getFeeReminderTemplate() {
        return '
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Fee Reminder</title>
    <style>
        body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
        .container { max-width: 600px; margin: 0 auto; padding: 20px; }
        .header { background: #ffc107; color: #212529; padding: 20px; text-align: center; }
        .content { padding: 20px; background: #f8f9fa; }
        .footer { text-align: center; padding: 20px; font-size: 12px; color: #666; }
        .button { background: #28a745; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px; display: inline-block; margin: 10px 0; }
        .fee-details { background: white; padding: 20px; border-radius: 5px; margin: 20px 0; border-left: 4px solid #ffc107; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>Fee Reminder</h1>
        </div>
        <div class="content">
            <h2>Dear {{first_name}} {{last_name}},</h2>
            <p>This is a friendly reminder about your upcoming fee payment.</p>
            
            <div class="fee-details">
                <h3>{{description}}</h3>
                <p><strong>Amount:</strong> ₹{{amount}}</p>
                <p><strong>Due Date:</strong> {{due_date}}</p>
            </div>
            
            <p>Please ensure timely payment to avoid any inconvenience. You can make the payment through:</p>
            <ul>
                <li>Online payment portal</li>
                <li>Bank transfer</li>
                <li>Visit our office</li>
            </ul>
            
            <a href="{{payment_url}}" class="button">Pay Now</a>
            
            <p>If you have already made the payment, please disregard this reminder.</p>
            
            <p>For any payment-related queries, contact us at {{support_email}}.</p>
            
            <p>Thank you for your cooperation.</p>
            
            <p>Best regards,<br>
            The KSO Chandigarh Finance Team</p>
        </div>
        <div class="footer">
            <p>&copy; {{current_year}} {{organization_name}}. All rights reserved.</p>
            <p>Visit us at: {{website_url}}</p>
        </div>
    </div>
</body>
</html>';
    }
    
    private function getPasswordResetTemplate() {
        return '
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Password Reset</title>
    <style>
        body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
        .container { max-width: 600px; margin: 0 auto; padding: 20px; }
        .header { background: #6f42c1; color: white; padding: 20px; text-align: center; }
        .content { padding: 20px; background: #f8f9fa; }
        .footer { text-align: center; padding: 20px; font-size: 12px; color: #666; }
        .button { background: #007bff; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px; display: inline-block; margin: 10px 0; }
        .warning { background: #f8d7da; border: 1px solid #f5c6cb; color: #721c24; padding: 15px; border-radius: 5px; margin: 15px 0; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>Password Reset Request</h1>
        </div>
        <div class="content">
            <h2>Dear {{first_name}} {{last_name}},</h2>
            <p>We received a request to reset your password for your KSO Chandigarh account.</p>
            
            <div class="warning">
                <strong>Security Notice:</strong> If you did not request this password reset, please ignore this email and contact us immediately.
            </div>
            
            <p>To reset your password, click the button below:</p>
            
            <a href="{{reset_url}}" class="button">Reset Password</a>
            
            <p>This link will expire in 24 hours for security reasons.</p>
            
            <p>If you\'re having trouble clicking the button, copy and paste the URL below into your web browser:</p>
            <p>{{reset_url}}</p>
            
            <p>For security reasons, please:</p>
            <ul>
                <li>Choose a strong password</li>
                <li>Don\'t share your password with anyone</li>
                <li>Change your password regularly</li>
            </ul>
            
            <p>If you need assistance, contact us at {{support_email}}.</p>
            
            <p>Best regards,<br>
            The KSO Chandigarh Security Team</p>
        </div>
        <div class="footer">
            <p>&copy; {{current_year}} {{organization_name}}. All rights reserved.</p>
            <p>Visit us at: {{website_url}}</p>
        </div>
    </div>
</body>
</html>';
    }
    
    public function previewTemplate($templateData, $sampleData = []) {
        $defaultSampleData = [
            'first_name' => 'John',
            'last_name' => 'Doe',
            'kso_id' => 'KSO2024001',
            'membership_type' => 'Annual',
            'event_name' => 'Sample Event',
            'event_date' => 'January 15, 2024',
            'event_time' => '6:00 PM',
            'event_location' => 'Community Hall',
            'event_description' => 'This is a sample event description.',
            'amount' => '1000',
            'due_date' => 'January 31, 2024',
            'description' => 'Annual Membership Fee',
            'reason' => 'Incomplete documentation',
            'current_date' => date('Y-m-d'),
            'current_time' => date('H:i:s'),
            'current_year' => date('Y'),
            'organization_name' => 'KSO Chandigarh',
            'website_url' => 'https://ksochandigarh.com',
            'support_email' => 'support@ksochandigarh.com',
            'login_url' => 'https://ksochandigarh.com/login',
            'payment_url' => 'https://ksochandigarh.com/payment',
            'registration_url' => 'https://ksochandigarh.com/event-registration',
            'reset_url' => 'https://ksochandigarh.com/reset-password'
        ];
        
        $data = array_merge($defaultSampleData, $sampleData);
        
        $subject = $templateData['subject'];
        $body = $templateData['body'];
        
        foreach ($data as $key => $value) {
            $subject = str_replace('{{' . $key . '}}', $value, $subject);
            $body = str_replace('{{' . $key . '}}', $value, $body);
        }
        
        return [
            'subject' => $subject,
            'body' => $body
        ];
    }
    
    public function getAvailableVariables() {
        return [
            'member' => [
                'first_name' => 'Member\'s first name',
                'last_name' => 'Member\'s last name',
                'kso_id' => 'KSO membership ID',
                'membership_type' => 'Type of membership',
                'email' => 'Member\'s email address'
            ],
            'event' => [
                'event_name' => 'Event title',
                'event_date' => 'Event date',
                'event_time' => 'Event time',
                'event_location' => 'Event location',
                'event_description' => 'Event description',
                'registration_url' => 'Event registration URL'
            ],
            'finance' => [
                'amount' => 'Fee amount',
                'due_date' => 'Payment due date',
                'description' => 'Fee description',
                'payment_url' => 'Payment URL'
            ],
            'system' => [
                'current_date' => 'Current date',
                'current_time' => 'Current time',
                'current_year' => 'Current year',
                'organization_name' => 'Organization name',
                'website_url' => 'Website URL',
                'support_email' => 'Support email',
                'login_url' => 'Login URL'
            ]
        ];
    }
}
