<?php
/**
 * KSO Chandigarh Management System - API Configuration
 * Compatible with local development and Razorhost Starter Plan
 */

// Prevent direct file access
if (!defined('API_ACCESS')) {
    die('Direct access denied');
}

// Set JSON response headers
header('Content-Type: application/json; charset=utf-8');
// Allow CORS from requesting origin and support credentials
header('Access-Control-Allow-Origin: ' . ($_SERVER['HTTP_ORIGIN'] ?? '*'));
header('Access-Control-Allow-Credentials: true');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle preflight requests
if (isset($_SERVER['REQUEST_METHOD']) && $_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Production flag - set to true when deployed to Razorhost
$is_production = false;

// Database driver check - detect if PDO MySQL driver is available
$mysql_driver_available = extension_loaded('pdo_mysql');

// Path adjustments for hosting
$base_path = dirname(dirname(__FILE__));
$storage_path = $base_path . '/storage';
$upload_path = $base_path . '/uploads';
$log_path = $base_path . '/logs';

// Ensure these directories exist and are writable
foreach ([$storage_path, $upload_path, $log_path] as $dir) {
    if (!file_exists($dir)) {
        mkdir($dir, 0755, true);
    }
}

// Database Configuration - Update these for Razorhost
$db_config = [
    'host' => 'localhost',      // Usually 'localhost' on shared hosting
    'dbname' => $is_production ? 'ktxmmosu_kso_chandigarh' : 'ktxmmosu_kso_chandigarh',
    'username' => $is_production ? 'ktxmmosu_kso_db' : 'root',
    'password' => $is_production ? 'KSO@2025' : '',
    'charset' => 'utf8mb4'
];

// Initialize PDO connection or null if not available
$pdo = null;

// Create PDO connection with error handling only if the driver is available
if ($mysql_driver_available) {
    try {
        $dsn = "mysql:host={$db_config['host']};dbname={$db_config['dbname']};charset={$db_config['charset']}";
        $pdo = new PDO($dsn, $db_config['username'], $db_config['password'], [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false
        ]);
    } catch (PDOException $e) {
        // Log the error but don't terminate since we'll use mock data
        error_log("Database connection failed: " . $e->getMessage());
        $pdo = null;
    }
} else {
    // Log that the driver is not available
    error_log("PDO MySQL driver not available - using mock data mode");
}

// Helper function to send JSON response
function sendResponse($code, $success, $message, $data = null) {
    http_response_code($code);
    $response = [
        'success' => $success,
        'message' => $message
    ];
    
    if ($data !== null) {
        $response['data'] = $data;
    }
    
    echo json_encode($response);
    exit();
}

// Helper function to get JSON input
function getJsonInput() {
    $input = file_get_contents('php://input');
    return json_decode($input, true) ?? [];
}

// Shared hosting file path helper functions
function getStoragePath($subdir = '') {
    global $storage_path;
    $path = $storage_path;
    if (!empty($subdir)) {
        $path .= '/' . trim($subdir, '/');
        if (!file_exists($path)) {
            mkdir($path, 0755, true);
        }
    }
    return $path;
}

function getUploadPath($subdir = '') {
    global $upload_path;
    $path = $upload_path;
    if (!empty($subdir)) {
        $path .= '/' . trim($subdir, '/');
        if (!file_exists($path)) {
            mkdir($path, 0755, true);
        }
    }
    return $path;
}

function getLogPath($filename = '') {
    global $log_path;
    if (!empty($filename)) {
        return $log_path . '/' . $filename;
    }
    return $log_path;
}

// Simple session management with improved security
function startSession() {
    if (session_status() === PHP_SESSION_NONE) {
        // Secure session settings
        ini_set('session.cookie_httponly', 1);
        
        // Don't require HTTPS for local development
        if (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') {
            ini_set('session.cookie_secure', 1);
        }
        
        // Set a specific session name to avoid conflicts
        session_name('KSO_CHANDIGARH');
        
        // Debug session
        error_log("Starting session with name: " . session_name());
        
        session_start();
    }
}

// Check if user is logged in
function isLoggedIn() {
    startSession();
    return isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in'] === true;
}

// Get current admin data
function getCurrentAdmin() {
    startSession();
    if (!isLoggedIn()) {
        return null;
    }
    
    return [
        'id' => $_SESSION['admin_id'] ?? null,
        'username' => $_SESSION['admin_username'] ?? null,
        'email' => $_SESSION['admin_email'] ?? null,
        'name' => $_SESSION['admin_name'] ?? null,
        'role' => $_SESSION['admin_role'] ?? null
    ];
}

// Write to log file safely
function logError($message, $context = []) {
    global $is_production;
    
    $logEntry = date('Y-m-d H:i:s') . ' - ' . $message;
    if (!empty($context)) {
        $logEntry .= ' - ' . json_encode($context);
    }
    
    // In production, write to log file, in development use error_log
    if ($is_production) {
        $log_file = getLogPath('error_' . date('Y-m-d') . '.log');
        file_put_contents($log_file, $logEntry . PHP_EOL, FILE_APPEND);
    } else {
        error_log($logEntry);
    }
}

// Simple activity logging
function logActivity($adminId, $action, $details = '', $context = []) {
    global $is_production;
    
    // Basic activity logging - can be enhanced later
    $logEntry = date('Y-m-d H:i:s') . " - Admin {$adminId} - {$action} - {$details}";
    if (!empty($context)) {
        $logEntry .= ' - ' . json_encode($context);
    }
    
    // In production, write to log file, in development use error_log
    if ($is_production) {
        $log_file = getLogPath('activity_' . date('Y-m-d') . '.log');
        file_put_contents($log_file, $logEntry . PHP_EOL, FILE_APPEND);
    } else {
        error_log($logEntry);
    }
}

// Require authentication function
function requireAuth() {
    if (!isLoggedIn()) {
        http_response_code(401);
        echo json_encode(['success' => false, 'message' => 'Authentication required']);
        exit();
    }
}

// Generate unique ID compatible with shared hosting
function generateUniqueId($prefix = '') {
    return uniqid($prefix, true);
}

// Get current user's IP address (accounting for shared hosting and proxies)
function getClientIp() {
    $ip = '0.0.0.0';
    
    if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
        $ip = $_SERVER['HTTP_CLIENT_IP'];
    } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
        $ip = $_SERVER['HTTP_X_FORWARDED_FOR'];
    } elseif (!empty($_SERVER['REMOTE_ADDR'])) {
        $ip = $_SERVER['REMOTE_ADDR'];
    }
    
    return filter_var($ip, FILTER_VALIDATE_IP) ? $ip : '0.0.0.0';
}

// Global timestamp function (timezone-aware)
function getCurrentTimestamp() {
    date_default_timezone_set('Asia/Kolkata'); // India timezone
    return date('Y-m-d H:i:s');
}
?>
