<?php
/**
 * Email Service Class for KSO Chandigarh Management System
 * Handles all email functionality including SMTP configuration, templates, and sending
 */

require_once '../config.php';

class EmailService {
    private $pdo;
    private $config;
    private $templates_dir;
    
    public function __construct() {
        global $pdo;
        $this->pdo = $pdo;
        $this->templates_dir = __DIR__ . '/../templates/email/';
        $this->loadConfig();
    }
    
    /**
     * Load email configuration from database settings
     */
    private function loadConfig() {
        try {
            $stmt = $this->pdo->prepare("SELECT setting_key, setting_value FROM settings WHERE setting_key LIKE 'email_%' OR setting_key LIKE 'smtp_%'");
            $stmt->execute();
            $settings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
            
            $this->config = [
                'smtp_host' => $settings['smtp_host'] ?? 'localhost',
                'smtp_port' => $settings['smtp_port'] ?? 587,
                'smtp_username' => $settings['smtp_username'] ?? '',
                'smtp_password' => $settings['smtp_password'] ?? '',
                'smtp_encryption' => $settings['smtp_encryption'] ?? 'tls',
                'from_email' => $settings['email_from_address'] ?? 'noreply@ksochandigarh.com',
                'from_name' => $settings['email_from_name'] ?? 'KSO Chandigarh',
                'enabled' => $settings['email_enabled'] ?? '1'
            ];
        } catch (Exception $e) {
            error_log("Email config load error: " . $e->getMessage());
            $this->config = $this->getDefaultConfig();
        }
    }
    
    /**
     * Get default email configuration
     */
    private function getDefaultConfig() {
        return [
            'smtp_host' => 'localhost',
            'smtp_port' => 587,
            'smtp_username' => '',
            'smtp_password' => '',
            'smtp_encryption' => 'tls',
            'from_email' => 'noreply@ksochandigarh.com',
            'from_name' => 'KSO Chandigarh',
            'enabled' => '0'
        ];
    }
    
    /**
     * Send a simple email using PHPMailer
     */
    public function sendEmail($to, $subject, $body, $isHTML = true, $priority = 'normal') {
        if ($this->config['enabled'] !== '1') {
            $this->logEmail($to, $subject, '', 'disabled', 'Email service is disabled');
            return false;
        }
        
        try {
            // Use mail() function with proper headers for now
            // In production, consider using PHPMailer for better SMTP support
            
            $headers = $this->buildHeaders($isHTML, $priority);
            
            // Convert HTML to plain text for fallback
            $plainBody = $isHTML ? strip_tags($body) : $body;
            
            $success = mail($to, $subject, $body, $headers);
            
            $status = $success ? 'sent' : 'failed';
            $error = $success ? null : 'Mail function failed';
            
            $this->logEmail($to, $subject, '', $status, $error);
            
            return $success;
            
        } catch (Exception $e) {
            $this->logEmail($to, $subject, '', 'failed', $e->getMessage());
            error_log("Email send error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Build email headers
     */
    private function buildHeaders($isHTML = true, $priority = 'normal') {
        $headers = [];
        
        // From header
        $headers[] = 'From: ' . $this->config['from_name'] . ' <' . $this->config['from_email'] . '>';
        $headers[] = 'Reply-To: ' . $this->config['from_email'];
        
        // Content type
        if ($isHTML) {
            $headers[] = 'Content-Type: text/html; charset=UTF-8';
        } else {
            $headers[] = 'Content-Type: text/plain; charset=UTF-8';
        }
        
        // Priority
        switch ($priority) {
            case 'urgent':
                $headers[] = 'X-Priority: 1';
                $headers[] = 'Importance: High';
                break;
            case 'high':
                $headers[] = 'X-Priority: 2';
                $headers[] = 'Importance: High';
                break;
            case 'low':
                $headers[] = 'X-Priority: 4';
                $headers[] = 'Importance: Low';
                break;
            default:
                $headers[] = 'X-Priority: 3';
                $headers[] = 'Importance: Normal';
        }
        
        // Additional headers
        $headers[] = 'X-Mailer: KSO Chandigarh Management System';
        $headers[] = 'MIME-Version: 1.0';
        
        return implode("\r\n", $headers);
    }
    
    /**
     * Send email using template
     */
    public function sendTemplate($template, $data, $to, $subject, $priority = 'normal') {
        $body = $this->renderTemplate($template, $data);
        if ($body === false) {
            $this->logEmail($to, $subject, $template, 'failed', 'Template not found or render error');
            return false;
        }
        
        $result = $this->sendEmail($to, $subject, $body, true, $priority);
        
        if ($result) {
            $this->logEmail($to, $subject, $template, 'sent');
        }
        
        return $result;
    }
    
    /**
     * Send bulk emails
     */
    public function sendBulkEmail($recipients, $subject, $body, $template = null) {
        $results = [];
        $success_count = 0;
        $failed_count = 0;
        
        foreach ($recipients as $recipient) {
            $email = is_array($recipient) ? $recipient['email'] : $recipient;
            $data = is_array($recipient) ? $recipient : [];
            
            if ($template) {
                $result = $this->sendTemplate($template, $data, $email, $subject);
            } else {
                $result = $this->sendEmail($email, $subject, $body);
            }
            
            $results[] = ['email' => $email, 'success' => $result];
            
            if ($result) {
                $success_count++;
            } else {
                $failed_count++;
            }
            
            // Add small delay to prevent spam issues
            usleep(100000); // 0.1 second
        }
        
        return [
            'total' => count($recipients),
            'success' => $success_count,
            'failed' => $failed_count,
            'results' => $results
        ];
    }
    
    /**
     * Render email template
     */
    private function renderTemplate($template, $data) {
        $template_file = $this->templates_dir . $template . '.html';
        
        if (!file_exists($template_file)) {
            error_log("Email template not found: " . $template_file);
            return false;
        }
        
        $content = file_get_contents($template_file);
        
        // Replace template variables
        foreach ($data as $key => $value) {
            $content = str_replace('{{' . $key . '}}', htmlspecialchars($value), $content);
        }
        
        // Replace system variables
        $system_vars = [
            'system_name' => 'KSO Chandigarh Management System',
            'current_year' => date('Y'),
            'current_date' => date('F j, Y'),
            'organization_name' => 'Kashmiri Students Organization Chandigarh',
            'website_url' => 'https://ksochandigarh.com'
        ];
        
        foreach ($system_vars as $key => $value) {
            $content = str_replace('{{' . $key . '}}', $value, $content);
        }
        
        return $content;
    }
    
    /**
     * Build email headers
     */
    private function buildHeaders($isHTML = true) {
        $headers = [];
        $headers[] = 'From: ' . $this->config['from_name'] . ' <' . $this->config['from_email'] . '>';
        $headers[] = 'Reply-To: ' . $this->config['from_email'];
        $headers[] = 'X-Mailer: KSO Chandigarh Management System';
        
        if ($isHTML) {
            $headers[] = 'MIME-Version: 1.0';
            $headers[] = 'Content-Type: text/html; charset=UTF-8';
        }
        
        return implode("\r\n", $headers);
    }
    
    /**
     * Log email activity
     */
    private function logEmail($to, $subject, $template, $status, $error = null) {
        try {
            $stmt = $this->pdo->prepare("
                INSERT INTO email_logs (to_email, subject, template_used, status, error_message, sent_at) 
                VALUES (?, ?, ?, ?, ?, NOW())
            ");
            $stmt->execute([$to, $subject, $template, $status, $error]);
        } catch (Exception $e) {
            error_log("Email log error: " . $e->getMessage());
        }
    }
    
    /**
     * Test email configuration
     */
    public function testEmail($test_email) {
        $subject = 'KSO Email Configuration Test';
        $body = '
            <h2>Email Test Successful!</h2>
            <p>This is a test email from the KSO Chandigarh Management System.</p>
            <p><strong>Sent at:</strong> ' . date('Y-m-d H:i:s') . '</p>
            <p>If you receive this email, your email configuration is working correctly.</p>
        ';
        
        return $this->sendEmail($test_email, $subject, $body);
    }
    
    /**
     * Get email statistics
     */
    public function getStats() {
        try {
            $stmt = $this->pdo->query("
                SELECT 
                    COUNT(*) as total,
                    SUM(CASE WHEN status = 'sent' THEN 1 ELSE 0 END) as sent,
                    SUM(CASE WHEN status = 'failed' THEN 1 ELSE 0 END) as failed,
                    SUM(CASE WHEN DATE(sent_at) = CURDATE() THEN 1 ELSE 0 END) as today,
                    SUM(CASE WHEN DATE(sent_at) >= DATE_SUB(CURDATE(), INTERVAL 7 DAY) THEN 1 ELSE 0 END) as this_week,
                    SUM(CASE WHEN DATE(sent_at) >= DATE_SUB(CURDATE(), INTERVAL 30 DAY) THEN 1 ELSE 0 END) as this_month
                FROM email_logs
            ");
            
            $stats = $stmt->fetch(PDO::FETCH_ASSOC);
            $stats['success_rate'] = $stats['total'] > 0 ? round(($stats['sent'] / $stats['total']) * 100, 2) : 0;
            
            return $stats;
        } catch (Exception $e) {
            error_log("Email stats error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get recent email logs
     */
    public function getRecentLogs($limit = 50) {
        try {
            $stmt = $this->pdo->prepare("
                SELECT to_email, subject, template_used, status, error_message, sent_at 
                FROM email_logs 
                ORDER BY sent_at DESC 
                LIMIT ?
            ");
            $stmt->execute([$limit]);
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            error_log("Email logs error: " . $e->getMessage());
            return [];
        }
    }
}

/**
 * Email notification helper functions
 */
class EmailNotifications {
    private static $emailService;
    
    private static function getEmailService() {
        if (self::$emailService === null) {
            self::$emailService = new EmailService();
        }
        return self::$emailService;
    }
    
    /**
     * Send welcome email to new member
     */
    public static function sendWelcomeEmail($member) {
        $data = [
            'first_name' => $member['first_name'],
            'last_name' => $member['last_name'],
            'kso_id' => $member['kso_id'],
            'membership_type' => $member['membership_type']
        ];
        
        return self::getEmailService()->sendTemplate(
            'welcome',
            $data,
            $member['email'],
            'Welcome to KSO Chandigarh!'
        );
    }
    
    /**
     * Send member approval notification
     */
    public static function sendApprovalEmail($member) {
        $data = [
            'first_name' => $member['first_name'],
            'last_name' => $member['last_name'],
            'kso_id' => $member['kso_id']
        ];
        
        return self::getEmailService()->sendTemplate(
            'approval',
            $data,
            $member['email'],
            'Your KSO Membership has been Approved!'
        );
    }
    
    /**
     * Send member rejection notification
     */
    public static function sendRejectionEmail($member, $reason = '') {
        $data = [
            'first_name' => $member['first_name'],
            'last_name' => $member['last_name'],
            'reason' => $reason
        ];
        
        return self::getEmailService()->sendTemplate(
            'rejection',
            $data,
            $member['email'],
            'KSO Membership Application Update'
        );
    }
    
    /**
     * Send event reminder
     */
    public static function sendEventReminder($member, $event) {
        $data = [
            'first_name' => $member['first_name'],
            'event_title' => $event['event_title'],
            'event_date' => $event['event_date'],
            'event_time' => $event['event_time'],
            'event_location' => $event['event_location']
        ];
        
        return self::getEmailService()->sendTemplate(
            'event-reminder',
            $data,
            $member['email'],
            'Reminder: ' . $event['event_title']
        );
    }
    
    /**
     * Send fee reminder
     */
    public static function sendFeeReminder($member, $fee_amount, $due_date) {
        $data = [
            'first_name' => $member['first_name'],
            'last_name' => $member['last_name'],
            'kso_id' => $member['kso_id'],
            'fee_amount' => $fee_amount,
            'due_date' => $due_date
        ];
        
        return self::getEmailService()->sendTemplate(
            'fee-reminder',
            $data,
            $member['email'],
            'Membership Fee Payment Reminder'
        );
    }
}
?>
