<?php
/**
 * Enhanced Photo Upload Service for KSO Chandigarh
 * Supports chunked uploads, large files, fast previews, and optimization
 */

require_once '../config.php';
require_once '../security/security.php';

class EnhancedPhotoService {
    private $pdo;
    private $upload_dir;
    private $temp_dir;
    private $max_file_size;
    private $allowed_types;
    
    public function __construct() {
        global $pdo;
        $this->pdo = $pdo;
        $this->upload_dir = __DIR__ . '/../../uploads/member-photos/';
        $this->temp_dir = __DIR__ . '/../../uploads/temp/';
        $this->max_file_size = 50 * 1024 * 1024; // 50MB
        $this->allowed_types = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
        
        $this->ensureDirectories();
    }
    
    /**
     * Ensure upload directories exist
     */
    private function ensureDirectories() {
        $directories = [
            $this->upload_dir,
            $this->temp_dir,
            $this->upload_dir . 'thumbnails/',
            $this->upload_dir . 'optimized/',
            $this->upload_dir . 'originals/'
        ];
        
        foreach ($directories as $dir) {
            if (!is_dir($dir)) {
                mkdir($dir, 0755, true);
            }
        }
    }
    
    /**
     * Handle chunked file upload
     */
    public function handleChunkedUpload($file_data, $chunk_info) {
        try {
            $chunk_index = $chunk_info['chunk_index'];
            $total_chunks = $chunk_info['total_chunks'];
            $file_id = $chunk_info['file_id'];
            $original_name = $chunk_info['original_name'];
            
            // Validate chunk data
            if (!$this->validateChunk($file_data, $chunk_info)) {
                return ['success' => false, 'error' => 'Invalid chunk data'];
            }
            
            // Save chunk to temporary directory
            $temp_file = $this->temp_dir . $file_id . '.part' . $chunk_index;
            if (file_put_contents($temp_file, $file_data) === false) {
                return ['success' => false, 'error' => 'Failed to save chunk'];
            }
            
            // If this is the last chunk, assemble the file
            if ($chunk_index == $total_chunks - 1) {
                return $this->assembleChunkedFile($file_id, $total_chunks, $original_name);
            }
            
            return [
                'success' => true, 
                'message' => 'Chunk uploaded successfully',
                'progress' => round(($chunk_index + 1) / $total_chunks * 100, 2)
            ];
            
        } catch (Exception $e) {
            error_log("Chunked upload error: " . $e->getMessage());
            return ['success' => false, 'error' => 'Upload failed'];
        }
    }
    
    /**
     * Validate chunk data
     */
    private function validateChunk($file_data, $chunk_info) {
        // Check chunk size (max 5MB per chunk)
        if (strlen($file_data) > 5 * 1024 * 1024) {
            return false;
        }
        
        // Validate chunk index
        if ($chunk_info['chunk_index'] < 0 || $chunk_info['chunk_index'] >= $chunk_info['total_chunks']) {
            return false;
        }
        
        // Validate file ID format
        if (!preg_match('/^[a-zA-Z0-9_-]+$/', $chunk_info['file_id'])) {
            return false;
        }
        
        return true;
    }
    
    /**
     * Assemble chunked file
     */
    private function assembleChunkedFile($file_id, $total_chunks, $original_name) {
        $assembled_file = $this->temp_dir . $file_id . '.assembled';
        $handle = fopen($assembled_file, 'wb');
        
        if (!$handle) {
            return ['success' => false, 'error' => 'Failed to create assembled file'];
        }
        
        // Combine all chunks
        for ($i = 0; $i < $total_chunks; $i++) {
            $chunk_file = $this->temp_dir . $file_id . '.part' . $i;
            if (!file_exists($chunk_file)) {
                fclose($handle);
                return ['success' => false, 'error' => 'Missing chunk: ' . $i];
            }
            
            $chunk_data = file_get_contents($chunk_file);
            fwrite($handle, $chunk_data);
            unlink($chunk_file); // Clean up chunk
        }
        
        fclose($handle);
        
        // Validate and process the assembled file
        return $this->processUploadedFile($assembled_file, $original_name);
    }
    
    /**
     * Process uploaded file (validation, optimization, thumbnails)
     */
    private function processUploadedFile($temp_file, $original_name) {
        // Validate file
        $validation = $this->validateUploadedFile($temp_file, $original_name);
        if (!$validation['valid']) {
            unlink($temp_file);
            return ['success' => false, 'error' => $validation['error']];
        }
        
        // Generate secure filename
        $file_extension = strtolower(pathinfo($original_name, PATHINFO_EXTENSION));
        $secure_filename = uniqid('photo_', true) . '_' . time() . '.' . $file_extension;
        
        // File paths
        $original_path = $this->upload_dir . 'originals/' . $secure_filename;
        $optimized_path = $this->upload_dir . 'optimized/' . $secure_filename;
        $thumbnail_path = $this->upload_dir . 'thumbnails/' . $secure_filename;
        
        // Move original file
        if (!move_uploaded_file($temp_file, $original_path)) {
            if (!rename($temp_file, $original_path)) {
                return ['success' => false, 'error' => 'Failed to save original file'];
            }
        }
        
        // Generate optimized version (max 1920x1080, 85% quality)
        $optimized_result = $this->createOptimizedImage($original_path, $optimized_path, 1920, 1080, 85);
        
        // Generate thumbnail (300x300)
        $thumbnail_result = $this->createThumbnail($original_path, $thumbnail_path, 300, 300);
        
        // Generate WebP versions for modern browsers
        $webp_optimized = $this->upload_dir . 'optimized/' . pathinfo($secure_filename, PATHINFO_FILENAME) . '.webp';
        $webp_thumbnail = $this->upload_dir . 'thumbnails/' . pathinfo($secure_filename, PATHINFO_FILENAME) . '.webp';
        
        $this->createWebPVersion($optimized_path, $webp_optimized);
        $this->createWebPVersion($thumbnail_path, $webp_thumbnail);
        
        // Save to database
        $photo_record = $this->savePhotoRecord($secure_filename, $original_name, filesize($original_path));
        
        return [
            'success' => true,
            'photo_id' => $photo_record['id'],
            'filename' => $secure_filename,
            'thumbnail_url' => '/uploads/member-photos/thumbnails/' . $secure_filename,
            'optimized_url' => '/uploads/member-photos/optimized/' . $secure_filename,
            'webp_thumbnail' => '/uploads/member-photos/thumbnails/' . pathinfo($secure_filename, PATHINFO_FILENAME) . '.webp',
            'webp_optimized' => '/uploads/member-photos/optimized/' . pathinfo($secure_filename, PATHINFO_FILENAME) . '.webp',
            'file_size' => filesize($original_path),
            'dimensions' => $this->getImageDimensions($original_path)
        ];
    }
    
    /**
     * Validate uploaded file
     */
    private function validateUploadedFile($file_path, $original_name) {
        // Check file exists
        if (!file_exists($file_path)) {
            return ['valid' => false, 'error' => 'File does not exist'];
        }
        
        // Check file size
        $file_size = filesize($file_path);
        if ($file_size > $this->max_file_size) {
            return ['valid' => false, 'error' => 'File too large (max 50MB)'];
        }
        
        if ($file_size < 1024) { // Minimum 1KB
            return ['valid' => false, 'error' => 'File too small'];
        }
        
        // Check MIME type
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mime_type = finfo_file($finfo, $file_path);
        finfo_close($finfo);
        
        if (!in_array($mime_type, $this->allowed_types)) {
            return ['valid' => false, 'error' => 'Invalid file type. Only JPEG, PNG, GIF, and WebP allowed.'];
        }
        
        // Check if it's actually an image
        $image_info = getimagesize($file_path);
        if ($image_info === false) {
            return ['valid' => false, 'error' => 'Invalid image file'];
        }
        
        // Check image dimensions (minimum 100x100, maximum 10000x10000)
        if ($image_info[0] < 100 || $image_info[1] < 100) {
            return ['valid' => false, 'error' => 'Image too small (minimum 100x100 pixels)'];
        }
        
        if ($image_info[0] > 10000 || $image_info[1] > 10000) {
            return ['valid' => false, 'error' => 'Image too large (maximum 10000x10000 pixels)'];
        }
        
        return ['valid' => true];
    }
    
    /**
     * Create optimized image
     */
    private function createOptimizedImage($source_path, $dest_path, $max_width, $max_height, $quality) {
        $image_info = getimagesize($source_path);
        $original_width = $image_info[0];
        $original_height = $image_info[1];
        $image_type = $image_info[2];
        
        // Calculate new dimensions
        $ratio = min($max_width / $original_width, $max_height / $original_height);
        if ($ratio >= 1) {
            // Don't upscale, just copy with compression
            $new_width = $original_width;
            $new_height = $original_height;
        } else {
            $new_width = round($original_width * $ratio);
            $new_height = round($original_height * $ratio);
        }
        
        // Create source image
        switch ($image_type) {
            case IMAGETYPE_JPEG:
                $source_image = imagecreatefromjpeg($source_path);
                break;
            case IMAGETYPE_PNG:
                $source_image = imagecreatefrompng($source_path);
                break;
            case IMAGETYPE_GIF:
                $source_image = imagecreatefromgif($source_path);
                break;
            case IMAGETYPE_WEBP:
                $source_image = imagecreatefromwebp($source_path);
                break;
            default:
                return false;
        }
        
        if (!$source_image) {
            return false;
        }
        
        // Create new image
        $new_image = imagecreatetruecolor($new_width, $new_height);
        
        // Preserve transparency for PNG and GIF
        if ($image_type == IMAGETYPE_PNG || $image_type == IMAGETYPE_GIF) {
            imagealphablending($new_image, false);
            imagesavealpha($new_image, true);
            $transparent = imagecolorallocatealpha($new_image, 255, 255, 255, 127);
            imagefill($new_image, 0, 0, $transparent);
        }
        
        // Resize image
        imagecopyresampled($new_image, $source_image, 0, 0, 0, 0, $new_width, $new_height, $original_width, $original_height);
        
        // Save optimized image
        $result = false;
        switch ($image_type) {
            case IMAGETYPE_JPEG:
                $result = imagejpeg($new_image, $dest_path, $quality);
                break;
            case IMAGETYPE_PNG:
                $result = imagepng($new_image, $dest_path, 9);
                break;
            case IMAGETYPE_GIF:
                $result = imagegif($new_image, $dest_path);
                break;
            case IMAGETYPE_WEBP:
                $result = imagewebp($new_image, $dest_path, $quality);
                break;
        }
        
        imagedestroy($source_image);
        imagedestroy($new_image);
        
        return $result;
    }
    
    /**
     * Create thumbnail (square crop)
     */
    private function createThumbnail($source_path, $dest_path, $size, $size_height = null) {
        $size_height = $size_height ?? $size;
        
        $image_info = getimagesize($source_path);
        $original_width = $image_info[0];
        $original_height = $image_info[1];
        $image_type = $image_info[2];
        
        // Create source image
        switch ($image_type) {
            case IMAGETYPE_JPEG:
                $source_image = imagecreatefromjpeg($source_path);
                break;
            case IMAGETYPE_PNG:
                $source_image = imagecreatefrompng($source_path);
                break;
            case IMAGETYPE_GIF:
                $source_image = imagecreatefromgif($source_path);
                break;
            case IMAGETYPE_WEBP:
                $source_image = imagecreatefromwebp($source_path);
                break;
            default:
                return false;
        }
        
        if (!$source_image) {
            return false;
        }
        
        // Calculate crop dimensions (center crop)
        $source_ratio = $original_width / $original_height;
        $thumb_ratio = $size / $size_height;
        
        if ($source_ratio > $thumb_ratio) {
            // Source is wider, crop width
            $crop_height = $original_height;
            $crop_width = $original_height * $thumb_ratio;
            $crop_x = ($original_width - $crop_width) / 2;
            $crop_y = 0;
        } else {
            // Source is taller, crop height
            $crop_width = $original_width;
            $crop_height = $original_width / $thumb_ratio;
            $crop_x = 0;
            $crop_y = ($original_height - $crop_height) / 2;
        }
        
        // Create thumbnail
        $thumbnail = imagecreatetruecolor($size, $size_height);
        
        // Preserve transparency
        if ($image_type == IMAGETYPE_PNG || $image_type == IMAGETYPE_GIF) {
            imagealphablending($thumbnail, false);
            imagesavealpha($thumbnail, true);
            $transparent = imagecolorallocatealpha($thumbnail, 255, 255, 255, 127);
            imagefill($thumbnail, 0, 0, $transparent);
        }
        
        // Copy and resize
        imagecopyresampled($thumbnail, $source_image, 0, 0, $crop_x, $crop_y, $size, $size_height, $crop_width, $crop_height);
        
        // Save thumbnail
        $result = false;
        switch ($image_type) {
            case IMAGETYPE_JPEG:
                $result = imagejpeg($thumbnail, $dest_path, 85);
                break;
            case IMAGETYPE_PNG:
                $result = imagepng($thumbnail, $dest_path, 9);
                break;
            case IMAGETYPE_GIF:
                $result = imagegif($thumbnail, $dest_path);
                break;
            case IMAGETYPE_WEBP:
                $result = imagewebp($thumbnail, $dest_path, 85);
                break;
        }
        
        imagedestroy($source_image);
        imagedestroy($thumbnail);
        
        return $result;
    }
    
    /**
     * Create WebP version
     */
    private function createWebPVersion($source_path, $dest_path) {
        if (!function_exists('imagewebp')) {
            return false;
        }
        
        $image_info = getimagesize($source_path);
        $image_type = $image_info[2];
        
        switch ($image_type) {
            case IMAGETYPE_JPEG:
                $source_image = imagecreatefromjpeg($source_path);
                break;
            case IMAGETYPE_PNG:
                $source_image = imagecreatefrompng($source_path);
                break;
            case IMAGETYPE_GIF:
                $source_image = imagecreatefromgif($source_path);
                break;
            default:
                return false;
        }
        
        if (!$source_image) {
            return false;
        }
        
        $result = imagewebp($source_image, $dest_path, 85);
        imagedestroy($source_image);
        
        return $result;
    }
    
    /**
     * Get image dimensions
     */
    private function getImageDimensions($file_path) {
        $image_info = getimagesize($file_path);
        return [
            'width' => $image_info[0],
            'height' => $image_info[1]
        ];
    }
    
    /**
     * Save photo record to database
     */
    private function savePhotoRecord($filename, $original_name, $file_size) {
        $stmt = $this->pdo->prepare("
            INSERT INTO member_photos (
                filename, original_name, file_size, upload_date, uploaded_by
            ) VALUES (?, ?, ?, NOW(), ?)
        ");
        
        $uploaded_by = $_SESSION['admin_id'] ?? null;
        $stmt->execute([$filename, $original_name, $file_size, $uploaded_by]);
        
        return [
            'id' => $this->pdo->lastInsertId(),
            'filename' => $filename
        ];
    }
    
    /**
     * Get photo information
     */
    public function getPhotoInfo($photo_id) {
        $stmt = $this->pdo->prepare("
            SELECT * FROM member_photos WHERE id = ?
        ");
        $stmt->execute([$photo_id]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    /**
     * Delete photo and all its versions
     */
    public function deletePhoto($photo_id) {
        $photo = $this->getPhotoInfo($photo_id);
        if (!$photo) {
            return ['success' => false, 'error' => 'Photo not found'];
        }
        
        $filename = $photo['filename'];
        $base_name = pathinfo($filename, PATHINFO_FILENAME);
        $extension = pathinfo($filename, PATHINFO_EXTENSION);
        
        // Delete all versions
        $files_to_delete = [
            $this->upload_dir . 'originals/' . $filename,
            $this->upload_dir . 'optimized/' . $filename,
            $this->upload_dir . 'thumbnails/' . $filename,
            $this->upload_dir . 'optimized/' . $base_name . '.webp',
            $this->upload_dir . 'thumbnails/' . $base_name . '.webp'
        ];
        
        foreach ($files_to_delete as $file) {
            if (file_exists($file)) {
                unlink($file);
            }
        }
        
        // Delete from database
        $stmt = $this->pdo->prepare("DELETE FROM member_photos WHERE id = ?");
        $stmt->execute([$photo_id]);
        
        return ['success' => true, 'message' => 'Photo deleted successfully'];
    }
    
    /**
     * Update member photo
     */
    public function updateMemberPhoto($member_id, $photo_id) {
        $stmt = $this->pdo->prepare("
            UPDATE members SET photo_id = ?, updated_at = NOW() WHERE id = ?
        ");
        $stmt->execute([$photo_id, $member_id]);
        
        return ['success' => true, 'message' => 'Member photo updated'];
    }
    
    /**
     * Clean up temporary files (call periodically)
     */
    public function cleanupTempFiles($max_age = 3600) {
        $temp_files = glob($this->temp_dir . '*');
        $current_time = time();
        
        foreach ($temp_files as $file) {
            if (is_file($file) && ($current_time - filemtime($file)) > $max_age) {
                unlink($file);
            }
        }
    }
    
    /**
     * Handle traditional upload (non-chunked)
     */
    public function handleTraditionalUpload($file, $member_id = 0) {
        try {
            // Validate uploaded file
            $validation = $this->validateUploadedFile($file['tmp_name'], $file['name']);
            if (!$validation['valid']) {
                return ['success' => false, 'error' => $validation['error']];
            }
            
            // Process the uploaded file
            $result = $this->processUploadedFile($file['tmp_name'], $file['name']);
            
            if ($result['success'] && $member_id > 0) {
                // Associate with member
                $this->associatePhotoWithMember($result['photo_id'], $member_id);
            }
            
            return $result;
            
        } catch (Exception $e) {
            error_log("Traditional upload error: " . $e->getMessage());
            return ['success' => false, 'error' => 'Upload failed'];
        }
    }
    
    /**
     * Complete chunked upload
     */
    public function completeUpload($file_id, $member_id = 0) {
        global $pdo;
        
        try {
            // Check if upload session exists
            $stmt = $pdo->prepare("
                SELECT * FROM photo_upload_sessions 
                WHERE session_id = ? AND upload_status = 'in_progress'
            ");
            $stmt->execute([$file_id]);
            $session = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$session) {
                return ['success' => false, 'error' => 'Upload session not found'];
            }
            
            // Mark session as completed
            $stmt = $pdo->prepare("
                UPDATE photo_upload_sessions 
                SET upload_status = 'completed', updated_at = NOW()
                WHERE session_id = ?
            ");
            $stmt->execute([$file_id]);
            
            // Get the assembled file path
            $assembled_file = $this->temp_dir . $file_id . '.assembled';
            
            if (!file_exists($assembled_file)) {
                return ['success' => false, 'error' => 'Assembled file not found'];
            }
            
            // Process the assembled file
            $result = $this->processUploadedFile($assembled_file, $session['original_filename']);
            
            if ($result['success'] && $member_id > 0) {
                // Associate with member
                $this->associatePhotoWithMember($result['photo_id'], $member_id);
            }
            
            return $result;
            
        } catch (Exception $e) {
            error_log("Complete upload error: " . $e->getMessage());
            return ['success' => false, 'error' => 'Upload completion failed'];
        }
    }
    
    /**
     * Associate photo with member
     */
    private function associatePhotoWithMember($photo_id, $member_id) {
        global $pdo;
        
        try {
            // Update photo record with member association
            $stmt = $pdo->prepare("
                UPDATE member_photos 
                SET member_id = ?, updated_at = NOW()
                WHERE id = ?
            ");
            $stmt->execute([$member_id, $photo_id]);
            
            // Update member's photo_id if they don't have a profile photo
            $stmt = $pdo->prepare("
                SELECT photo_id FROM members WHERE id = ?
            ");
            $stmt->execute([$member_id]);
            $current_photo = $stmt->fetchColumn();
            
            if (!$current_photo) {
                $stmt = $pdo->prepare("
                    UPDATE members 
                    SET photo_id = ?, updated_at = NOW()
                    WHERE id = ?
                ");
                $stmt->execute([$photo_id, $member_id]);
            }
            
        } catch (Exception $e) {
            error_log("Associate photo error: " . $e->getMessage());
        }
    }
    
    /**
     * Clean up upload session files
     */
    public function cleanupUploadSession($session_id) {
        global $pdo;
        
        try {
            // Remove temporary chunk files
            $chunk_files = glob($this->temp_dir . $session_id . '.part*');
            foreach ($chunk_files as $file) {
                if (file_exists($file)) {
                    unlink($file);
                }
            }
            
            // Remove assembled file
            $assembled_file = $this->temp_dir . $session_id . '.assembled';
            if (file_exists($assembled_file)) {
                unlink($assembled_file);
            }
            
            // Update database session status
            $stmt = $pdo->prepare("
                UPDATE photo_upload_sessions 
                SET upload_status = 'cancelled', updated_at = NOW()
                WHERE session_id = ?
            ");
            $stmt->execute([$session_id]);
            
        } catch (Exception $e) {
            error_log("Cleanup session error: " . $e->getMessage());
        }
    }
}
?>
