<?php
/**
 * Enhanced Notification Service with Real-time Updates and Email Integration
 */

require_once '../config.php';
require_once '../services/EmailService.php';

class NotificationService {
    private $pdo;
    private $emailService;
    
    public function __construct() {
        global $pdo;
        $this->pdo = $pdo;
        $this->emailService = new EmailService();
    }
    
    /**
     * Create notification with email and real-time features
     */
    public function createNotification($data) {
        try {
            $this->pdo->beginTransaction();
            
            // Insert notification
            $stmt = $this->pdo->prepare("
                INSERT INTO notifications (
                    type, title, message, target_audience, target_user_id, 
                    created_by, priority, expires_at, related_entity_type, 
                    related_entity_id, metadata
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            
            $stmt->execute([
                $data['type'],
                $data['title'],
                $data['message'],
                $data['target_audience'],
                $data['target_user_id'] ?? null,
                $data['created_by'],
                $data['priority'] ?? 'normal',
                $data['expires_at'] ?? null,
                $data['related_entity_type'] ?? null,
                $data['related_entity_id'] ?? null,
                json_encode($data['metadata'] ?? [])
            ]);
            
            $notificationId = $this->pdo->lastInsertId();
            
            // Create notification recipients
            $recipients = $this->getRecipients($data['target_audience'], $data['target_user_id']);
            foreach ($recipients as $recipient) {
                $stmt = $this->pdo->prepare("
                    INSERT INTO notification_recipients (
                        notification_id, user_id, user_type
                    ) VALUES (?, ?, ?)
                ");
                $stmt->execute([$notificationId, $recipient['user_id'], $recipient['user_type']]);
            }
            
            $this->pdo->commit();
            
            // Send emails for recipients who have email notifications enabled
            $this->sendEmailNotifications($notificationId, $data, $recipients);
            
            // Trigger real-time notification
            $this->triggerRealTimeNotification($notificationId, $data, $recipients);
            
            return [
                'success' => true,
                'notification_id' => $notificationId,
                'recipients_count' => count($recipients)
            ];
            
        } catch (Exception $e) {
            $this->pdo->rollback();
            error_log("Notification creation error: " . $e->getMessage());
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Get recipients based on target audience
     */
    private function getRecipients($targetAudience, $targetUserId = null) {
        $recipients = [];
        
        switch ($targetAudience) {
            case 'all':
                // Get all members and admins
                $stmt = $this->pdo->query("SELECT id as user_id, 'member' as user_type FROM members WHERE status = 'active'");
                $recipients = array_merge($recipients, $stmt->fetchAll(PDO::FETCH_ASSOC));
                
                $stmt = $this->pdo->query("SELECT id as user_id, 'admin' as user_type FROM admins WHERE status = 'active'");
                $recipients = array_merge($recipients, $stmt->fetchAll(PDO::FETCH_ASSOC));
                break;
                
            case 'members':
                $stmt = $this->pdo->query("SELECT id as user_id, 'member' as user_type FROM members WHERE status = 'active'");
                $recipients = $stmt->fetchAll(PDO::FETCH_ASSOC);
                break;
                
            case 'admins':
                $stmt = $this->pdo->query("SELECT id as user_id, 'admin' as user_type FROM admins WHERE status = 'active'");
                $recipients = $stmt->fetchAll(PDO::FETCH_ASSOC);
                break;
                
            case 'specific':
                if ($targetUserId) {
                    // Determine user type
                    $stmt = $this->pdo->prepare("SELECT 'member' as user_type FROM members WHERE id = ? AND status = 'active'");
                    $stmt->execute([$targetUserId]);
                    $user = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    if ($user) {
                        $recipients[] = ['user_id' => $targetUserId, 'user_type' => 'member'];
                    } else {
                        $stmt = $this->pdo->prepare("SELECT 'admin' as user_type FROM admins WHERE id = ? AND status = 'active'");
                        $stmt->execute([$targetUserId]);
                        $user = $stmt->fetch(PDO::FETCH_ASSOC);
                        
                        if ($user) {
                            $recipients[] = ['user_id' => $targetUserId, 'user_type' => 'admin'];
                        }
                    }
                }
                break;
        }
        
        return $recipients;
    }
    
    /**
     * Send email notifications to recipients
     */
    private function sendEmailNotifications($notificationId, $notificationData, $recipients) {
        foreach ($recipients as $recipient) {
            // Check if user has email notifications enabled
            $stmt = $this->pdo->prepare("
                SELECT email_notifications 
                FROM notification_settings 
                WHERE user_id = ? AND user_type = ?
            ");
            $stmt->execute([$recipient['user_id'], $recipient['user_type']]);
            $settings = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$settings || $settings['email_notifications']) { // Default to enabled if no settings
                // Get user email
                $email = $this->getUserEmail($recipient['user_id'], $recipient['user_type']);
                
                if ($email) {
                    // Send email based on notification type
                    $this->sendNotificationEmail($email, $notificationData, $recipient);
                }
            }
        }
    }
    
    /**
     * Get user email address
     */
    private function getUserEmail($userId, $userType) {
        $table = $userType === 'admin' ? 'admins' : 'members';
        $stmt = $this->pdo->prepare("SELECT email FROM {$table} WHERE id = ?");
        $stmt->execute([$userId]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result ? $result['email'] : null;
    }
    
    /**
     * Send notification email using template
     */
    private function sendNotificationEmail($email, $notificationData, $recipient) {
        $templateData = [
            'title' => $notificationData['title'],
            'message' => $notificationData['message'],
            'type' => $notificationData['type'],
            'priority' => $notificationData['priority'] ?? 'normal',
            'notification_url' => $this->getNotificationUrl(),
            'first_name' => $this->getUserName($recipient['user_id'], $recipient['user_type'])
        ];
        
        // Choose template based on notification type
        $template = $this->getEmailTemplate($notificationData['type']);
        
        if ($template) {
            $success = $this->emailService->sendTemplate(
                $template,
                $templateData,
                $email,
                $notificationData['title'],
                $notificationData['priority'] ?? 'normal'
            );
        } else {
            // Fallback to generic notification
            $success = $this->emailService->sendEmail(
                $email,
                $notificationData['title'],
                $this->buildGenericEmailBody($notificationData),
                true,
                $notificationData['priority'] ?? 'normal'
            );
        }
        
        return $success;
    }
    
    /**
     * Get notification URL based on current domain
     */
    private function getNotificationUrl() {
        $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        return $protocol . '://' . $host . '/KSOCHD/frontend/pages/notifications.html';
    }
    
    /**
     * Get user's first name
     */
    private function getUserName($userId, $userType) {
        $table = $userType === 'admin' ? 'admins' : 'members';
        $stmt = $this->pdo->prepare("SELECT first_name FROM {$table} WHERE id = ?");
        $stmt->execute([$userId]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result ? $result['first_name'] : 'Member';
    }
    
    /**
     * Get email template based on notification type
     */
    private function getEmailTemplate($type) {
        $templates = [
            'event' => 'event-reminder',
            'payment' => 'fee-reminder',
            'membership' => 'welcome',
            'announcement' => 'notification'
        ];
        
        return $templates[$type] ?? 'notification';
    }
    
    /**
     * Build generic email body
     */
    private function buildGenericEmailBody($notificationData) {
        $priorityColor = $this->getPriorityColor($notificationData['priority'] ?? 'normal');
        
        return "
        <div style='max-width: 600px; margin: 0 auto; font-family: Arial, sans-serif;'>
            <div style='background: {$priorityColor}; color: white; padding: 20px; text-align: center;'>
                <h1>{$notificationData['title']}</h1>
            </div>
            <div style='padding: 20px; background: #f9f9f9;'>
                <p>{$notificationData['message']}</p>
                <p style='margin-top: 20px;'>
                    <a href='https://yourdomain.com/notifications' 
                       style='background: {$priorityColor}; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px;'>
                        View Notification
                    </a>
                </p>
            </div>
            <div style='background: #333; color: white; padding: 10px; text-align: center; font-size: 12px;'>
                KSO Chandigarh - Kashmiri Students Organization
            </div>
        </div>";
    }
    
    /**
     * Get priority color for styling
     */
    private function getPriorityColor($priority) {
        $colors = [
            'low' => '#6c757d',
            'normal' => '#007bff',
            'high' => '#fd7e14',
            'urgent' => '#dc3545'
        ];
        
        return $colors[$priority] ?? '#007bff';
    }
    
    /**
     * Trigger real-time notification (WebSocket/SSE)
     */
    private function triggerRealTimeNotification($notificationId, $notificationData, $recipients) {
        // Write to real-time notification file/queue
        $realtimeData = [
            'id' => $notificationId,
            'type' => $notificationData['type'],
            'title' => $notificationData['title'],
            'message' => $notificationData['message'],
            'priority' => $notificationData['priority'] ?? 'normal',
            'recipients' => $recipients,
            'timestamp' => time()
        ];
        
        // Write to SSE file (simple implementation)
        $sseFile = __DIR__ . '/../realtime/notifications.json';
        $notifications = [];
        
        if (file_exists($sseFile)) {
            $notifications = json_decode(file_get_contents($sseFile), true) ?: [];
        }
        
        $notifications[] = $realtimeData;
        
        // Keep only last 100 notifications
        $notifications = array_slice($notifications, -100);
        
        file_put_contents($sseFile, json_encode($notifications));
    }
    
    /**
     * Get unread notifications for user
     */
    public function getUnreadNotifications($userId, $userType) {
        $stmt = $this->pdo->prepare("
            SELECT n.*, nr.is_read, nr.delivered_at
            FROM notifications n
            JOIN notification_recipients nr ON n.id = nr.notification_id
            WHERE nr.user_id = ? AND nr.user_type = ? AND nr.is_read = 0
            AND (n.expires_at IS NULL OR n.expires_at > NOW())
            ORDER BY n.created_at DESC
        ");
        
        $stmt->execute([$userId, $userType]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    /**
     * Mark notification as read
     */
    public function markAsRead($notificationId, $userId, $userType) {
        $stmt = $this->pdo->prepare("
            UPDATE notification_recipients 
            SET is_read = 1, read_at = NOW()
            WHERE notification_id = ? AND user_id = ? AND user_type = ?
        ");
        
        return $stmt->execute([$notificationId, $userId, $userType]);
    }
    
    /**
     * Get notification statistics
     */
    public function getStats() {
        $stats = [];
        
        // Total notifications
        $stmt = $this->pdo->query("SELECT COUNT(*) as total FROM notifications");
        $stats['total_notifications'] = $stmt->fetchColumn();
        
        // Delivered notifications
        $stmt = $this->pdo->query("SELECT COUNT(*) as delivered FROM notification_recipients");
        $stats['delivered_notifications'] = $stmt->fetchColumn();
        
        // Unread notifications
        $stmt = $this->pdo->query("SELECT COUNT(*) as unread FROM notification_recipients WHERE is_read = 0");
        $stats['unread_notifications'] = $stmt->fetchColumn();
        
        // Failed notifications (could be based on email logs)
        $stats['failed_notifications'] = 0; // Placeholder
        
        return $stats;
    }
}
?>
