<?php
/**
 * PDF Generation Service using TCPDF for KSO Chandigarh
 * Handles PDF generation for receipts, invoices, and other documents
 */

require_once __DIR__ . '/../../libraries/TCPDF-main/tcpdf.php';

class PDFService extends TCPDF {
    private $organization_name;
    private $organization_address;
    private $organization_phone;
    private $organization_email;
    
    public function __construct($orientation = 'P', $unit = 'mm', $format = 'A4') {
        parent::__construct($orientation, $unit, $format, true, 'UTF-8', false);
        
        // Set default organization details
        $this->organization_name = 'KSO Chandigarh';
        $this->organization_address = 'Chandigarh, India';
        $this->organization_phone = '+91-XXXXXXXXXX';
        $this->organization_email = 'info@ksochandigarh.org';
        
        // Set document properties
        $this->SetCreator('KSO Chandigarh System');
        $this->SetAuthor('KSO Chandigarh');
        $this->SetTitle('KSO Document');
        $this->SetSubject('KSO Chandigarh Document');
        $this->SetKeywords('KSO, Chandigarh, Receipt, Invoice');
        
        // Set margins
        $this->SetMargins(15, 15, 15);
        $this->SetHeaderMargin(5);
        $this->SetFooterMargin(10);
        
        // Set auto page breaks
        $this->SetAutoPageBreak(true, 25);
        
        // Set image scale factor
        $this->setImageScale(PDF_IMAGE_SCALE_RATIO);
    }
    
    /**
     * Set organization details
     */
    public function setOrganizationDetails($name, $address, $phone, $email) {
        $this->organization_name = $name;
        $this->organization_address = $address;
        $this->organization_phone = $phone;
        $this->organization_email = $email;
    }
    
    /**
     * Custom header for KSO documents
     */
    public function Header() {
        // Logo (if exists)
        $logo_path = __DIR__ . '/../assets/images/kso-logo.png';
        if (file_exists($logo_path)) {
            $this->Image($logo_path, 15, 10, 20, 20, 'PNG', '', 'T', false, 300, '', false, false, 0, false, false, false);
        }
        
        // Organization name and details
        $this->SetFont('helvetica', 'B', 16);
        $this->SetTextColor(0, 72, 153); // KSO Blue
        $this->Cell(0, 10, $this->organization_name, 0, 1, 'C');
        
        $this->SetFont('helvetica', '', 10);
        $this->SetTextColor(100, 100, 100);
        $this->Cell(0, 5, $this->organization_address, 0, 1, 'C');
        $this->Cell(0, 5, 'Phone: ' . $this->organization_phone . ' | Email: ' . $this->organization_email, 0, 1, 'C');
        
        // Line separator
        $this->Ln(5);
        $this->SetDrawColor(0, 72, 153);
        $this->Line(15, $this->GetY(), $this->getPageWidth() - 15, $this->GetY());
        $this->Ln(10);
    }
    
    /**
     * Custom footer for KSO documents
     */
    public function Footer() {
        $this->SetY(-15);
        $this->SetFont('helvetica', 'I', 8);
        $this->SetTextColor(128, 128, 128);
        
        // Line separator
        $this->SetDrawColor(200, 200, 200);
        $this->Line(15, $this->GetY() - 5, $this->getPageWidth() - 15, $this->GetY() - 5);
        
        // Footer text
        $this->Cell(0, 5, 'Generated on ' . date('Y-m-d H:i:s') . ' | KSO Chandigarh Management System', 0, 1, 'C');
        $this->Cell(0, 5, 'Page ' . $this->getAliasNumPage() . ' of ' . $this->getAliasNbPages(), 0, 0, 'C');
    }
    
    /**
     * Generate receipt PDF
     */
    public function generateReceipt($transaction_data) {
        // Add a page
        $this->AddPage();
        
        // Document title
        $this->SetFont('helvetica', 'B', 20);
        $this->SetTextColor(0, 72, 153);
        $this->Cell(0, 15, 'RECEIPT', 0, 1, 'C');
        
        // Receipt number and date
        $this->SetFont('helvetica', 'B', 12);
        $this->SetTextColor(0, 0, 0);
        $this->Cell(0, 8, 'Receipt No: ' . $transaction_data['receipt_number'], 0, 1, 'C');
        $this->Cell(0, 8, 'Date: ' . date('d/m/Y', strtotime($transaction_data['transaction_date'])), 0, 1, 'C');
        $this->Ln(10);
        
        // Member details section
        $this->SetFont('helvetica', 'B', 12);
        $this->SetFillColor(240, 240, 240);
        $this->Cell(0, 8, 'MEMBER DETAILS', 0, 1, 'L', true);
        $this->Ln(2);
        
        $this->SetFont('helvetica', '', 10);
        $this->Cell(40, 6, 'Name:', 0, 0, 'L');
        $this->Cell(0, 6, $transaction_data['first_name'] . ' ' . $transaction_data['last_name'], 0, 1, 'L');
        
        $this->Cell(40, 6, 'KSO ID:', 0, 0, 'L');
        $this->Cell(0, 6, $transaction_data['kso_id'], 0, 1, 'L');
        
        $this->Cell(40, 6, 'Email:', 0, 0, 'L');
        $this->Cell(0, 6, $transaction_data['email'], 0, 1, 'L');
        
        if (!empty($transaction_data['phone'])) {
            $this->Cell(40, 6, 'Phone:', 0, 0, 'L');
            $this->Cell(0, 6, $transaction_data['phone'], 0, 1, 'L');
        }
        
        $this->Cell(40, 6, 'Membership:', 0, 0, 'L');
        $this->Cell(0, 6, ucfirst($transaction_data['membership_type']), 0, 1, 'L');
        $this->Ln(10);
        
        // Transaction details section
        $this->SetFont('helvetica', 'B', 12);
        $this->Cell(0, 8, 'TRANSACTION DETAILS', 0, 1, 'L', true);
        $this->Ln(2);
        
        // Transaction table
        $this->SetFont('helvetica', 'B', 10);
        $this->SetFillColor(220, 220, 220);
        $this->Cell(80, 8, 'Description', 1, 0, 'L', true);
        $this->Cell(30, 8, 'Type', 1, 0, 'C', true);
        $this->Cell(25, 8, 'Amount', 1, 0, 'C', true);
        $this->Cell(25, 8, 'Status', 1, 1, 'C', true);
        
        $this->SetFont('helvetica', '', 10);
        $this->SetFillColor(255, 255, 255);
        $this->Cell(80, 8, $transaction_data['description'], 1, 0, 'L');
        $this->Cell(30, 8, ucfirst(str_replace('_', ' ', $transaction_data['transaction_type'])), 1, 0, 'C');
        $this->Cell(25, 8, '₹' . number_format($transaction_data['amount'], 2), 1, 0, 'C');
        $this->Cell(25, 8, ucfirst($transaction_data['status']), 1, 1, 'C');
        
        // Total amount
        $this->SetFont('helvetica', 'B', 12);
        $this->SetFillColor(240, 240, 240);
        $this->Cell(110, 8, 'TOTAL AMOUNT', 1, 0, 'R', true);
        $this->Cell(50, 8, '₹' . number_format($transaction_data['amount'], 2), 1, 1, 'C', true);
        $this->Ln(10);
        
        // Payment details
        if (!empty($transaction_data['payment_method'])) {
            $this->SetFont('helvetica', 'B', 12);
            $this->Cell(0, 8, 'PAYMENT DETAILS', 0, 1, 'L', true);
            $this->Ln(2);
            
            $this->SetFont('helvetica', '', 10);
            $this->Cell(40, 6, 'Method:', 0, 0, 'L');
            $this->Cell(0, 6, ucfirst($transaction_data['payment_method']), 0, 1, 'L');
            
            if (!empty($transaction_data['reference_number'])) {
                $this->Cell(40, 6, 'Reference:', 0, 0, 'L');
                $this->Cell(0, 6, $transaction_data['reference_number'], 0, 1, 'L');
            }
            
            $this->Ln(10);
        }
        
        // Notes section
        if (!empty($transaction_data['notes'])) {
            $this->SetFont('helvetica', 'B', 12);
            $this->Cell(0, 8, 'NOTES', 0, 1, 'L', true);
            $this->Ln(2);
            
            $this->SetFont('helvetica', '', 10);
            $this->MultiCell(0, 6, $transaction_data['notes'], 0, 'L');
            $this->Ln(10);
        }
        
        // Signature section
        $this->Ln(20);
        $this->SetFont('helvetica', '', 10);
        $this->Cell(80, 6, 'Received by:', 0, 0, 'L');
        $this->Cell(80, 6, 'Authorized Signature:', 0, 1, 'L');
        $this->Ln(15);
        $this->Cell(80, 6, '________________________', 0, 0, 'L');
        $this->Cell(80, 6, '________________________', 0, 1, 'L');
        
        if (!empty($transaction_data['processed_by_name'])) {
            $this->Cell(80, 6, $transaction_data['processed_by_name'], 0, 0, 'L');
        }
        
        // Thank you message
        $this->Ln(15);
        $this->SetFont('helvetica', 'I', 10);
        $this->SetTextColor(100, 100, 100);
        $this->Cell(0, 6, 'Thank you for your contribution to KSO Chandigarh!', 0, 1, 'C');
        
        return true;
    }
    
    /**
     * Generate invoice PDF
     */
    public function generateInvoice($invoice_data) {
        // Add a page
        $this->AddPage();
        
        // Document title
        $this->SetFont('helvetica', 'B', 20);
        $this->SetTextColor(0, 72, 153);
        $this->Cell(0, 15, 'INVOICE', 0, 1, 'C');
        
        // Invoice number and date
        $this->SetFont('helvetica', 'B', 12);
        $this->SetTextColor(0, 0, 0);
        $this->Cell(0, 8, 'Invoice No: ' . $invoice_data['invoice_number'], 0, 1, 'C');
        $this->Cell(0, 8, 'Date: ' . date('d/m/Y', strtotime($invoice_data['invoice_date'])), 0, 1, 'C');
        $this->Cell(0, 8, 'Due Date: ' . date('d/m/Y', strtotime($invoice_data['due_date'])), 0, 1, 'C');
        $this->Ln(10);
        
        // Bill to section
        $this->SetFont('helvetica', 'B', 12);
        $this->SetFillColor(240, 240, 240);
        $this->Cell(0, 8, 'BILL TO', 0, 1, 'L', true);
        $this->Ln(2);
        
        $this->SetFont('helvetica', '', 10);
        $this->Cell(0, 6, $invoice_data['first_name'] . ' ' . $invoice_data['last_name'], 0, 1, 'L');
        $this->Cell(0, 6, 'KSO ID: ' . $invoice_data['kso_id'], 0, 1, 'L');
        $this->Cell(0, 6, 'Email: ' . $invoice_data['email'], 0, 1, 'L');
        
        if (!empty($invoice_data['address'])) {
            $this->MultiCell(0, 6, 'Address: ' . $invoice_data['address'], 0, 'L');
        }
        $this->Ln(10);
        
        // Invoice items table
        $this->SetFont('helvetica', 'B', 10);
        $this->SetFillColor(220, 220, 220);
        $this->Cell(80, 8, 'Description', 1, 0, 'L', true);
        $this->Cell(20, 8, 'Qty', 1, 0, 'C', true);
        $this->Cell(30, 8, 'Unit Price', 1, 0, 'C', true);
        $this->Cell(30, 8, 'Total', 1, 1, 'C', true);
        
        $this->SetFont('helvetica', '', 10);
        $total_amount = 0;
        
        foreach ($invoice_data['items'] as $item) {
            $item_total = $item['quantity'] * $item['unit_price'];
            $total_amount += $item_total;
            
            $this->Cell(80, 8, $item['description'], 1, 0, 'L');
            $this->Cell(20, 8, $item['quantity'], 1, 0, 'C');
            $this->Cell(30, 8, '₹' . number_format($item['unit_price'], 2), 1, 0, 'C');
            $this->Cell(30, 8, '₹' . number_format($item_total, 2), 1, 1, 'C');
        }
        
        // Total section
        $this->SetFont('helvetica', 'B', 12);
        $this->SetFillColor(240, 240, 240);
        $this->Cell(130, 8, 'TOTAL AMOUNT', 1, 0, 'R', true);
        $this->Cell(30, 8, '₹' . number_format($total_amount, 2), 1, 1, 'C', true);
        $this->Ln(10);
        
        // Payment instructions
        $this->SetFont('helvetica', 'B', 12);
        $this->Cell(0, 8, 'PAYMENT INSTRUCTIONS', 0, 1, 'L', true);
        $this->Ln(2);
        
        $this->SetFont('helvetica', '', 10);
        $this->MultiCell(0, 6, 'Please make payment by the due date mentioned above. You can pay online through our website or contact our office for payment methods.', 0, 'L');
        $this->Ln(10);
        
        // Status
        $this->SetFont('helvetica', 'B', 12);
        $this->Cell(0, 8, 'Status: ' . ucfirst($invoice_data['status']), 0, 1, 'L');
        
        return true;
    }
    
    /**
     * Save PDF to file
     */
    public function savePDF($filename, $dest = 'F') {
        return $this->Output($filename, $dest);
    }
    
    /**
     * Get PDF as string
     */
    public function getPDFString() {
        return $this->Output('', 'S');
    }
}
?>
