<?php
/**
 * Receipt and Invoice Generation Service for KSO Chandigarh
 * Handles PDF generation, storage, and management of receipts and invoices
 */

require_once '../config.php';
require_once 'PDFService.php';

class ReceiptService {
    private $pdo;
    private $receipt_dir;
    private $invoice_dir;
    
    public function __construct() {
        global $pdo;
        $this->pdo = $pdo;
        $this->receipt_dir = __DIR__ . '/../storage/receipts/';
        $this->invoice_dir = __DIR__ . '/../storage/invoices/';
        
        // Create directories if they don't exist
        $this->ensureDirectoriesExist();
    }
    
    /**
     * Ensure storage directories exist
     */
    private function ensureDirectoriesExist() {
        if (!is_dir($this->receipt_dir)) {
            mkdir($this->receipt_dir, 0755, true);
        }
        if (!is_dir($this->invoice_dir)) {
            mkdir($this->invoice_dir, 0755, true);
        }
        
        // Create year-based subdirectories
        $current_year = date('Y');
        if (!is_dir($this->receipt_dir . $current_year)) {
            mkdir($this->receipt_dir . $current_year, 0755, true);
        }
        if (!is_dir($this->invoice_dir . $current_year)) {
            mkdir($this->invoice_dir . $current_year, 0755, true);
        }
    }
    
    /**
     * Generate receipt number
     */
    public function generateReceiptNumber($transaction_type = 'general') {
        $prefix = $this->getReceiptPrefix($transaction_type);
        $year = date('Y');
        $month = date('m');
        
        // Get the next sequence number for this month
        $stmt = $this->pdo->prepare("
            SELECT MAX(CAST(SUBSTRING(receipt_number, -4) AS UNSIGNED)) as max_seq
            FROM receipts 
            WHERE receipt_number LIKE ? AND YEAR(created_at) = ? AND MONTH(created_at) = ?
        ");
        $stmt->execute(["{$prefix}{$year}{$month}%", $year, $month]);
        $result = $stmt->fetch();
        
        $next_seq = ($result['max_seq'] ?? 0) + 1;
        $sequence = str_pad($next_seq, 4, '0', STR_PAD_LEFT);
        
        return "{$prefix}{$year}{$month}{$sequence}";
    }
    
    /**
     * Get receipt prefix based on transaction type
     */
    private function getReceiptPrefix($transaction_type) {
        $prefixes = [
            'membership_fee' => 'KSO-MF-',
            'event_fee' => 'KSO-EF-',
            'donation' => 'KSO-DN-',
            'other' => 'KSO-OT-',
            'general' => 'KSO-GN-'
        ];
        
        return $prefixes[$transaction_type] ?? $prefixes['general'];
    }
    
    /**
     * Generate PDF receipt
     */
    public function generateReceipt($transaction_id, $template = 'standard') {
        try {
            // Get transaction details
            $transaction = $this->getTransactionDetails($transaction_id);
            if (!$transaction) {
                throw new Exception('Transaction not found');
            }
            
            // Generate receipt number if not exists
            if (empty($transaction['receipt_number'])) {
                $receipt_number = $this->generateReceiptNumber($transaction['transaction_type']);
                $this->updateTransactionReceiptNumber($transaction_id, $receipt_number);
                $transaction['receipt_number'] = $receipt_number;
            }
            
            // Generate PDF
            $pdf_content = $this->generateReceiptPDF($transaction, $template);
            
            // Save to database and file system
            $receipt_id = $this->saveReceipt($transaction, $pdf_content);
            
            return [
                'success' => true,
                'receipt_id' => $receipt_id,
                'receipt_number' => $transaction['receipt_number'],
                'file_path' => $this->getReceiptFilePath($transaction['receipt_number'])
            ];
            
        } catch (Exception $e) {
            error_log("Receipt generation error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Get transaction details for receipt generation
     */
    private function getTransactionDetails($transaction_id) {
        $stmt = $this->pdo->prepare("
            SELECT f.*, 
                   m.first_name, m.last_name, m.kso_id, m.email, m.phone, m.address,
                   m.membership_type,
                   a.username as processed_by_name,
                   s.setting_value as org_name
            FROM finance f
            LEFT JOIN members m ON f.member_id = m.id
            LEFT JOIN admins a ON f.processed_by = a.id
            LEFT JOIN settings s ON s.setting_key = 'org_name'
            WHERE f.id = ?
        ");
        $stmt->execute([$transaction_id]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    /**
     * Update transaction with receipt number
     */
    private function updateTransactionReceiptNumber($transaction_id, $receipt_number) {
        $stmt = $this->pdo->prepare("
            UPDATE finance 
            SET receipt_number = ?, updated_at = NOW() 
            WHERE id = ?
        ");
        return $stmt->execute([$receipt_number, $transaction_id]);
    }
    
    /**
     * Generate PDF content for receipt using TCPDF
     */
    private function generateReceiptPDF($transaction, $template = 'standard') {
        try {
            // Initialize PDF service
            $pdf = new PDFService();
            
            // Set document information
            $pdf->SetTitle('Receipt - ' . $transaction['receipt_number']);
            $pdf->SetSubject('Payment Receipt');
            
            // Generate the receipt
            $pdf->generateReceipt($transaction);
            
            // Get PDF as string
            $pdf_content = $pdf->getPDFString();
            
            // Save PDF file
            $file_path = $this->getReceiptFilePath($transaction['receipt_number'], 'pdf');
            $pdf->savePDF($file_path);
            
            return $pdf_content;
            
        } catch (Exception $e) {
            error_log("PDF generation error: " . $e->getMessage());
            throw new Exception("Failed to generate PDF: " . $e->getMessage());
        }
    }
    
    /**
     * Convert HTML to PDF (simplified approach)
     * In production, integrate with PDF libraries like TCPDF or mPDF
     */
    private function convertHTMLToPDF($html_content, $receipt_number) {
        // For now, we'll save as HTML with print-friendly CSS
        // In production, replace this with actual PDF generation
        $pdf_content = $this->addPrintStyles($html_content);
        
        // Save HTML file (can be converted to PDF later)
        $file_path = $this->getReceiptFilePath($receipt_number, 'html');
        file_put_contents($file_path, $pdf_content);
        
        return $pdf_content;
    }
    
    /**
     * Add print-friendly styles to HTML
     */
    private function addPrintStyles($html_content) {
        $print_styles = '
        <style>
            @media print {
                body { margin: 0; padding: 20px; font-family: Arial, sans-serif; }
                .no-print { display: none; }
                .receipt { max-width: 100%; }
            }
            @page { margin: 20mm; }
        </style>';
        
        return str_replace('</head>', $print_styles . '</head>', $html_content);
    }
    
    /**
     * Save receipt to database and file system
     */
    private function saveReceipt($transaction, $pdf_content) {
        $year = date('Y');
        $file_path = $this->getReceiptFilePath($transaction['receipt_number'], 'html');
        
        // Save to database
        $stmt = $this->pdo->prepare("
            INSERT INTO receipts (
                receipt_number, transaction_id, member_id, transaction_type,
                amount, file_path, file_type, generated_by, created_at
            ) VALUES (?, ?, ?, ?, ?, ?, 'html', ?, NOW())
        ");
        
        $stmt->execute([
            $transaction['receipt_number'],
            $transaction['id'],
            $transaction['member_id'],
            $transaction['transaction_type'],
            $transaction['amount'],
            $file_path,
            $_SESSION['admin_id'] ?? 1
        ]);
        
        return $this->pdo->lastInsertId();
    }
    
    /**
     * Get receipt file path
     */
    private function getReceiptFilePath($receipt_number, $extension = 'pdf') {
        $year = date('Y');
        return $this->receipt_dir . $year . '/' . $receipt_number . '.' . $extension;
    }
    
    /**
     * Get all receipts with filters
     */
    public function getReceipts($filters = []) {
        $where_conditions = ['1=1'];
        $params = [];
        
        if (!empty($filters['search'])) {
            $where_conditions[] = "(r.receipt_number LIKE ? OR m.first_name LIKE ? OR m.last_name LIKE ? OR m.kso_id LIKE ?)";
            $search = "%{$filters['search']}%";
            $params = array_merge($params, [$search, $search, $search, $search]);
        }
        
        if (!empty($filters['transaction_type'])) {
            $where_conditions[] = "r.transaction_type = ?";
            $params[] = $filters['transaction_type'];
        }
        
        if (!empty($filters['year'])) {
            $where_conditions[] = "YEAR(r.created_at) = ?";
            $params[] = $filters['year'];
        }
        
        if (!empty($filters['month'])) {
            $where_conditions[] = "MONTH(r.created_at) = ?";
            $params[] = $filters['month'];
        }
        
        $where_clause = implode(' AND ', $where_conditions);
        $limit = $filters['limit'] ?? 50;
        $offset = $filters['offset'] ?? 0;
        
        $stmt = $this->pdo->prepare("
            SELECT r.*, 
                   m.first_name, m.last_name, m.kso_id, m.email,
                   f.payment_method, f.payment_date, f.description,
                   a.username as generated_by_name
            FROM receipts r
            LEFT JOIN members m ON r.member_id = m.id
            LEFT JOIN finance f ON r.transaction_id = f.id
            LEFT JOIN admins a ON r.generated_by = a.id
            WHERE {$where_clause}
            ORDER BY r.created_at DESC
            LIMIT ? OFFSET ?
        ");
        
        $params[] = $limit;
        $params[] = $offset;
        $stmt->execute($params);
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    /**
     * Get receipt by ID
     */
    public function getReceipt($receipt_id) {
        $stmt = $this->pdo->prepare("
            SELECT r.*, 
                   m.first_name, m.last_name, m.kso_id, m.email, m.phone, m.address,
                   f.payment_method, f.payment_date, f.description, f.notes,
                   a.username as generated_by_name
            FROM receipts r
            LEFT JOIN members m ON r.member_id = m.id
            LEFT JOIN finance f ON r.transaction_id = f.id
            LEFT JOIN admins a ON r.generated_by = a.id
            WHERE r.id = ?
        ");
        $stmt->execute([$receipt_id]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    /**
     * Get receipt statistics
     */
    public function getReceiptStats() {
        $stmt = $this->pdo->query("
            SELECT 
                COUNT(*) as total_receipts,
                COUNT(CASE WHEN DATE(created_at) = CURDATE() THEN 1 END) as today_receipts,
                COUNT(CASE WHEN YEARWEEK(created_at) = YEARWEEK(CURDATE()) THEN 1 END) as this_week_receipts,
                COUNT(CASE WHEN YEAR(created_at) = YEAR(CURDATE()) AND MONTH(created_at) = MONTH(CURDATE()) THEN 1 END) as this_month_receipts,
                SUM(amount) as total_amount,
                SUM(CASE WHEN YEAR(created_at) = YEAR(CURDATE()) THEN amount ELSE 0 END) as this_year_amount
            FROM receipts
        ");
        
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    /**
     * Delete receipt
     */
    public function deleteReceipt($receipt_id) {
        try {
            $receipt = $this->getReceipt($receipt_id);
            if (!$receipt) {
                return ['success' => false, 'error' => 'Receipt not found'];
            }
            
            // Delete file if exists
            if (file_exists($receipt['file_path'])) {
                unlink($receipt['file_path']);
            }
            
            // Delete from database
            $stmt = $this->pdo->prepare("DELETE FROM receipts WHERE id = ?");
            $stmt->execute([$receipt_id]);
            
            return ['success' => true, 'message' => 'Receipt deleted successfully'];
            
        } catch (Exception $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Regenerate receipt
     */
    public function regenerateReceipt($receipt_id, $template = 'standard') {
        try {
            $receipt = $this->getReceipt($receipt_id);
            if (!$receipt) {
                return ['success' => false, 'error' => 'Receipt not found'];
            }
            
            // Delete old file
            if (file_exists($receipt['file_path'])) {
                unlink($receipt['file_path']);
            }
            
            // Generate new receipt
            $result = $this->generateReceipt($receipt['transaction_id'], $template);
            
            if ($result['success']) {
                // Update database record
                $stmt = $this->pdo->prepare("
                    UPDATE receipts 
                    SET file_path = ?, updated_at = NOW() 
                    WHERE id = ?
                ");
                $stmt->execute([$result['file_path'], $receipt_id]);
            }
            
            return $result;
            
        } catch (Exception $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
}

// Invoice-specific methods (extending receipts)
class InvoiceService extends ReceiptService {
    
    /**
     * Generate invoice (before payment)
     */
    public function generateInvoice($member_id, $items, $due_date = null) {
        try {
            $due_date = $due_date ?? date('Y-m-d', strtotime('+30 days'));
            
            // Create invoice record
            $invoice_number = $this->generateInvoiceNumber();
            $total_amount = array_sum(array_column($items, 'amount'));
            
            $stmt = $this->pdo->prepare("
                INSERT INTO invoices (
                    invoice_number, member_id, total_amount, due_date, 
                    status, items, created_at
                ) VALUES (?, ?, ?, ?, 'pending', ?, NOW())
            ");
            
            $stmt->execute([
                $invoice_number,
                $member_id,
                $total_amount,
                $due_date,
                json_encode($items)
            ]);
            
            $invoice_id = $this->pdo->lastInsertId();
            
            // Generate PDF
            $pdf_content = $this->generateInvoicePDF($invoice_id);
            
            return [
                'success' => true,
                'invoice_id' => $invoice_id,
                'invoice_number' => $invoice_number,
                'total_amount' => $total_amount
            ];
            
        } catch (Exception $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Generate invoice number
     */
    private function generateInvoiceNumber() {
        $year = date('Y');
        $month = date('m');
        
        $stmt = $this->pdo->prepare("
            SELECT MAX(CAST(SUBSTRING(invoice_number, -4) AS UNSIGNED)) as max_seq
            FROM invoices 
            WHERE invoice_number LIKE ? AND YEAR(created_at) = ? AND MONTH(created_at) = ?
        ");
        $stmt->execute(["KSO-INV-{$year}{$month}%", $year, $month]);
        $result = $stmt->fetch();
        
        $next_seq = ($result['max_seq'] ?? 0) + 1;
        $sequence = str_pad($next_seq, 4, '0', STR_PAD_LEFT);
        
        return "KSO-INV-{$year}{$month}{$sequence}";
    }
    
    /**
     * Generate invoice PDF using TCPDF
     */
    private function generateInvoicePDF($invoice_id) {
        try {
            // Get invoice details
            $stmt = $this->pdo->prepare("
                SELECT i.*, m.first_name, m.last_name, m.kso_id, m.email, m.address
                FROM invoices i
                LEFT JOIN members m ON i.member_id = m.id
                WHERE i.id = ?
            ");
            $stmt->execute([$invoice_id]);
            $invoice = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$invoice) {
                throw new Exception('Invoice not found');
            }
            
            // Decode items
            $invoice['items'] = json_decode($invoice['items'], true);
            
            // Initialize PDF service
            $pdf = new PDFService();
            
            // Set document information
            $pdf->SetTitle('Invoice - ' . $invoice['invoice_number']);
            $pdf->SetSubject('Invoice');
            
            // Generate the invoice
            $pdf->generateInvoice($invoice);
            
            // Save PDF file
            $file_path = $this->getInvoiceFilePath($invoice['invoice_number']);
            $pdf->savePDF($file_path);
            
            // Update database with file path
            $stmt = $this->pdo->prepare("UPDATE invoices SET file_path = ? WHERE id = ?");
            $stmt->execute([$file_path, $invoice_id]);
            
            return $pdf->getPDFString();
            
        } catch (Exception $e) {
            error_log("Invoice PDF generation error: " . $e->getMessage());
            throw new Exception("Failed to generate invoice PDF: " . $e->getMessage());
        }
    }
    
    /**
     * Get invoice file path
     */
    private function getInvoiceFilePath($invoice_number) {
        $year = date('Y');
        return $this->invoice_dir . $year . '/' . $invoice_number . '.pdf';
    }
}

?>
