# KSO Chandigarh HTML Files CSS/JS Reference Checker and Fixer
# This script checks all HTML files for proper CSS and JS references

Write-Host "KSO Chandigarh - HTML Files CSS/JS Reference Check" -ForegroundColor Green
Write-Host "=" * 60 -ForegroundColor Green

# Define expected CSS files in order
$expectedCSS = @(
    "../assets/css/source-sans-3.css",
    "../assets/css/bootstrap-icons.min.css", 
    "../assets/css/overlayscrollbars.min.css",
    "../assets/css/adminlte.min.css",
    "../assets/css/kso-chandigarh.css"
)

# Define expected JS files in order
$expectedJS = @(
    "../assets/js/overlayscrollbars.browser.es6.min.js",
    "../assets/js/popper.min.js",
    "../assets/js/bootstrap.min.js", 
    "../assets/js/adminlte.min.js"
)

# Get all HTML files in frontend/pages
$htmlFiles = Get-ChildItem "frontend\pages\*.html" -Recurse

Write-Host "`nAnalyzing HTML Files:" -ForegroundColor Yellow
Write-Host "-" * 60

foreach ($file in $htmlFiles) {
    $relativePath = $file.FullName.Replace($PWD.Path + "\", "")
    Write-Host "`nChecking: $relativePath" -ForegroundColor Cyan
    
    $content = Get-Content $file.FullName -Raw
    $issues = @()
    
    # Check for each expected CSS file
    foreach ($css in $expectedCSS) {
        $cssFile = $css.Split("/")[-1]
        if ($content -notmatch [regex]::Escape($css)) {
            $issues += "Missing CSS: $cssFile"
        }
    }
    
    # Check for each expected JS file  
    foreach ($js in $expectedJS) {
        $jsFile = $js.Split("/")[-1]
        if ($content -notmatch [regex]::Escape($js)) {
            $issues += "Missing JS: $jsFile"
        }
    }
    
    # Check for incorrect references
    if ($content -match "jquery\.min\.js") {
        $issues += "Incorrect: jquery.min.js (file not found)"
    }
    if ($content -match "bootstrap\.bundle\.min\.js") {
        $issues += "Incorrect: bootstrap.bundle.min.js (should be bootstrap.min.js)"
    }
    if ($content -match "overlayScrollbars\.min\.js") {
        $issues += "Incorrect: overlayScrollbars.min.js (should be overlayscrollbars.browser.es6.min.js)"
    }
    
    # Report results
    if ($issues.Count -eq 0) {
        Write-Host "  ✓ All references correct" -ForegroundColor Green
    } else {
        Write-Host "  ✗ Issues found:" -ForegroundColor Red
        foreach ($issue in $issues) {
            Write-Host "    - $issue" -ForegroundColor Yellow
        }
    }
}

Write-Host "`nCSS/JS Reference Check Complete!" -ForegroundColor Green

# Additional checks
Write-Host "`nAdditional Checks:" -ForegroundColor Yellow
Write-Host "-" * 60

# Check if all referenced files actually exist
$cssFiles = @(
    "frontend\assets\css\source-sans-3.css",
    "frontend\assets\css\bootstrap-icons.min.css",
    "frontend\assets\css\overlayscrollbars.min.css", 
    "frontend\assets\css\adminlte.min.css",
    "frontend\assets\css\kso-chandigarh.css"
)

$jsFiles = @(
    "frontend\assets\js\overlayscrollbars.browser.es6.min.js",
    "frontend\assets\js\popper.min.js",
    "frontend\assets\js\bootstrap.min.js",
    "frontend\assets\js\adminlte.min.js"
)

Write-Host "CSS Files:" -ForegroundColor Cyan
foreach ($css in $cssFiles) {
    if (Test-Path $css) {
        Write-Host "  ✓ $($css.Split('\')[-1])" -ForegroundColor Green
    } else {
        Write-Host "  ✗ $($css.Split('\')[-1]) - NOT FOUND" -ForegroundColor Red
    }
}

Write-Host "`nJS Files:" -ForegroundColor Cyan  
foreach ($js in $jsFiles) {
    if (Test-Path $js) {
        Write-Host "  ✓ $($js.Split('\')[-1])" -ForegroundColor Green
    } else {
        Write-Host "  ✗ $($js.Split('\')[-1]) - NOT FOUND" -ForegroundColor Red
    }
}

Write-Host "`nCustom JS Files:" -ForegroundColor Cyan
$customJSFiles = Get-ChildItem "frontend\js\*.js" | Select-Object Name
foreach ($js in $customJSFiles) {
    Write-Host "  ✓ $($js.Name)" -ForegroundColor Green
}

Write-Host "`nAnalysis Complete!" -ForegroundColor Green -BackgroundColor Black
