<?php
require_once '../config/database.php';

header('Content-Type: application/json');

try {
    // Check if user is authenticated
    if (!isAuthenticated()) {
        echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
        exit;
    }

    $database = new Database();
    $conn = $database->getConnection();

    // Function to safely query table
    function safelyQueryTable($conn, $query, $tableName) {
        try {
            // Check if table exists
            $checkTable = "SHOW TABLES LIKE '$tableName'";
            $stmt = $conn->prepare($checkTable);
            $stmt->execute();
            $tableExists = $stmt->rowCount() > 0;
            
            if ($tableExists) {
                $stmt = $conn->query($query);
                $result = $stmt->fetch(PDO::FETCH_ASSOC);
                
                // Convert null values to 0 for numeric fields
                if ($result) {
                    foreach ($result as $key => $value) {
                        if (is_null($value) && (strpos($key, 'total') === 0 || 
                                               strpos($key, 'count') === 0 || 
                                               strpos($key, 'active') === 0 ||
                                               strpos($key, 'pending') === 0 ||
                                               strpos($key, 'upcoming') === 0 ||
                                               strpos($key, 'completed') === 0)) {
                            $result[$key] = 0;
                        }
                    }
                    return $result;
                }
            }
            
            // Return default empty values based on tableName
            return getDefaultEmptyValues($tableName);
        } catch (Exception $e) {
            error_log("Error querying $tableName: " . $e->getMessage());
            return getDefaultEmptyValues($tableName);
        }
    }
    
    // Function to provide default empty values for different tables
    function getDefaultEmptyValues($tableName) {
        switch ($tableName) {
            case 'members':
                return [
                    'total_members' => 0,
                    'active_members' => 0,
                    'pending_members' => 0,
                    'approved_members' => 0
                ];
            case 'events':
                return [
                    'total_events' => 0,
                    'upcoming_events' => 0,
                    'completed_events' => 0
                ];
            case 'transactions':
                return [
                    'total_transactions' => 0,
                    'total_income' => 0,
                    'pending_payments' => 0
                ];
            case 'expenses':
                return [
                    'total_expenses' => 0,
                    'total_expenses_amount' => 0,
                    'pending_expenses' => 0
                ];
            case 'documents':
                return [
                    'total_documents' => 0,
                    'recently_added' => 0
                ];
            case 'id_cards':
                return [
                    'total_cards_issued' => 0,
                    'pending_cards' => 0
                ];
            case 'member_photos':
                return [
                    'total_photos' => 0,
                    'pending_approval' => 0
                ];
            case 'receipts':
                return [
                    'total_receipts' => 0,
                    'recent_receipts' => 0
                ];
            default:
                return [];
        }
    }
    
    // Initialize default values for all stat sections
    $member_stats = getDefaultEmptyValues('members');
    $event_stats = getDefaultEmptyValues('events');
    $finance_stats = getDefaultEmptyValues('transactions');
    $expense_stats = getDefaultEmptyValues('expenses');
    $document_stats = getDefaultEmptyValues('documents');
    $id_card_stats = getDefaultEmptyValues('id_cards');
    $photo_stats = getDefaultEmptyValues('member_photos');
    $receipt_stats = getDefaultEmptyValues('receipts');
    
    // Get member statistics
    $memberQuery = "
        SELECT 
            COUNT(*) as total_members,
            COUNT(CASE WHEN status = 'active' THEN 1 END) as active_members,
            COUNT(CASE WHEN status = 'pending' THEN 1 END) as pending_members,
            COUNT(CASE WHEN status = 'approved' THEN 1 END) as approved_members
        FROM members
    ";
    $memberResult = safelyQueryTable($conn, $memberQuery, 'members');
    if (!empty($memberResult)) {
        $member_stats = $memberResult;
    }
    
    // Get event statistics
    $eventQuery = "
        SELECT 
            COUNT(*) as total_events,
            COUNT(CASE WHEN status = 'upcoming' THEN 1 END) as upcoming_events,
            COUNT(CASE WHEN status = 'completed' THEN 1 END) as completed_events
        FROM events
    ";
    $eventResult = safelyQueryTable($conn, $eventQuery, 'events');
    if (!empty($eventResult)) {
        $event_stats = $eventResult;
    }
    
    // Get financial statistics
    $financeQuery = "
        SELECT 
            COUNT(*) as total_transactions,
            COALESCE(SUM(CASE WHEN type = 'income' AND status = 'completed' THEN amount ELSE 0 END), 0) as total_income,
            COUNT(CASE WHEN status = 'pending' THEN 1 END) as pending_payments
        FROM transactions
    ";
    $financeResult = safelyQueryTable($conn, $financeQuery, 'transactions');
    if (!empty($financeResult)) {
        $finance_stats = $financeResult;
    }
    
    // Get expense statistics
    $expenseQuery = "
        SELECT 
            COUNT(CASE WHEN type = 'expense' THEN 1 END) as total_expenses,
            COALESCE(SUM(CASE WHEN type = 'expense' AND status = 'completed' THEN amount ELSE 0 END), 0) as total_expenses_amount,
            COUNT(CASE WHEN type = 'expense' AND status = 'pending' THEN 1 END) as pending_expenses
        FROM transactions
    ";
    $expenseResult = safelyQueryTable($conn, $expenseQuery, 'transactions');
    if (!empty($expenseResult)) {
        $expense_stats = $expenseResult;
    }
    
    // Get document statistics
    $documentQuery = "
        SELECT 
            COUNT(*) as total_documents,
            COUNT(CASE WHEN created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY) THEN 1 END) as recently_added
        FROM documents
    ";
    $documentResult = safelyQueryTable($conn, $documentQuery, 'documents');
    if (!empty($documentResult)) {
        $document_stats = $documentResult;
    }
    
    // Get ID card statistics
    $idCardQuery = "
        SELECT 
            COUNT(*) as total_cards_issued,
            COUNT(CASE WHEN status = 'pending' THEN 1 END) as pending_cards
        FROM id_cards
    ";
    $idCardResult = safelyQueryTable($conn, $idCardQuery, 'id_cards');
    if (!empty($idCardResult)) {
        $id_card_stats = $idCardResult;
    }
    
    // Get photo statistics
    $photoQuery = "
        SELECT 
            COUNT(*) as total_photos,
            COUNT(CASE WHEN approval_status = 'pending' THEN 1 END) as pending_approval
        FROM member_photos
    ";
    $photoResult = safelyQueryTable($conn, $photoQuery, 'member_photos');
    if (!empty($photoResult)) {
        $photo_stats = $photoResult;
    }
    
    // Get receipt statistics
    $receiptQuery = "
        SELECT 
            COUNT(*) as total_receipts,
            COUNT(CASE WHEN created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY) THEN 1 END) as recent_receipts
        FROM receipts
    ";
    $receiptResult = safelyQueryTable($conn, $receiptQuery, 'receipts');
    if (!empty($receiptResult)) {
        $receipt_stats = $receiptResult;
    }

    // If chart data was requested
    $chart_data = null;
    if (isset($_GET['chart']) && $_GET['chart'] === 'members_growth') {
        // Get monthly member signup data for the past 12 months
        $chartQuery = "
            SELECT 
                DATE_FORMAT(created_at, '%Y-%m') as month,
                COUNT(*) as new_members
            FROM members
            WHERE created_at >= DATE_SUB(NOW(), INTERVAL 12 MONTH)
            GROUP BY DATE_FORMAT(created_at, '%Y-%m')
            ORDER BY month ASC
        ";
        
        try {
            $stmt = $conn->query($chartQuery);
            $chartResults = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Format for ChartJS
            $labels = [];
            $values = [];
            
            // Create a full 12-month dataset with zeroes for months with no signups
            $end = new DateTime();
            $start = (new DateTime())->modify('-11 months')->modify('first day of this month');
            $period = new DatePeriod(
                $start,
                new DateInterval('P1M'),
                $end
            );
            
            $monthData = [];
            foreach ($chartResults as $row) {
                $monthData[$row['month']] = (int)$row['new_members'];
            }
            
            foreach ($period as $date) {
                $monthKey = $date->format('Y-m');
                $labels[] = $date->format('M Y');
                $values[] = isset($monthData[$monthKey]) ? $monthData[$monthKey] : 0;
            }
            
            $chart_data = [
                'labels' => $labels,
                'values' => $values
            ];
        } catch (Exception $e) {
            error_log("Error generating chart data: " . $e->getMessage());
        }
    }

    echo json_encode([
        'success' => true, 
        'data' => [
            'member_stats' => isset($member_stats) ? $member_stats : [],
            'event_stats' => isset($event_stats) ? $event_stats : [],
            'finance_stats' => isset($finance_stats) ? $finance_stats : [],
            'expense_stats' => isset($expense_stats) ? $expense_stats : [],
            'document_stats' => isset($document_stats) ? $document_stats : [],
            'id_card_stats' => isset($id_card_stats) ? $id_card_stats : [],
            'photo_stats' => isset($photo_stats) ? $photo_stats : [],
            'receipt_stats' => isset($receipt_stats) ? $receipt_stats : [],
            'chart_data' => $chart_data
        ]
    ]);

} catch (Exception $e) {
    echo json_encode([
        'success' => false, 
        'message' => 'Database error: ' . $e->getMessage()
    ]);
}
?>
