<?php
/**
 * API endpoint to get or update notification settings
 * 
 * GET: Retrieve notification settings for a user
 * POST: Update notification settings for a user
 * 
 * Parameters:
 * - user_id: ID of the user
 * - user_type: Type of the user ('admin' or 'member')
 * 
 * POST Parameters for update:
 * - browser_notifications: Enable browser notifications (0/1)
 * - sound_notifications: Enable sound notifications (0/1)
 * - email_notifications: Enable email notifications (0/1)
 * - notification_types: Array of notification types to receive
 */

// Include dependencies
require_once '../../config/database.php';
require_once '../../config/auth_enhanced.php';
require_once '../../services/notification/NotificationService.php';

use Services\Notification\NotificationService;

// Check if user is authenticated
checkAuthentication();

// Default response structure
$response = [
    'success' => false,
    'data' => [],
    'message' => 'Error processing notification settings'
];

try {
    // Get user ID and type
    if (isset($_REQUEST['user_id']) && isset($_REQUEST['user_type'])) {
        // Specific user (admin only can access other users' settings)
        if ($_SESSION['user_type'] !== 'admin' && 
            ($_REQUEST['user_id'] != $_SESSION['user_id'] || 
             $_REQUEST['user_type'] != $_SESSION['user_type'])) {
            // Non-admin users can only access their own settings
            throw new Exception('Access denied: You can only access your own settings');
        }
        
        $userId = (int)$_REQUEST['user_id'];
        $userType = $_REQUEST['user_type'];
    } else {
        // Current user
        $userId = $_SESSION['user_id'];
        $userType = $_SESSION['user_type'];
    }
    
    // Validate user info
    if ($userId <= 0 || empty($userType) || !in_array($userType, ['admin', 'member'])) {
        throw new Exception('Invalid user information');
    }
    
    // Create notification service
    $notificationService = new NotificationService();
    
    // Handle GET request (retrieve settings)
    if ($_SERVER['REQUEST_METHOD'] === 'GET') {
        $result = $notificationService->getSettings($userId, $userType);
        $response = $result;
    }
    // Handle POST request (update settings)
    elseif ($_SERVER['REQUEST_METHOD'] === 'POST') {
        // Get POST data
        $postData = json_decode(file_get_contents('php://input'), true);
        
        // If POST data is empty, try regular POST
        if (empty($postData)) {
            $postData = $_POST;
            
            // Convert form values to the right format
            if (isset($postData['notification_types']) && is_array($postData['notification_types'])) {
                // Already an array from JSON input
            } elseif (isset($postData['notification_types'])) {
                // Convert string to array
                $postData['notification_types'] = explode(',', $postData['notification_types']);
            } else {
                $postData['notification_types'] = [];
            }
            
            // Convert checkboxes to 1/0
            $postData['browser_notifications'] = isset($postData['browser_notifications']) ? 1 : 0;
            $postData['sound_notifications'] = isset($postData['sound_notifications']) ? 1 : 0;
            $postData['email_notifications'] = isset($postData['email_notifications']) ? 1 : 0;
        }
        
        // Validate required fields
        $requiredFields = ['browser_notifications', 'sound_notifications', 'email_notifications', 'notification_types'];
        foreach ($requiredFields as $field) {
            if (!isset($postData[$field])) {
                throw new Exception("Field '$field' is required");
            }
        }
        
        // Prepare settings data
        $settings = [
            'browser_notifications' => (int)$postData['browser_notifications'],
            'sound_notifications' => (int)$postData['sound_notifications'],
            'email_notifications' => (int)$postData['email_notifications'],
            'notification_types' => $postData['notification_types']
        ];
        
        // Update settings
        $result = $notificationService->updateSettings($userId, $userType, $settings);
        $response = $result;
    }
    // Handle other methods
    else {
        http_response_code(405);
        $response['message'] = 'Method not allowed. Use GET or POST.';
    }
    
} catch (Exception $e) {
    $response['message'] = 'Error: ' . $e->getMessage();
}

// Send JSON response
header('Content-Type: application/json');
echo json_encode($response);
exit;
