<?php
session_start();
require_once '../config/config.php';

header('Content-Type: application/json');

if (!isset($_SESSION['admin_id'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit();
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit();
}

// Get payment gateway and merchant ID
$gateway = $_POST['gateway'] ?? '';
$merchant_id = $_POST['merchant_id'] ?? '';

if (empty($gateway) || empty($merchant_id)) {
    echo json_encode(['success' => false, 'message' => 'Gateway and Merchant ID are required']);
    exit();
}

try {
    $conn = new PDO($dsn, $username, $password);
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    // Save payment gateway settings
    $stmt = $conn->prepare("
        SELECT COUNT(*) FROM system_settings 
        WHERE category = 'payment' AND setting_key = ?
    ");
    $stmt->execute([$gateway]);
    $exists = $stmt->fetchColumn() > 0;
    
    if ($exists) {
        $stmt = $conn->prepare("
            UPDATE system_settings 
            SET value = ?, updated_at = CURRENT_TIMESTAMP
            WHERE category = 'payment' AND setting_key = ?
        ");
        $stmt->execute([$merchant_id, $gateway]);
    } else {
        $stmt = $conn->prepare("
            INSERT INTO system_settings (category, setting_key, value, created_at, updated_at)
            VALUES ('payment', ?, ?, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
        ");
        $stmt->execute([$gateway, $merchant_id]);
    }
    
    // Set this gateway as the active one
    $stmt = $conn->prepare("
        SELECT COUNT(*) FROM system_settings 
        WHERE category = 'payment' AND setting_key = 'active_gateway'
    ");
    $stmt->execute();
    $active_exists = $stmt->fetchColumn() > 0;
    
    if ($active_exists) {
        $stmt = $conn->prepare("
            UPDATE system_settings 
            SET value = ?, updated_at = CURRENT_TIMESTAMP
            WHERE category = 'payment' AND setting_key = 'active_gateway'
        ");
        $stmt->execute([$gateway]);
    } else {
        $stmt = $conn->prepare("
            INSERT INTO system_settings (category, setting_key, value, created_at, updated_at)
            VALUES ('payment', 'active_gateway', ?, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
        ");
        $stmt->execute([$gateway]);
    }
    
    // Log the action
    $stmt = $conn->prepare("
        INSERT INTO admin_logs (admin_id, action, details, created_at)
        VALUES (?, 'update_payment_settings', ?, CURRENT_TIMESTAMP)
    ");
    $stmt->execute([
        $_SESSION['admin_id'],
        'Updated payment gateway settings for ' . $gateway
    ]);
    
    // Simulate a test connection
    $test_result = true;
    $error_message = '';
    
    // Simple validation for demonstration purposes
    switch ($gateway) {
        case 'razorpay':
            if (strlen($merchant_id) < 8) {
                $test_result = false;
                $error_message = 'Invalid Razorpay Merchant ID format';
            }
            break;
        case 'paytm':
            if (strlen($merchant_id) < 10) {
                $test_result = false;
                $error_message = 'Invalid Paytm Merchant ID format';
            }
            break;
        case 'paypal':
            if (!filter_var($merchant_id, FILTER_VALIDATE_EMAIL)) {
                $test_result = false;
                $error_message = 'PayPal Merchant ID should be a valid email';
            }
            break;
        case 'stripe':
            if (strpos($merchant_id, 'pk_') !== 0 && strpos($merchant_id, 'sk_') !== 0) {
                $test_result = false;
                $error_message = 'Invalid Stripe API key format';
            }
            break;
    }
    
    if ($test_result) {
        echo json_encode([
            'success' => true,
            'message' => 'Connection to ' . ucfirst($gateway) . ' successful'
        ]);
    } else {
        echo json_encode([
            'success' => false,
            'message' => $error_message
        ]);
    }
    
} catch (PDOException $e) {
    echo json_encode([
        'success' => false,
        'message' => 'Database error: ' . $e->getMessage()
    ]);
}
?>
