/**
 * Quick Print Button for KSO Chandigarh Admin Panel
 * - Adds a floating print button to pages that can be used to print any visible content
 */

(function() {
    // Create a floating print button
    document.addEventListener('DOMContentLoaded', function() {
        // Only add the button if we're not already in a print view
        if (window.location.href.indexOf('print=true') === -1) {
            createQuickPrintButton();
        }
    });

    function createQuickPrintButton() {
        const button = document.createElement('button');
        button.id = 'kso-quick-print-btn';
        button.innerHTML = '<i class="fas fa-print"></i>';
        button.title = 'Print Current Page';
        
        // Add styles for the button
        const style = document.createElement('style');
        style.textContent = `
            #kso-quick-print-btn {
                position: fixed;
                bottom: 20px;
                right: 20px;
                width: 50px;
                height: 50px;
                border-radius: 50%;
                background: #007bff;
                color: white;
                border: none;
                box-shadow: 0 2px 10px rgba(0, 0, 0, 0.2);
                cursor: pointer;
                z-index: 1000;
                display: flex;
                align-items: center;
                justify-content: center;
                font-size: 20px;
                transition: all 0.3s ease;
            }
            
            #kso-quick-print-btn:hover {
                background: #0056b3;
                transform: scale(1.1);
                box-shadow: 0 4px 15px rgba(0, 0, 0, 0.3);
            }
            
            #kso-quick-print-btn:active {
                transform: scale(0.95);
            }
            
            @media print {
                #kso-quick-print-btn {
                    display: none !important;
                }
            }
        `;
        
        document.head.appendChild(style);
        document.body.appendChild(button);
        
        // Add click event to handle printing
        button.addEventListener('click', function() {
            if (window.ksoPrint) {
                // If our custom print system is available, show options
                if (typeof ksoPrint.showPrintOptions === 'function') {
                    // Find the main content table or div to print
                    const mainTable = document.querySelector('.card .table') || 
                                      document.querySelector('.card-body .table');
                    
                    if (mainTable) {
                        // Check if we have the advanced printTable function from print-system.js
                        if (typeof printTable === 'function') {
                            printTable(mainTable);
                        } else {
                            ksoPrint.showPrintOptions(mainTable);
                        }
                    } else {
                        // Check if we're on a member page
                        if (window.location.href.includes('members.php') && typeof printMemberCards === 'function') {
                            printMemberCards();
                        } 
                        // Check if we're on a finance or reports page
                        else if ((window.location.href.includes('finance') || window.location.href.includes('reports')) && 
                                typeof printFinancialReport === 'function') {
                            printFinancialReport();
                        }
                        // If no special case, just use window.print()
                        else {
                            prepareForPrint();
                            window.print();
                        }
                    }
                } else {
                    // Fallback to basic print
                    prepareForPrint();
                    window.print();
                }
            } else {
                // No custom print system, use window.print()
                prepareForPrint();
                window.print();
            }
        });
    }
    
    function prepareForPrint() {
        // Add KSO header to the print version if not already present
        if (!document.getElementById('kso-print-header')) {
            const printHeader = document.createElement('div');
            printHeader.id = 'kso-print-header';
            printHeader.className = 'd-none d-print-block';
            printHeader.innerHTML = `
                <div style="text-align: center; margin-bottom: 20px;">
                    <img src="public/KSO LOGO.png" alt="KSO Logo" style="height: 80px; margin-bottom: 10px;">
                    <h2 style="margin: 5px 0; font-weight: bold;">KUKI STUDENTS' ORGANISATION</h2>
                    <h3 style="margin: 5px 0;">Chandigarh</h3>
                </div>
                <hr style="border-top: 2px solid #333;">
            `;
            document.body.insertBefore(printHeader, document.body.firstChild);
        }
        
        // Add watermark if not already present
        if (!document.getElementById('kso-print-watermark')) {
            const watermark = document.createElement('div');
            watermark.id = 'kso-print-watermark';
            watermark.className = 'd-none d-print-block';
            watermark.style.position = 'fixed';
            watermark.style.top = '0';
            watermark.style.left = '0';
            watermark.style.width = '100%';
            watermark.style.height = '100%';
            watermark.style.display = 'flex';
            watermark.style.justifyContent = 'center';
            watermark.style.alignItems = 'center';
            watermark.style.pointerEvents = 'none';
            watermark.style.zIndex = '-1000';
            watermark.innerHTML = `
                <img src="public/KSO LOGO.png" alt="KSO Logo" style="opacity: 0.07; width: 50%; max-width: 400px; transform: rotate(-30deg);">
            `;
            document.body.appendChild(watermark);
        }

        // Hide elements that shouldn't be printed
        const elementsToHide = [
            '.main-header', '.main-sidebar', '.main-footer', 
            '.btn-delete', '.btn-edit', '.btn-view',
            '.dropdown-menu', '.pagination', '.card-tools',
            '.no-print'
        ];
        
        // Add a temporary style for printing
        const printStyle = document.createElement('style');
        printStyle.id = 'kso-temp-print-style';
        printStyle.textContent = `
            @media print {
                ${elementsToHide.join(', ')} {
                    display: none !important;
                }
                
                .content-wrapper {
                    margin-left: 0 !important;
                    padding: 0 !important;
                    width: 100% !important;
                }
                
                .card {
                    box-shadow: none !important;
                    border: 1px solid #ddd !important;
                    margin-bottom: 20px !important;
                }
                
                .card-header {
                    background: #f8f9fa !important;
                    font-weight: bold !important;
                }
                
                .table {
                    width: 100% !important;
                    border-collapse: collapse !important;
                }
                
                .table th, .table td {
                    border: 1px solid #ddd !important;
                    padding: 8px !important;
                }
                
                .table th {
                    background-color: #f2f2f2 !important;
                    color: black !important;
                    font-weight: bold !important;
                }
                
                body {
                    padding: 20px !important;
                    font-family: Arial, sans-serif !important;
                }
                
                /* Handle member card display */
                .member-card {
                    max-width: 100% !important;
                    page-break-inside: avoid !important;
                }
                
                /* Format page title */
                .content-header h1 {
                    font-size: 24px !important;
                    text-align: center !important;
                    margin: 20px 0 !important;
                }
                
                /* Make sure QR codes are visible */
                .qr-code img {
                    display: block !important;
                    max-width: 100% !important;
                }
                
                /* Remove URL display for links */
                a[href]:after {
                    content: "" !important;
                }
                
                /* Add print footer with date and page numbers */
                @page {
                    margin: 0.5in;
                    @bottom-right {
                        content: "Page " counter(page) " of " counter(pages);
                    }
                    @bottom-left {
                        content: "Printed on " attr(data-print-date);
                    }
                }
            }
        `;
        
        // Add current date attribute for print footer
        document.documentElement.setAttribute('data-print-date', new Date().toLocaleDateString());
        `;
        
        document.head.appendChild(printStyle);
        
        // Clean up after printing
        window.addEventListener('afterprint', function() {
            const tempStyle = document.getElementById('kso-temp-print-style');
            if (tempStyle) {
                tempStyle.remove();
            }
        });
    }
})();
