/**
 * KSO Chandigarh Admin Panel - Enhanced Assignment System
 * 
 * A comprehensive JavaScript module for managing member assignments
 * Features:
 * - Real-time updates using SSE (Server-Sent Events)
 * - Assignment creation, editing, deletion
 * - Bulk operations for multiple assignments
 * - Filtering and searching capabilities
 * - Statistical visualization with Chart.js
 * - Mobile-responsive design adaptations
 * - Email notification integration
 */

// Initialize Assignment System as a global namespace
window.assignmentSystem = (function() {
  'use strict';
  
  // Private variables
  let assignmentsTable;
  let assignments = [];
  let selectedFilter = 'all';
  let selectedPriorityFilter = 'all';
  let charts = {};
  let eventSource; // For SSE connection
  
  // Cache DOM elements
  const $assignmentsTableBody = $('#assignmentsTableBody');
  const $noAssignmentsRow = $('#noAssignmentsRow');
  const $recentActivityList = $('#recentActivityList');
  const $dueSoonList = $('#dueSoonList');
  
  // Stats elements
  const $totalAssignments = $('#totalAssignments');
  const $completedAssignments = $('#completedAssignments');
  const $pendingAssignments = $('#pendingAssignments');
  const $overdueAssignments = $('#overdueAssignments');
  
  // Toast notification helper
  const showToast = (message, type = 'success', title = '') => {
    const Toast = Swal.mixin({
      toast: true,
      position: 'top-end',
      showConfirmButton: false,
      timer: 3000
    });
    
    Toast.fire({
      icon: type,
      title: title || message
    });
  };
  
  // Error handler
  const handleError = (error, fallbackMessage = 'An error occurred') => {
    console.error('Assignment System Error:', error);
    let errorMessage = fallbackMessage;
    
    if (error.responseJSON && error.responseJSON.message) {
      errorMessage = error.responseJSON.message;
    } else if (error.statusText) {
      errorMessage = `${fallbackMessage}: ${error.statusText}`;
    }
    
    showToast(errorMessage, 'error');
  };
  
  // Format a date for display
  const formatDate = (dateString) => {
    if (!dateString) return 'N/A';
    
    const date = new Date(dateString);
    return date.toLocaleString('en-US', { 
      year: 'numeric', 
      month: 'short', 
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    });
  };
  
  // Check if a date is in the past
  const isOverdue = (dateString) => {
    if (!dateString) return false;
    
    const dueDate = new Date(dateString);
    const now = new Date();
    
    return dueDate < now;
  };
  
  // Format a priority label with icon
  const formatPriorityLabel = (priority) => {
    let icon, color;
    
    switch (priority) {
      case 'urgent':
        icon = 'exclamation-circle';
        color = 'danger';
        break;
      case 'high':
        icon = 'arrow-up';
        color = 'warning';
        break;
      case 'normal':
        icon = 'minus';
        color = 'info';
        break;
      case 'low':
        icon = 'arrow-down';
        color = 'secondary';
        break;
      default:
        icon = 'question';
        color = 'secondary';
    }
    
    return `<span class="badge badge-${color}"><i class="fas fa-${icon} mr-1"></i> ${priority.charAt(0).toUpperCase() + priority.slice(1)}</span>`;
  };
  
  // Format a status label with icon
  const formatStatusLabel = (status) => {
    let icon, color;
    
    switch (status) {
      case 'active':
        icon = 'clock';
        color = 'warning';
        break;
      case 'completed':
        icon = 'check-circle';
        color = 'success';
        break;
      case 'cancelled':
        icon = 'ban';
        color = 'secondary';
        break;
      case 'overdue':
        icon = 'exclamation-triangle';
        color = 'danger';
        break;
      default:
        icon = 'question';
        color = 'secondary';
    }
    
    return `<span class="badge badge-${color}"><i class="fas fa-${icon} mr-1"></i> ${status.charAt(0).toUpperCase() + status.slice(1)}</span>`;
  };
  
  // Format an assignment type label
  const formatTypeLabel = (type) => {
    let icon;
    
    switch (type) {
      case 'review':
        icon = 'search';
        break;
      case 'approval':
        icon = 'thumbs-up';
        break;
      case 'verification':
        icon = 'check-double';
        break;
      case 'follow_up':
        icon = 'phone';
        break;
      default:
        icon = 'tasks';
    }
    
    return `<span><i class="fas fa-${icon} mr-1"></i> ${type.replace('_', ' ').charAt(0).toUpperCase() + type.replace('_', ' ').slice(1)}</span>`;
  };
  
  // Format due date with warning if close or overdue
  const formatDueDate = (dueDate) => {
    if (!dueDate) return 'No due date';
    
    const now = new Date();
    const due = new Date(dueDate);
    const diffTime = due - now;
    const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
    
    let formattedDate = formatDate(dueDate);
    
    if (diffDays < 0) {
      // Overdue
      return `<span class="due-date-warning"><i class="fas fa-exclamation-triangle mr-1"></i> ${formattedDate} (${Math.abs(diffDays)} days overdue)</span>`;
    } else if (diffDays === 0) {
      // Due today
      return `<span class="text-warning"><i class="fas fa-clock mr-1"></i> ${formattedDate} (Today)</span>`;
    } else if (diffDays <= 2) {
      // Due soon
      return `<span class="text-warning"><i class="fas fa-clock mr-1"></i> ${formattedDate} (${diffDays} days left)</span>`;
    } else {
      // Regular date
      return formattedDate;
    }
  };
  
  // Create a row for the assignments table
  const createAssignmentRow = (assignment) => {
    const isSelected = assignment.selected ? 'checked' : '';
    const priorityClass = `priority-${assignment.priority}`;
    
    return `
      <tr class="assignment-row ${priorityClass}" data-id="${assignment.id}" data-status="${assignment.status}" data-priority="${assignment.priority}">
        <td><input type="checkbox" class="assignment-checkbox" data-id="${assignment.id}" ${isSelected}></td>
        <td>${assignment.id}</td>
        <td>${assignment.title}</td>
        <td>
          <a href="members.php?id=${assignment.member_id}" target="_blank">
            ${assignment.member_name}
          </a>
        </td>
        <td>${assignment.assigned_to_name}</td>
        <td>${formatTypeLabel(assignment.assignment_type)}</td>
        <td>${formatPriorityLabel(assignment.priority)}</td>
        <td>${formatStatusLabel(assignment.status)}</td>
        <td>${formatDueDate(assignment.due_date)}</td>
        <td>
          <div class="btn-group action-buttons">
            <button type="button" class="btn btn-sm btn-info view-assignment-btn" data-id="${assignment.id}" title="View Details">
              <i class="fas fa-eye"></i>
            </button>
            <button type="button" class="btn btn-sm btn-primary edit-assignment-btn" data-id="${assignment.id}" title="Edit">
              <i class="fas fa-edit"></i>
            </button>
            <button type="button" class="btn btn-sm btn-success complete-assignment-btn" data-id="${assignment.id}" title="Mark Completed">
              <i class="fas fa-check"></i>
            </button>
            <button type="button" class="btn btn-sm btn-danger delete-assignment-btn" data-id="${assignment.id}" title="Delete">
              <i class="fas fa-trash"></i>
            </button>
          </div>
        </td>
      </tr>
    `;
  };
  
  // Create a recent activity item
  const createActivityItem = (activity) => {
    const date = formatDate(activity.timestamp);
    let actionText = '';
    let icon = '';
    
    switch (activity.action) {
      case 'created':
        actionText = 'created assignment';
        icon = 'plus-circle text-success';
        break;
      case 'updated':
        actionText = 'updated assignment';
        icon = 'edit text-primary';
        break;
      case 'completed':
        actionText = 'completed assignment';
        icon = 'check-circle text-success';
        break;
      case 'deleted':
        actionText = 'deleted assignment';
        icon = 'trash text-danger';
        break;
      case 'assigned':
        actionText = `assigned to ${activity.assigned_to_name}`;
        icon = 'user-tag text-info';
        break;
      default:
        actionText = 'modified assignment';
        icon = 'sync text-secondary';
    }
    
    return `
      <li class="item">
        <div class="product-img">
          <i class="fas fa-${icon} fa-2x"></i>
        </div>
        <div class="product-info">
          <a href="javascript:void(0)" class="product-title">
            ${activity.admin_name}
            <span class="badge badge-info float-right">${date}</span>
          </a>
          <span class="product-description">
            ${actionText} "${activity.assignment_title}" for ${activity.member_name}
          </span>
        </div>
      </li>
    `;
  };
  
  // Create a due soon item
  const createDueSoonItem = (assignment) => {
    const dueDate = formatDate(assignment.due_date);
    const now = new Date();
    const due = new Date(assignment.due_date);
    const diffTime = due - now;
    const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
    
    let dueDaysText = '';
    let badgeClass = 'badge-info';
    
    if (diffDays < 0) {
      dueDaysText = `${Math.abs(diffDays)} days overdue`;
      badgeClass = 'badge-danger';
    } else if (diffDays === 0) {
      dueDaysText = 'Due today';
      badgeClass = 'badge-warning';
    } else {
      dueDaysText = `Due in ${diffDays} days`;
      badgeClass = diffDays <= 2 ? 'badge-warning' : 'badge-info';
    }
    
    return `
      <li class="item">
        <div class="product-img">
          <i class="fas fa-calendar-alt fa-2x text-${diffDays < 0 ? 'danger' : (diffDays <= 2 ? 'warning' : 'info')}"></i>
        </div>
        <div class="product-info">
          <a href="javascript:void(0)" class="product-title view-assignment-link" data-id="${assignment.id}">
            ${assignment.title}
            <span class="badge ${badgeClass} float-right">${dueDaysText}</span>
          </a>
          <span class="product-description">
            For ${assignment.member_name} | Assigned to ${assignment.assigned_to_name}
          </span>
        </div>
      </li>
    `;
  };
  
  // Update the assignments table
  const updateAssignmentsTable = () => {
    // Clear the current table content
    $assignmentsTableBody.empty();
    
    // Apply filters to assignments
    let filteredAssignments = assignments;
    
    if (selectedFilter !== 'all') {
      filteredAssignments = assignments.filter(assignment => {
        if (selectedFilter === 'overdue') {
          return isOverdue(assignment.due_date) && assignment.status === 'active';
        }
        return assignment.status === selectedFilter;
      });
    }
    
    if (selectedPriorityFilter !== 'all') {
      filteredAssignments = filteredAssignments.filter(assignment => 
        assignment.priority === selectedPriorityFilter
      );
    }
    
    // Check if we have assignments after filtering
    if (filteredAssignments.length === 0) {
      $assignmentsTableBody.html(`
        <tr>
          <td colspan="10" class="text-center">
            <div class="alert alert-info m-3">
              <i class="fas fa-info-circle mr-2"></i> No assignments found for the current filter.
            </div>
          </td>
        </tr>
      `);
      return;
    }
    
    // Add rows for each assignment
    filteredAssignments.forEach(assignment => {
      $assignmentsTableBody.append(createAssignmentRow(assignment));
    });
    
    // Reinitialize DataTable if it exists
    if ($.fn.DataTable.isDataTable('#assignmentsTable')) {
      $('#assignmentsTable').DataTable().destroy();
    }
    
    // Initialize DataTable
    assignmentsTable = $('#assignmentsTable').DataTable({
      "responsive": true,
      "lengthChange": true,
      "autoWidth": false,
      "buttons": ["excel", "pdf", "print"]
    });
    
    // Re-attach event handlers for dynamically created buttons
    attachEventHandlers();
  };
  
  // Update the recent activity list
  const updateRecentActivity = (activities) => {
    if (!activities || activities.length === 0) {
      $recentActivityList.html(`
        <li class="item">
          <div class="product-info">
            <div class="text-center py-3">
              <i class="fas fa-info-circle mr-2"></i> No recent activity found.
            </div>
          </div>
        </li>
      `);
      return;
    }
    
    $recentActivityList.empty();
    
    // Add the five most recent activities
    activities.slice(0, 5).forEach(activity => {
      $recentActivityList.append(createActivityItem(activity));
    });
  };
  
  // Update the due soon list
  const updateDueSoonList = () => {
    // Find active assignments that are due within the next 7 days
    const now = new Date();
    const nextWeek = new Date(now.getTime() + 7 * 24 * 60 * 60 * 1000);
    
    const dueSoonAssignments = assignments.filter(assignment => {
      if (assignment.status !== 'active') return false;
      
      const dueDate = new Date(assignment.due_date);
      return dueDate <= nextWeek;
    });
    
    // Sort by due date (closest first)
    dueSoonAssignments.sort((a, b) => {
      const dateA = new Date(a.due_date);
      const dateB = new Date(b.due_date);
      return dateA - dateB;
    });
    
    if (dueSoonAssignments.length === 0) {
      $dueSoonList.html(`
        <li class="item">
          <div class="product-info">
            <div class="text-center py-3">
              <i class="fas fa-check-circle mr-2"></i> No assignments due soon.
            </div>
          </div>
        </li>
      `);
      return;
    }
    
    $dueSoonList.empty();
    
    // Add the five most urgent assignments
    dueSoonAssignments.slice(0, 5).forEach(assignment => {
      $dueSoonList.append(createDueSoonItem(assignment));
    });
  };
  
  // Update statistics display
  const updateStatistics = (stats) => {
    $totalAssignments.text(stats.total || 0);
    $completedAssignments.text(stats.completed || 0);
    $pendingAssignments.text(stats.active || 0);
    $overdueAssignments.text(stats.overdue || 0);
  };
  
  // Initialize charts
  const initCharts = () => {
    // Status Chart
    const statusCtx = document.getElementById('statusChart').getContext('2d');
    charts.statusChart = new Chart(statusCtx, {
      type: 'pie',
      data: {
        labels: ['Active', 'Completed', 'Cancelled', 'Overdue'],
        datasets: [{
          data: [0, 0, 0, 0],
          backgroundColor: ['#ffc107', '#28a745', '#6c757d', '#dc3545']
        }]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false
      }
    });
    
    // Priority Chart
    const priorityCtx = document.getElementById('priorityChart').getContext('2d');
    charts.priorityChart = new Chart(priorityCtx, {
      type: 'doughnut',
      data: {
        labels: ['Urgent', 'High', 'Normal', 'Low'],
        datasets: [{
          data: [0, 0, 0, 0],
          backgroundColor: ['#dc3545', '#fd7e14', '#17a2b8', '#6c757d']
        }]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false
      }
    });
    
    // Type Chart
    const typeCtx = document.getElementById('typeChart').getContext('2d');
    charts.typeChart = new Chart(typeCtx, {
      type: 'bar',
      data: {
        labels: ['Review', 'Approval', 'Verification', 'Follow Up', 'Other'],
        datasets: [{
          label: 'Assignments by Type',
          data: [0, 0, 0, 0, 0],
          backgroundColor: '#3b8bba'
        }]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        scales: {
          y: {
            beginAtZero: true,
            precision: 0
          }
        }
      }
    });
    
    // Completion Trend Chart (last 7 days)
    const trendCtx = document.getElementById('completionTrendChart').getContext('2d');
    
    // Generate labels for the last 7 days
    const labels = [];
    for (let i = 6; i >= 0; i--) {
      const date = new Date();
      date.setDate(date.getDate() - i);
      labels.push(date.toLocaleDateString('en-US', { month: 'short', day: 'numeric' }));
    }
    
    charts.trendChart = new Chart(trendCtx, {
      type: 'line',
      data: {
        labels: labels,
        datasets: [{
          label: 'Completed',
          data: [0, 0, 0, 0, 0, 0, 0],
          borderColor: '#28a745',
          backgroundColor: 'rgba(40, 167, 69, 0.1)',
          fill: true,
          tension: 0.4
        }, {
          label: 'Created',
          data: [0, 0, 0, 0, 0, 0, 0],
          borderColor: '#17a2b8',
          backgroundColor: 'transparent',
          fill: false,
          tension: 0.4
        }]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        scales: {
          y: {
            beginAtZero: true,
            precision: 0
          }
        }
      }
    });
  };
  
  // Update charts with data
  const updateCharts = (chartData) => {
    if (!chartData) return;
    
    // Update Status Chart
    if (charts.statusChart && chartData.status) {
      charts.statusChart.data.datasets[0].data = [
        chartData.status.active || 0,
        chartData.status.completed || 0,
        chartData.status.cancelled || 0,
        chartData.status.overdue || 0
      ];
      charts.statusChart.update();
    }
    
    // Update Priority Chart
    if (charts.priorityChart && chartData.priority) {
      charts.priorityChart.data.datasets[0].data = [
        chartData.priority.urgent || 0,
        chartData.priority.high || 0,
        chartData.priority.normal || 0,
        chartData.priority.low || 0
      ];
      charts.priorityChart.update();
    }
    
    // Update Type Chart
    if (charts.typeChart && chartData.type) {
      charts.typeChart.data.datasets[0].data = [
        chartData.type.review || 0,
        chartData.type.approval || 0,
        chartData.type.verification || 0,
        chartData.type.follow_up || 0,
        chartData.type.other || 0
      ];
      charts.typeChart.update();
    }
    
    // Update Completion Trend Chart
    if (charts.trendChart && chartData.trend) {
      charts.trendChart.data.datasets[0].data = chartData.trend.completed || [0, 0, 0, 0, 0, 0, 0];
      charts.trendChart.data.datasets[1].data = chartData.trend.created || [0, 0, 0, 0, 0, 0, 0];
      charts.trendChart.update();
    }
  };
  
  // Update batch action buttons based on selection
  const updateBatchActionButtons = () => {
    const hasSelectedAssignments = $('.assignment-checkbox:checked').length > 0;
    $('#batchUpdateBtn').prop('disabled', !hasSelectedAssignments);
  };
  
  // Load members for dropdowns
  const loadMembers = () => {
    $.ajax({
      url: 'api/members/list.php',
      type: 'GET',
      dataType: 'json',
      success: (response) => {
        if (response.success && response.data) {
          const memberSelect = $('#assignMemberId, #bulkMembers');
          memberSelect.empty();
          memberSelect.append('<option value="">Select Member</option>');
          
          response.data.forEach(member => {
            memberSelect.append(`<option value="${member.id}">${member.first_name} ${member.last_name} (${member.email})</option>`);
          });
        }
      },
      error: (xhr) => {
        handleError(xhr, 'Failed to load members');
      }
    });
  };
  
  // Load admins for dropdowns
  const loadAdmins = () => {
    $.ajax({
      url: 'api/admins/list.php',
      type: 'GET',
      dataType: 'json',
      success: (response) => {
        if (response.success && response.data) {
          const adminSelect = $('#assignToAdmin, #bulkAssignToAdmin, #batchUpdateAssignTo');
          adminSelect.empty();
          adminSelect.append('<option value="">Select Admin</option>');
          
          response.data.forEach(admin => {
            adminSelect.append(`<option value="${admin.id}">${admin.name} (${admin.role})</option>`);
          });
        }
      },
      error: (xhr) => {
        handleError(xhr, 'Failed to load admins');
      }
    });
  };
  
  // Load all assignments
  const loadAssignments = () => {
    $.ajax({
      url: 'api/assignments/list.php',
      type: 'GET',
      dataType: 'json',
      success: (response) => {
        if (response.success && response.data) {
          assignments = response.data;
          updateAssignmentsTable();
          updateDueSoonList();
        } else {
          assignments = [];
          $assignmentsTableBody.html(`
            <tr>
              <td colspan="10" class="text-center">
                <div class="alert alert-info m-3">
                  <i class="fas fa-info-circle mr-2"></i> No assignments found.
                </div>
              </td>
            </tr>
          `);
        }
      },
      error: (xhr) => {
        handleError(xhr, 'Failed to load assignments');
      }
    });
  };
  
  // Load assignment statistics
  const loadStatistics = () => {
    $.ajax({
      url: 'api/assignments/stats.php',
      type: 'GET',
      dataType: 'json',
      success: (response) => {
        if (response.success && response.data) {
          updateStatistics(response.data.counts);
          updateCharts(response.data.charts);
          
          // Update recent activity
          if (response.data.recent_activity) {
            updateRecentActivity(response.data.recent_activity);
          }
        }
      },
      error: (xhr) => {
        handleError(xhr, 'Failed to load statistics');
      }
    });
  };
  
  // Create a new assignment
  const createAssignment = () => {
    // Validate the form
    const form = $('#createAssignmentForm')[0];
    if (!form.checkValidity()) {
      form.reportValidity();
      return;
    }
    
    // Get form values
    const data = {
      title: $('#assignmentTitle').val(),
      description: $('#assignmentDescription').val(),
      member_id: $('#assignMemberId').val(),
      assigned_to: $('#assignToAdmin').val(),
      assignment_type: $('#assignmentType').val(),
      priority: $('#assignmentPriority').val(),
      due_date: $('#assignmentDueDate').val(),
      send_email: $('#sendEmailNotification').is(':checked')
    };
    
    // Send request
    $.ajax({
      url: 'api/assignments/create.php',
      type: 'POST',
      data: data,
      dataType: 'json',
      success: (response) => {
        if (response.success) {
          // Show success message
          showToast('Assignment created successfully');
          
          // Close the modal
          $('#createAssignmentModal').modal('hide');
          
          // Refresh data
          loadAssignments();
          loadStatistics();
          
          // Reset form
          $('#createAssignmentForm')[0].reset();
        } else {
          showToast(response.message || 'Failed to create assignment', 'error');
        }
      },
      error: (xhr) => {
        handleError(xhr, 'Failed to create assignment');
      }
    });
  };
  
  // Bulk assign members
  const bulkAssign = () => {
    // Validate the form
    const form = $('#bulkAssignForm')[0];
    if (!form.checkValidity()) {
      form.reportValidity();
      return;
    }
    
    // Get selected member IDs
    const memberIds = $('#bulkMembers').val();
    
    if (!memberIds || memberIds.length === 0) {
      showToast('Please select at least one member', 'warning');
      return;
    }
    
    // Get form values
    const data = {
      member_ids: memberIds,
      title: $('#bulkAssignmentTitle').val(),
      description: $('#bulkAssignmentDescription').val(),
      assigned_to: $('#bulkAssignToAdmin').val(),
      assignment_type: $('#bulkAssignmentType').val(),
      priority: $('#bulkAssignmentPriority').val(),
      due_date: $('#bulkAssignmentDueDate').val(),
      send_email: $('#bulkSendEmailNotification').is(':checked')
    };
    
    // Show loading indicator
    const loadingModal = Swal.fire({
      title: 'Processing Bulk Assignment',
      html: `Creating assignments for ${memberIds.length} members...`,
      allowOutsideClick: false,
      didOpen: () => {
        Swal.showLoading();
      }
    });
    
    // Send request
    $.ajax({
      url: 'api/assignments/bulk-assign.php',
      type: 'POST',
      data: data,
      dataType: 'json',
      success: (response) => {
        loadingModal.close();
        
        if (response.success) {
          // Show success message
          Swal.fire({
            icon: 'success',
            title: 'Bulk Assignment Completed',
            html: `Successfully assigned ${response.data.successful} members.<br>
                  ${response.data.failed > 0 ? `Failed to assign ${response.data.failed} members.` : ''}`,
            confirmButtonText: 'OK'
          });
          
          // Close the modal
          $('#bulkAssignModal').modal('hide');
          
          // Refresh data
          loadAssignments();
          loadStatistics();
          
          // Reset form
          $('#bulkAssignForm')[0].reset();
        } else {
          Swal.fire({
            icon: 'error',
            title: 'Bulk Assignment Failed',
            text: response.message || 'Failed to process bulk assignments',
            confirmButtonText: 'OK'
          });
        }
      },
      error: (xhr) => {
        loadingModal.close();
        handleError(xhr, 'Failed to process bulk assignments');
      }
    });
  };
  
  // Batch update assignments
  const batchUpdate = () => {
    // Get selected assignment IDs
    const assignmentIds = [];
    $('.assignment-checkbox:checked').each(function() {
      assignmentIds.push($(this).data('id'));
    });
    
    if (assignmentIds.length === 0) {
      showToast('Please select at least one assignment', 'warning');
      return;
    }
    
    // Check if any update values are provided
    const status = $('#batchUpdateStatus').val();
    const priority = $('#batchUpdatePriority').val();
    const assignedTo = $('#batchUpdateAssignTo').val();
    const dueDate = $('#batchUpdateDueDate').val();
    
    if (!status && !priority && !assignedTo && !dueDate) {
      showToast('Please specify at least one value to update', 'warning');
      return;
    }
    
    // Get form values
    const data = {
      assignment_ids: assignmentIds,
      send_email: $('#batchUpdateSendEmail').is(':checked')
    };
    
    // Only add properties that are being updated
    if (status) data.status = status;
    if (priority) data.priority = priority;
    if (assignedTo) data.assigned_to = assignedTo;
    if (dueDate) data.due_date = dueDate;
    
    // Show loading indicator
    const loadingModal = Swal.fire({
      title: 'Processing Batch Update',
      html: `Updating ${assignmentIds.length} assignments...`,
      allowOutsideClick: false,
      didOpen: () => {
        Swal.showLoading();
      }
    });
    
    // Send request
    $.ajax({
      url: 'api/assignments/batch-update.php',
      type: 'POST',
      data: data,
      dataType: 'json',
      success: (response) => {
        loadingModal.close();
        
        if (response.success) {
          // Show success message
          Swal.fire({
            icon: 'success',
            title: 'Batch Update Completed',
            html: `Successfully updated ${response.data.successful} assignments.<br>
                  ${response.data.failed > 0 ? `Failed to update ${response.data.failed} assignments.` : ''}`,
            confirmButtonText: 'OK'
          });
          
          // Close the modal
          $('#batchUpdateModal').modal('hide');
          
          // Refresh data
          loadAssignments();
          loadStatistics();
          
          // Reset form
          $('#batchUpdateForm')[0].reset();
          
          // Uncheck all checkboxes
          $('.assignment-checkbox').prop('checked', false);
          $('#selectAll').prop('checked', false);
          
          // Update batch action buttons
          updateBatchActionButtons();
        } else {
          Swal.fire({
            icon: 'error',
            title: 'Batch Update Failed',
            text: response.message || 'Failed to update assignments',
            confirmButtonText: 'OK'
          });
        }
      },
      error: (xhr) => {
        loadingModal.close();
        handleError(xhr, 'Failed to update assignments');
      }
    });
  };
  
  // View assignment details
  const viewAssignment = (assignmentId) => {
    $.ajax({
      url: `api/assignments/get.php?id=${assignmentId}`,
      type: 'GET',
      dataType: 'json',
      success: (response) => {
        if (response.success && response.data) {
          const assignment = response.data.assignment;
          
          // Fill modal with assignment details
          $('#viewAssignmentId').text(assignment.id);
          $('#viewAssignmentTitle').text('Assignment Details: ' + assignment.title);
          $('#viewAssignmentTitleText').text(assignment.title);
          $('#viewAssignmentDescription').text(assignment.description || 'No description provided');
          $('#viewAssignmentMember').html(`<a href="members.php?id=${assignment.member_id}" target="_blank">${assignment.member_name}</a>`);
          $('#viewAssignmentAssignedTo').text(assignment.assigned_to_name);
          $('#viewAssignmentType').html(formatTypeLabel(assignment.assignment_type));
          $('#viewAssignmentPriority').html(formatPriorityLabel(assignment.priority));
          $('#viewAssignmentStatus').html(formatStatusLabel(assignment.status));
          $('#viewAssignmentDueDate').html(formatDueDate(assignment.due_date));
          $('#viewAssignmentCreated').text(formatDate(assignment.created_at));
          $('#viewAssignmentUpdated').text(formatDate(assignment.updated_at));
          
          // Show/hide action buttons based on assignment status
          if (assignment.status === 'completed' || assignment.status === 'cancelled') {
            $('#markCompletedBtn').hide();
          } else {
            $('#markCompletedBtn').show();
            $('#markCompletedBtn').data('id', assignment.id);
          }
          
          $('#editAssignmentBtn').data('id', assignment.id);
          
          // Fill activity log
          if (response.data.activity && response.data.activity.length > 0) {
            const activityLog = $('#assignmentActivityLog');
            activityLog.empty();
            
            response.data.activity.forEach(activity => {
              let icon, colorClass;
              
              switch (activity.action) {
                case 'created':
                  icon = 'plus-circle';
                  colorClass = 'bg-success';
                  break;
                case 'updated':
                  icon = 'edit';
                  colorClass = 'bg-primary';
                  break;
                case 'completed':
                  icon = 'check-circle';
                  colorClass = 'bg-success';
                  break;
                case 'deleted':
                  icon = 'trash';
                  colorClass = 'bg-danger';
                  break;
                case 'assigned':
                  icon = 'user-tag';
                  colorClass = 'bg-info';
                  break;
                default:
                  icon = 'sync';
                  colorClass = 'bg-secondary';
              }
              
              activityLog.append(`
                <li>
                  <i class="fas fa-${icon} ${colorClass}"></i>
                  <div class="timeline-item">
                    <span class="time"><i class="fas fa-clock"></i> ${formatDate(activity.timestamp)}</span>
                    <h3 class="timeline-header"><a href="#">${activity.admin_name}</a> ${activity.action} the assignment</h3>
                    <div class="timeline-body">
                      ${activity.details || ''}
                    </div>
                  </div>
                </li>
              `);
            });
            
            // Add final node
            activityLog.append('<li><i class="fas fa-clock bg-gray"></i></li>');
          } else {
            $('#assignmentActivityLog').html(`
              <li>
                <i class="fas fa-info-circle bg-info"></i>
                <div class="timeline-item">
                  <h3 class="timeline-header">No activity recorded</h3>
                </div>
              </li>
              <li><i class="fas fa-clock bg-gray"></i></li>
            `);
          }
          
          // Show the modal
          $('#viewAssignmentModal').modal('show');
        } else {
          showToast(response.message || 'Failed to load assignment details', 'error');
        }
      },
      error: (xhr) => {
        handleError(xhr, 'Failed to load assignment details');
      }
    });
  };
  
  // Mark assignment as completed
  const markCompleted = (assignmentId) => {
    Swal.fire({
      title: 'Confirm Completion',
      text: "Are you sure you want to mark this assignment as completed?",
      icon: 'question',
      showCancelButton: true,
      confirmButtonColor: '#28a745',
      cancelButtonColor: '#6c757d',
      confirmButtonText: 'Yes, complete it!'
    }).then((result) => {
      if (result.isConfirmed) {
        $.ajax({
          url: 'api/assignments/update_status.php',
          type: 'POST',
          data: {
            id: assignmentId,
            status: 'completed',
            send_email: true
          },
          dataType: 'json',
          success: (response) => {
            if (response.success) {
              showToast('Assignment marked as completed');
              
              // Close modal if open
              $('#viewAssignmentModal').modal('hide');
              
              // Refresh data
              loadAssignments();
              loadStatistics();
            } else {
              showToast(response.message || 'Failed to update assignment', 'error');
            }
          },
          error: (xhr) => {
            handleError(xhr, 'Failed to update assignment');
          }
        });
      }
    });
  };
  
  // Delete an assignment
  const deleteAssignment = (assignmentId) => {
    Swal.fire({
      title: 'Confirm Deletion',
      text: "Are you sure you want to delete this assignment? This action cannot be undone.",
      icon: 'warning',
      showCancelButton: true,
      confirmButtonColor: '#dc3545',
      cancelButtonColor: '#6c757d',
      confirmButtonText: 'Yes, delete it!'
    }).then((result) => {
      if (result.isConfirmed) {
        $.ajax({
          url: 'api/assignments/delete.php',
          type: 'POST',
          data: {
            id: assignmentId
          },
          dataType: 'json',
          success: (response) => {
            if (response.success) {
              showToast('Assignment deleted successfully');
              
              // Refresh data
              loadAssignments();
              loadStatistics();
            } else {
              showToast(response.message || 'Failed to delete assignment', 'error');
            }
          },
          error: (xhr) => {
            handleError(xhr, 'Failed to delete assignment');
          }
        });
      }
    });
  };
  
  // Filter assignments by status
  const filterAssignments = (status) => {
    selectedFilter = status;
    updateAssignmentsTable();
  };
  
  // Filter assignments by priority
  const filterByPriority = (priority) => {
    selectedPriorityFilter = priority;
    updateAssignmentsTable();
  };
  
  // Search assignments
  const searchAssignments = (searchTerm) => {
    if (!searchTerm) {
      updateAssignmentsTable();
      return;
    }
    
    // If DataTable is initialized, use its search function
    if (assignmentsTable) {
      assignmentsTable.search(searchTerm).draw();
    }
  };
  
  // Sort assignments by due date
  const sortByDueDate = () => {
    if (assignmentsTable) {
      // Find the due date column index
      const dueDateColIndex = $('#assignmentsTable thead th:contains("Due Date")').index();
      
      if (dueDateColIndex !== -1) {
        assignmentsTable.order([dueDateColIndex, 'asc']).draw();
      }
    }
  };
  
  // Set up real-time updates using Server-Sent Events
  const setupRealTimeUpdates = () => {
    // Close existing connection if any
    if (eventSource) {
      eventSource.close();
    }
    
    // Create new connection
    eventSource = new EventSource('api/assignments/stream.php');
    
    // Listen for assignment updates
    eventSource.addEventListener('assignment_update', (event) => {
      const data = JSON.parse(event.data);
      
      // Show notification
      showToast(`Assignment "${data.title}" was ${data.action}`, 'info');
      
      // Refresh data
      loadAssignments();
      loadStatistics();
    });
    
    // Listen for connection errors
    eventSource.onerror = function() {
      console.error('SSE connection error');
      
      // Attempt to reconnect after delay
      setTimeout(() => {
        setupRealTimeUpdates();
      }, 5000);
    };
    
    // Ensure connection is closed when page is unloaded
    window.addEventListener('beforeunload', () => {
      if (eventSource) {
        eventSource.close();
      }
    });
  };
  
  // Attach event handlers to dynamic elements
  const attachEventHandlers = () => {
    // View assignment
    $('.view-assignment-btn').on('click', function() {
      const assignmentId = $(this).data('id');
      viewAssignment(assignmentId);
    });
    
    // View assignment from due soon list
    $('.view-assignment-link').on('click', function() {
      const assignmentId = $(this).data('id');
      viewAssignment(assignmentId);
    });
    
    // Complete assignment
    $('.complete-assignment-btn').on('click', function() {
      const assignmentId = $(this).data('id');
      markCompleted(assignmentId);
    });
    
    // Delete assignment
    $('.delete-assignment-btn').on('click', function() {
      const assignmentId = $(this).data('id');
      deleteAssignment(assignmentId);
    });
    
    // Edit assignment
    $('.edit-assignment-btn').on('click', function() {
      const assignmentId = $(this).data('id');
      // Redirect to edit page or show edit modal
      // TODO: Implement edit functionality
    });
    
    // Checkbox change event
    $('.assignment-checkbox').on('change', function() {
      updateBatchActionButtons();
    });
  };
  
  // Initialize the module
  const init = () => {
    // Initial setup
    loadAssignments();
    loadStatistics();
    
    // Event handlers for modal buttons
    $('#markCompletedBtn').on('click', function() {
      const assignmentId = $(this).data('id');
      markCompleted(assignmentId);
    });
    
    // Handle responsive design
    if (window.innerWidth < 768) {
      // Mobile optimizations
      $('.small-box-footer').addClass('btn').addClass('btn-sm');
      $('.btn-group').addClass('btn-group-sm').addClass('flex-wrap');
    }
    
    // Window resize handler for responsive adjustments
    $(window).on('resize', function() {
      if (window.innerWidth < 768) {
        $('.small-box-footer').addClass('btn').addClass('btn-sm');
        $('.btn-group').addClass('btn-group-sm').addClass('flex-wrap');
      } else {
        $('.small-box-footer').removeClass('btn').removeClass('btn-sm');
        $('.btn-group').removeClass('btn-group-sm').removeClass('flex-wrap');
      }
    });
  };
  
  // Public API
  return {
    init,
    loadAssignments,
    loadMembers,
    loadAdmins,
    loadStatistics,
    createAssignment,
    bulkAssign,
    batchUpdate,
    filterAssignments,
    filterByPriority,
    searchAssignments,
    sortByDueDate,
    initCharts,
    setupRealTimeUpdates,
    updateBatchActionButtons
  };
})();
