<?php
/**
 * Database Class
 * 
 * Provides PDO database connection and utility methods
 */

namespace Services\Core;

class Database {
    private $host;
    private $db;
    private $user;
    private $pass;
    private $charset;
    private $connection;
    
    /**
     * Constructor
     * 
     * @param string $host Database host
     * @param string $db Database name
     * @param string $user Database user
     * @param string $pass Database password
     * @param string $charset Database charset
     */
    public function __construct($host = null, $db = null, $user = null, $pass = null, $charset = 'utf8mb4') {
        // Load config if not provided
        if ($host === null || $db === null || $user === null || $pass === null) {
            $config = $this->loadConfig();
            $this->host = $host ?? $config['host'];
            $this->db = $db ?? $config['db'];
            $this->user = $user ?? $config['user'];
            $this->pass = $pass ?? $config['pass'];
        } else {
            $this->host = $host;
            $this->db = $db;
            $this->user = $user;
            $this->pass = $pass;
        }
        
        $this->charset = $charset;
    }
    
    /**
     * Load database configuration
     * 
     * @return array Database configuration
     */
    private function loadConfig() {
        $configFile = __DIR__ . '/../../config/database.php';
        
        // Check if config file exists
        if (!file_exists($configFile)) {
            throw new \Exception("Database configuration file not found: $configFile");
        }
        
        // Include configuration
        $config = include $configFile;
        
        return $config;
    }
    
    /**
     * Get PDO connection
     * 
     * @return \PDO PDO connection
     */
    public function getConnection() {
        // Return existing connection if already established
        if ($this->connection !== null) {
            return $this->connection;
        }
        
        try {
            // Create DSN
            $dsn = "mysql:host={$this->host};dbname={$this->db};charset={$this->charset}";
            
            // PDO options
            $options = [
                \PDO::ATTR_ERRMODE => \PDO::ERRMODE_EXCEPTION,
                \PDO::ATTR_DEFAULT_FETCH_MODE => \PDO::FETCH_ASSOC,
                \PDO::ATTR_EMULATE_PREPARES => false,
            ];
            
            // Create connection
            $this->connection = new \PDO($dsn, $this->user, $this->pass, $options);
            
            return $this->connection;
        } catch (\PDOException $e) {
            // Log error
            error_log('Database connection error: ' . $e->getMessage());
            
            throw new \Exception('Database connection failed: ' . $e->getMessage());
        }
    }
    
    /**
     * Safely query a table with default empty values if table doesn't exist
     * 
     * @param string $tableName Table name
     * @param string $query SQL query
     * @param array $params Query parameters
     * @param array $defaultEmptyValues Default values to return if table doesn't exist
     * @return array Query results or default empty values
     */
    public function safelyQueryTable($tableName, $query, $params = [], $defaultEmptyValues = []) {
        try {
            $conn = $this->getConnection();
            
            // Check if table exists
            $checkTableQuery = "SHOW TABLES LIKE :tableName";
            $checkTableStmt = $conn->prepare($checkTableQuery);
            $checkTableStmt->bindParam(':tableName', $tableName);
            $checkTableStmt->execute();
            
            $tableExists = $checkTableStmt->rowCount() > 0;
            
            if ($tableExists) {
                // Execute original query
                $stmt = $conn->prepare($query);
                
                // Bind parameters
                foreach ($params as $param => $value) {
                    $stmt->bindValue($param, $value);
                }
                
                $stmt->execute();
                return $stmt->fetchAll(\PDO::FETCH_ASSOC);
            } else {
                // Return default empty values
                return $defaultEmptyValues;
            }
        } catch (\Exception $e) {
            // Log error
            error_log('Database query error: ' . $e->getMessage());
            
            // Return default empty values on error
            return $defaultEmptyValues;
        }
    }
    
    /**
     * Begin a transaction
     * 
     * @return bool Success status
     */
    public function beginTransaction() {
        try {
            $conn = $this->getConnection();
            return $conn->beginTransaction();
        } catch (\Exception $e) {
            error_log('Begin transaction error: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Commit a transaction
     * 
     * @return bool Success status
     */
    public function commit() {
        try {
            $conn = $this->getConnection();
            return $conn->commit();
        } catch (\Exception $e) {
            error_log('Commit transaction error: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Rollback a transaction
     * 
     * @return bool Success status
     */
    public function rollBack() {
        try {
            $conn = $this->getConnection();
            return $conn->rollBack();
        } catch (\Exception $e) {
            error_log('Rollback transaction error: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Execute a query and return the last insert ID
     * 
     * @param string $query SQL query
     * @param array $params Query parameters
     * @return int|false Last insert ID or false on failure
     */
    public function insert($query, $params = []) {
        try {
            $conn = $this->getConnection();
            
            // Prepare and execute
            $stmt = $conn->prepare($query);
            
            // Bind parameters
            foreach ($params as $param => $value) {
                $stmt->bindValue($param, $value);
            }
            
            $stmt->execute();
            
            // Return last insert ID
            return $conn->lastInsertId();
        } catch (\Exception $e) {
            error_log('Database insert error: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Execute a query and return the number of affected rows
     * 
     * @param string $query SQL query
     * @param array $params Query parameters
     * @return int|false Number of affected rows or false on failure
     */
    public function execute($query, $params = []) {
        try {
            $conn = $this->getConnection();
            
            // Prepare and execute
            $stmt = $conn->prepare($query);
            
            // Bind parameters
            foreach ($params as $param => $value) {
                $stmt->bindValue($param, $value);
            }
            
            $stmt->execute();
            
            // Return affected rows
            return $stmt->rowCount();
        } catch (\Exception $e) {
            error_log('Database execute error: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Execute a query and return a single row
     * 
     * @param string $query SQL query
     * @param array $params Query parameters
     * @return array|false Single row or false on failure
     */
    public function fetchOne($query, $params = []) {
        try {
            $conn = $this->getConnection();
            
            // Prepare and execute
            $stmt = $conn->prepare($query);
            
            // Bind parameters
            foreach ($params as $param => $value) {
                $stmt->bindValue($param, $value);
            }
            
            $stmt->execute();
            
            // Return single row
            return $stmt->fetch(\PDO::FETCH_ASSOC);
        } catch (\Exception $e) {
            error_log('Database fetchOne error: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Execute a query and return all rows
     * 
     * @param string $query SQL query
     * @param array $params Query parameters
     * @return array|false All rows or false on failure
     */
    public function fetchAll($query, $params = []) {
        try {
            $conn = $this->getConnection();
            
            // Prepare and execute
            $stmt = $conn->prepare($query);
            
            // Bind parameters
            foreach ($params as $param => $value) {
                $stmt->bindValue($param, $value);
            }
            
            $stmt->execute();
            
            // Return all rows
            return $stmt->fetchAll(\PDO::FETCH_ASSOC);
        } catch (\Exception $e) {
            error_log('Database fetchAll error: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Check if a table exists
     * 
     * @param string $tableName Table name
     * @return bool True if table exists, false otherwise
     */
    public function tableExists($tableName) {
        try {
            $conn = $this->getConnection();
            
            // Check if table exists
            $query = "SHOW TABLES LIKE :tableName";
            $stmt = $conn->prepare($query);
            $stmt->bindParam(':tableName', $tableName);
            $stmt->execute();
            
            return $stmt->rowCount() > 0;
        } catch (\Exception $e) {
            error_log('Table exists check error: ' . $e->getMessage());
            return false;
        }
    }
}
